"""Module for exporting a WaveWatch3 simulation with feedback."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
from functools import cached_property
import logging

# 2. Third party modules
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists
from xms.snap.snap_exterior_arc import SnapExteriorArc

# 4. Local modules
from xms.wavewatch3.dmi.xms_data import XmsData
from xms.wavewatch3.file_io.bounc_nml_writer import WW3BouncNmlWriter
from xms.wavewatch3.file_io.gmsh_writer import GmshWriter
from xms.wavewatch3.file_io.grid_nml_writer import WW3GridNamelistWriter
from xms.wavewatch3.file_io.namelists_nml_writer import WW3NamelistsNmlWriter
from xms.wavewatch3.file_io.ounf_nml_writer import WW3OunfNmlWriter
from xms.wavewatch3.file_io.ounp_nml_writer import WW3OunpNmlWriter
from xms.wavewatch3.file_io.prnc_nml_writer import WW3PrncNmlWriter
from xms.wavewatch3.file_io.shell_nml_writer import WW3ShellNamelistWriter


class ExportSimulationRunner(QThread):
    """Class for exporting a WaveWatch3 simulation in a worker thread."""
    processing_finished = Signal()

    def __init__(self, parent):
        """Constructor.

        Args:
            parent (:obj:`QWidget`): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent)
        self._logger = logging.getLogger('xms.wavewatch3')
        self.co_grid = None

    def run(self):
        """Creates the worker thread."""
        try:
            xms_data = XmsData()
            # Ensure a grid is linked to the sim
            if xms_data.do_ugrid is None:
                raise ValueError('No grid linked to simulation.')
            elif xms_data.spectral_coverages is None:
                raise ValueError('No spectral coverage linked to simulation.')
            for coverage in xms_data.spectral_coverages:
                if not coverage.m_cov:
                    raise ValueError('Spectral point missing spectral data.')

            self.co_grid = xms_data.cogrid
            self._logger.info('Exporting ww3_shel.nml file...')
            shell_writer = WW3ShellNamelistWriter(xms_data)
            shell_writer.write()
            self._logger.info('Exporting mesh file...')
            gmsh_writer = GmshWriter(xms_data.cogrid, xms_data.do_ugrid.name)
            gmsh_writer.write()
            ocean, lateral = None, None
            if xms_data.bc_data:
                # Check for the existence of later Boundary Condition info, and write meshbnd.msh if so
                ocean, lateral = self.get_node_ids(xms_data.bc_coverage, xms_data.bc_data)
                if len(lateral) > 0:
                    self._logger.info('Exporting meshbnd.msh file...')
                    meshbnd_writer = GmshWriter(xms_data.cogrid, 'meshbnd', lateral, ocean)
                    meshbnd_writer.write()
            self._logger.info('Exporting ww3_grid.nml file...')
            grid_writer = WW3GridNamelistWriter(xms_data, lateral, ocean)
            grid_writer.write()
            self._logger.info('Exporting namelists.nml file...')
            namelists_writer = WW3NamelistsNmlWriter(xms_data)
            namelists_writer.write()
            self._logger.info('Exporting ww3_ounf.nml file...')
            ounf_writer = WW3OunfNmlWriter(xms_data)
            ounf_writer.write()
            ounp_writer = WW3OunpNmlWriter(xms_data)
            ounp_writer.write()
            self._logger.info('Exporting ww3_bounc.nml file...')
            bounc_writer = WW3BouncNmlWriter(xms_data)
            bounc_writer.write()
            self._logger.info('Exporting ww3_prnc.nml file(s)...')
            self._write_prnc_files(xms_data)
            self._logger.info('Finished!')
        except Exception as e:
            self._logger.exception(f'Error exporting simulation: {str(e)}')
        finally:
            self.processing_finished.emit()

    def get_node_ids(self, coverage, data):
        """Gets input and lateral nodes."""
        input_nodes = []
        lateral_nodes = []
        for arc in coverage.arcs:
            feature_type = data.feature_type(TargetType.arc, feature_id=arc.id)
            node_indexes = self._snapper.get_snapped_points(arc)['id']
            nodes = [node_index + 1 for node_index in node_indexes]
            if feature_type == 'input':
                input_nodes.extend(nodes)
            elif feature_type == 'lateral':
                lateral_nodes.extend(nodes)
        return input_nodes, lateral_nodes

    @cached_property
    def _snapper(self) -> SnapExteriorArc:
        """Set up the polygon snapper."""
        snapper = SnapExteriorArc()
        snapper.set_grid(self.co_grid, target_cells=False)
        return snapper

    def _write_prnc_files(self, xms_data):
        """Write out the forcing field PRNC files.

        Args:
            xms_data (:obj:`XmsData`): Simulation data retrieved from SMS.
        """
        run_control = xms_data.sim_data_model_control.group('run_control')
        ice_and_mud = xms_data.sim_data_model_control.group('ice_and_mud')

        forcing_fields = []
        forcing_fields.append(
            [
                'water_levels',
                run_control.parameter('water_level_dataset').value,
                run_control.parameter('water_levels').value
            ]
        )
        forcing_fields.append(
            ['currents',
             run_control.parameter('currents_dataset').value,
             run_control.parameter('currents').value]
        )
        forcing_fields.append(
            ['winds', run_control.parameter('winds_dataset').value,
             run_control.parameter('winds').value]
        )
        forcing_fields.append(
            [
                'atm_momentum',
                run_control.parameter('atm_momentum_dataset').value,
                run_control.parameter('define_atm_momentum').value
            ]
        )
        forcing_fields.append(
            [
                'air_density',
                run_control.parameter('air_density_dataset').value,
                run_control.parameter('air_density').value
            ]
        )
        forcing_fields.append(
            [
                'ice_concentration',
                ice_and_mud.parameter('ice_concentration').value,
                ice_and_mud.parameter('concentration').value
            ]
        )
        forcing_fields.append(
            [
                'ice_parameter1',
                ice_and_mud.parameter('ice_param_1_dataset').value,
                ice_and_mud.parameter('param_1').value
            ]
        )
        forcing_fields.append(
            [
                'ice_parameter2',
                ice_and_mud.parameter('ice_param_2_dataset').value,
                ice_and_mud.parameter('param_2').value
            ]
        )
        forcing_fields.append(
            [
                'ice_parameter3',
                ice_and_mud.parameter('ice_param_3_dataset').value,
                ice_and_mud.parameter('param_3').value
            ]
        )
        forcing_fields.append(
            [
                'ice_parameter4',
                ice_and_mud.parameter('ice_param_4_dataset').value,
                ice_and_mud.parameter('param_4').value
            ]
        )
        forcing_fields.append(
            [
                'ice_parameter5',
                ice_and_mud.parameter('ice_param_5_dataset').value,
                ice_and_mud.parameter('param_5').value
            ]
        )
        forcing_fields.append(
            [
                'mud_density',
                ice_and_mud.parameter('mud_density_dataset').value,
                ice_and_mud.parameter('mud_density').value
            ]
        )
        forcing_fields.append(
            [
                'mud_thickness',
                ice_and_mud.parameter('mud_thickness_dataset').value,
                ice_and_mud.parameter('mud_thickness').value
            ]
        )
        forcing_fields.append(
            [
                'mud_viscosity',
                ice_and_mud.parameter('mud_viscosity_dataset').value,
                ice_and_mud.parameter('mud_viscosity').value
            ]
        )
        for forcing_field in forcing_fields:
            if forcing_field[1] != '' and forcing_field[2] == 'T: external forcing file':
                prnc_writer = WW3PrncNmlWriter(xms_data, forcing_field[0], forcing_field[1])
                prnc_writer.write()


def export_simulation_with_feedback():
    """Export a simulation with a feedback dialog."""
    ensure_qapplication_exists()
    worker = ExportSimulationRunner(None)  # No Qt parent for export scripts yet
    display_text = {
        'title': 'WaveWatch3 Export Simulation',
        'working_prompt': 'Exporting WaveWatch3 simulation files. Please wait...',
        'warning_prompt': 'Warning(s) encountered while exporting simulation. Review log output for more details.',
        'error_prompt': 'Error(s) encountered while exporting simulation. Review log output for more details.',
        'success_prompt': 'Successfully exported simulation',
        'note': '',
        'auto_load': 'Close this dialog automatically when exporting is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(
        display_text=display_text, logger_name='xms.wavewatch3', worker=worker, parent=None
    )
    if feedback_dlg.exec():
        if LogEchoQSignalStream.logged_error:
            XmEnv.report_export_error()
    else:
        XmEnv.report_export_aborted()
