"""Class to write a WaveWatch3 ounf nml file."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
from io import StringIO
import logging
import shutil

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.time_format import string_to_datetime

# 4. Local modules
from xms.wavewatch3.file_io.namelists_defaults_util import WW3NamelistsDefaultsUtil
from xms.wavewatch3.gui.gui_util import get_formatted_date_string


class WW3OunpNmlWriter:
    """Class to write a WaveWatch3 ounp nml file (related to point namelist)."""
    def __init__(self, xms_data):
        """Constructor.

        Args:
            xms_data (:obj:`XmsData`): Simulation data retrieved from SMS
        """
        self._ss = StringIO()
        self._logger = logging.getLogger('xms.wavewatch3')
        self._bound_process = None
        self._grid_process = None
        self._xms_data = xms_data
        self._defaults = WW3NamelistsDefaultsUtil()

    def _write_ounp_nml_file(self):
        """Writes the ounp namelist file."""
        file_w_path = "ww3_ounp.nml"

        self._write_ounp_header()
        self._write_point_namelist()
        self._write_file_namelist()
        self._write_end_comments()
        self._flush(file_w_path)

    def _write_ounp_header(self):
        """Writes the header comments for the ounp namelist file."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! WAVEWATCH III - ww3_ounp.nml - Point output post-processing           !\n'
            '! -------------------------------------------------------------------- !\n'
            '\n\n'
        )

    def _write_point_namelist(self):
        """Writes the FIELD_NML namelist for the output fields to postprocess."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! Define the output fields to postprocess via POINT_NML namelist\n'
            '!\n'
            '! * namelist must be terminated with /\n'
            '! * definitions & defaults:\n'
            "!     POINT%TIMESTART            = '19000101 000000'  ! Stop date for the output field\n"
            "!     POINT%TIMESTRIDE           = '0'                ! Time stride for the output field\n"
            "!     POINT%TIMECOUNT            = '1000000000'       ! Number of time steps\n"
            '!     POINT%TIMESPLIT            = 6                  '
            '! [4(yearly),6(monthly),8(daily),10(hourly)]\n'
            "!     POINT%LIST                 = 'all'              ! List of points index ['all'|'1 2 3']\n"
            '!     POINT%SAMEFILE             = T                  ! All the variables in the same file\n'
            '!     POINT%BUFFER               = 150                ! Number of points to process per pass\n'
            '!     POINT%TYPE                 = 1                  '
            '! [0 = inventory | 1 = spectra | 2 = mean param | 3 = source terms]\n'
            '!     POINT%DIMORDER             = T                  ! [time,station=T | station,time=f]]\n'
            '! -------------------------------------------------------------------- !\n'
        )
        attrs = self._xms_data.sim_data_model_control.group('output')
        self._ss.write('&POINT_NML\n')
        time_start = string_to_datetime(attrs.parameter('output_start_date_point').value)
        self._ss.write(f"    POINT%TIMESTART      = '{get_formatted_date_string(time_start)}'\n")
        if attrs.parameter('output_second_interval_point').value != 0:
            self._ss.write(f"    POINT%TIMESTRIDE     = '{attrs.parameter('output_second_interval_point').value}'\n")
        self._ss.write('/\n\n\n')

    def _write_file_namelist(self):
        """Writes the FILE_NML namelist for the defining the content of the output file."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! Define the content of the output file via FILE_NML namelist\n'
            '!\n'
            '! * namelist must be terminated with /\n'
            '! * definitions & defaults:\n'
            "!     FILE%PREFIX        = 'ww3.'            ! Prefix for output file name\n"
            '!     FILE%NETCDF        = 3                 ! Netcdf version [3|4]\n'
            '! -------------------------------------------------------------------- !\n'
        )
        attrs = self._xms_data.sim_data_output_fields.group('user_defined')
        self._ss.write('&FILE_NML\n')
        netcdf_version = [3, 4]
        if attrs.parameter('file_prefix').value != 'ww3.':
            self._ss.write(f"    FILE%PREFIX          = '{attrs.parameter('file_prefix').value}'\n")
        if attrs.parameter('file_netcdf').value in netcdf_version:
            self._ss.write(f"    FILE%NETCDF          = {attrs.parameter('file_netcdf').value}\n")
        self._ss.write('/\n\n\n')

    def _write_end_comments(self):
        """Writes out the end comments at the end of the file."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! WAVEWATCH III - end of namelist                                      !\n'
            '! -------------------------------------------------------------------- !\n'
            '\n'
        )

    def _flush(self, file_w_path):
        """Writes the StringIO previously processed to a file.

        Args:
            file_w_path (:obj:`str`):  String of the filename to write to.
        """
        f = open(file_w_path, 'w')
        self._ss.seek(0)
        shutil.copyfileobj(self._ss, f, 100000)
        f.close()

    def write(self):
        """Top-level entry point for the WaveWatch3 ounf file writer."""
        self._write_ounp_nml_file()
