"""Class to read a WaveWatch3 spectral solution file."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.wavewatch3.file_io.netcdf_dataset_reader import NetcdfDatasetReader
from xms.wavewatch3.file_io.spec_list_reader import WW3SpecListReader


class WW3SpecSolutionReader:
    """Class to read a WaveWatch3 spectral solution file."""
    def read(self):
        """Top-level entry point for the WaveWatch3 spectral solution reader."""
        query = Query()
        output_types = [
            'dpt', 'cur', 'wnd', 'ast', 'wlv', 'ice', 'ibg', 'd50', 'ic1', 'ic5', 'hs', 'lm', 't02', 't0m1', 't01',
            'fp', 'dir', 'spr', 'dp', 'hig', 'mxe', 'mxes', 'mxh', 'mxhc', 'sdmh', 'sdmhc', 'wbt', 'tp', 'ef', 'th1m',
            'sth1m', 'th2m', 'sth2m', 'wn', 'phs', 'ptp', 'plp', 'pdir', 'pspr', 'pws', 'pdp', 'pqp', 'ppe', 'pgw',
            'psw', 'ptm10', 'pt01', 'pt02', 'pep', 'tws', 'pnr', 'ust', 'cha', 'cge', 'faw', 'taw', 'twa', 'wcc', 'wcf',
            'wch', 'wcm', 'fws', 'sxy', 'two', 'bhd', 'foc', 'tus', 'uss', 'p2s', 'usf', 'p2l', 'twi', 'fic', 'usp',
            'abr', 'ubr', 'bed', 'fbb', 'tbb', 'mss', 'msc', 'wl02', 'axt', 'ayt', 'axy', 'dtd', 'fc', 'cfx', 'cfd',
            'cfk', 'u1', 'u2'
        ]
        netcdf_file = query.read_file
        dt = datetime.datetime(1990, 1, 1)
        temp_dir = query.xms_temp_directory if query else '.'
        if '_spec' in netcdf_file:
            reader = WW3SpecListReader(netcdf_file)
            reader.read_spectra_nc_file(nc_file=netcdf_file)
            query.add_coverage(reader.spectral_coverages[0])
        else:
            netcdf_reader = NetcdfDatasetReader(
                filename=netcdf_file, reftime=dt, query=query, temp_dir=temp_dir, output_types=output_types
            )
            netcdf_reader.read()
        query.send()
