"""Utility methods used by the GUI modules."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.time_format import datetime_to_string

# 4. Local modules

NULL_SELECTION = '(none selected)'


def get_date_time_string(date_part, time_part):
    """Gets a date time string from a QDateTime built from the date and time strings passed in.

    Args:
        date_part (:obj:`str`): The part of the string containing the date, ex:  20080225.
        time_part (:obj:`str`): The part of the string containing the time, ex:  105500.

    Returns:
        string(:obj:`str`):  A string formatted via QDateTime.
    """
    year = int(date_part[1:5])
    month = int(date_part[5:7])
    day = int(date_part[7:9])

    hour = int(time_part[0:2])
    minute = int(time_part[2:4])
    second = int(time_part[4:6])

    dt = datetime.datetime(year=year, month=month, day=day, hour=hour, minute=minute, second=second)
    return datetime_to_string(dt)


def get_formatted_date_string(date_time_val):
    """Gets a formatted date/time string.

    An example is:   20080225 000000
                     YYYYMMDD HHMMSS

    Args:
        date_time_val (:obj:`datetime.datetime`): A date/time object to format.

    Returns:
        string(:obj:`str`):  The shell input date/time string.
    """
    if not isinstance(date_time_val, datetime.datetime):
        return None

    return (
        f"{date_time_val.year}{date_time_val.month:0>2d}"
        f"{date_time_val.day:0>2d} "
        f"{date_time_val.hour:0>2d}{date_time_val.minute:0>2d}"
        f"{date_time_val.second:0>2d}"
    )


def get_output_field_type_list(output_fields_section):
    """Gets the type field list strings as a list for the Output Type namelist.

    Args:
        output_fields_section(:obj:`xms.gmi.data.generic_model.Section`):  The output fields data.

    Returns:
        (:obj:`list`):  A list containing strings of all the fields turned on.
    """
    attrs = {}
    for parameter in output_fields_section.parameters:
        attrs[parameter[1]] = parameter[2].value
    field_list = []
    # Forcing fields
    if attrs['DPT']:
        field_list.append('DPT')
    if attrs['CUR']:
        field_list.append('CUR')
    if attrs['WND']:
        field_list.append('WND')
    if attrs['AST']:
        field_list.append('AST')
    if attrs['WLV']:
        field_list.append('WLV')
    if attrs['ICE']:
        field_list.append('ICE')
    if attrs['IBG']:
        field_list.append('IBG')
    if attrs['D50']:
        field_list.append('D50')
    if attrs['IC1']:
        field_list.append('IC1')
    if attrs['IC5']:
        field_list.append('IC5')
    # Standard mean wave Parameters
    if attrs['HS']:
        field_list.append('HS')
    if attrs['LM']:
        field_list.append('LM')
    if attrs['T02']:
        field_list.append('T02')
    if attrs['T0M1']:
        field_list.append('T0M1')
    if attrs['T01']:
        field_list.append('T01')
    if attrs['FP']:
        field_list.append('FP')
    if attrs['DIR']:
        field_list.append('DIR')
    if attrs['SPR']:
        field_list.append('SPR')
    if attrs['DP']:
        field_list.append('DP')
    if attrs['HIG']:
        field_list.append('HIG')
    if attrs['MXE']:
        field_list.append('MXE')
    if attrs['MXES']:
        field_list.append('MXES')
    if attrs['MXH']:
        field_list.append('MXH')
    if attrs['MXHC']:
        field_list.append('MXHC')
    if attrs['SDMH']:
        field_list.append('SDMH')
    if attrs['SDMHC']:
        field_list.append('SDMHC')
    if attrs['WBT']:
        field_list.append('WBT')
    if attrs['TP']:
        field_list.append('TP')
    # Spectral Parameters (first 5)
    if attrs['EF']:
        field_list.append('EF')
    if attrs['TH1M']:
        field_list.append('TH1M')
    if attrs['STH1M']:
        field_list.append('STH1M')
    if attrs['TH2M']:
        field_list.append('TH2M')
    if attrs['STH2M']:
        field_list.append('STH2M')
    if attrs['WN']:
        field_list.append('WN')
    # Spectral Partition Parameters
    if attrs['PHS']:
        field_list.append('PHS')
    if attrs['PTP']:
        field_list.append('PTP')
    if attrs['PLP']:
        field_list.append('PLP')
    if attrs['PDIR']:
        field_list.append('PDIR')
    if attrs['PSPR']:
        field_list.append('PSPR')
    if attrs['PWS']:
        field_list.append('PWS')
    if attrs['PDP']:
        field_list.append('PDP')
    if attrs['PQP']:
        field_list.append('PQP')
    if attrs['PPE']:
        field_list.append('PPE')
    if attrs['PGW']:
        field_list.append('PGW')
    if attrs['PSW']:
        field_list.append('PSW')
    if attrs['PTM10']:
        field_list.append('PTM10')
    if attrs['PT01']:
        field_list.append('PT01')
    if attrs['PT02']:
        field_list.append('PT02')
    if attrs['PEP']:
        field_list.append('PEP')
    if attrs['TWS']:
        field_list.append('TWS')
    if attrs['PNR']:
        field_list.append('PNR')
    # Atmosphere-waves layer
    if attrs['UST']:
        field_list.append('UST')
    if attrs['CHA']:
        field_list.append('CHA')
    if attrs['CGE']:
        field_list.append('CGE')
    if attrs['FAW']:
        field_list.append('FAW')
    if attrs['TAW']:
        field_list.append('TAW')
    if attrs['TWA']:
        field_list.append('TWA')
    if attrs['WCC']:
        field_list.append('WCC')
    if attrs['WCF']:
        field_list.append('WCF')
    if attrs['WCH']:
        field_list.append('WCH')
    if attrs['WCM']:
        field_list.append('WCM')
    if attrs['FWS']:
        field_list.append('FWS')
    # Wave-ocean layer
    if attrs['SXY']:
        field_list.append('SXY')
    if attrs['TWO']:
        field_list.append('TWO')
    if attrs['BHD']:
        field_list.append('BHD')
    if attrs['FOC']:
        field_list.append('FOC')
    if attrs['TUS']:
        field_list.append('TUS')
    if attrs['USS']:
        field_list.append('PNR')
    if attrs['P2S']:
        field_list.append('P2S')
    if attrs['USF']:
        field_list.append('USF')
    if attrs['P2L']:
        field_list.append('P2L')
    if attrs['TWI']:
        field_list.append('TWI')
    if attrs['FIC']:
        field_list.append('FIC')
    if attrs['USP']:
        field_list.append('USP')
    # Wave-bottom layer
    if attrs['ABR']:
        field_list.append('ABR')
    if attrs['UBR']:
        field_list.append('UBR')
    if attrs['BED']:
        field_list.append('BED')
    if attrs['FBB']:
        field_list.append('FBB')
    if attrs['TBB']:
        field_list.append('TBB')
    # Spectrum parameters
    if attrs['MSS']:
        field_list.append('MSS')
    if attrs['MSC']:
        field_list.append('MSC')
    if attrs['WL02']:
        field_list.append('WL02')
    if attrs['AXT']:
        field_list.append('AXT')
    if attrs['AYT']:
        field_list.append('AYT')
    if attrs['AXY']:
        field_list.append('AXY')
    # Numerical diagnostics
    if attrs['DTD']:
        field_list.append('DTD')
    if attrs['FC']:
        field_list.append('FC')
    if attrs['CFX']:
        field_list.append('CFX')
    if attrs['CFD']:
        field_list.append('CFD')
    if attrs['CFK']:
        field_list.append('CFK')
    # User defined
    if attrs['U1']:
        field_list.append('U1')
    if attrs['U2']:
        field_list.append('U2')
    return field_list


def get_output_field_type_string(output_fields):
    """Gets the type field list string for the Output Type namelist.

    Args:
        output_fields(:obj:`xms.gmi.data.generic_model.Section`):  The output fields data.

    Returns:
        (:obj:`str`):  The string containing all the fields turned on.
    """
    field_list = get_output_field_type_list(output_fields)
    field_list_str = ''
    for field_val in field_list:
        field_list_str += f'{field_val} '
    return field_list_str[:-1]
