"""This module handles reading and writing files used by AdH."""

# 1. Standard python modules
from collections import defaultdict
import copy
import logging
import os
import re
import shlex
import sys
from typing import TextIO, TYPE_CHECKING

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules

# 4. Local modules
if TYPE_CHECKING:
    from .boundary_conditions import BoundaryConditions  # pragma: no cover
from .constituent_properties import GENERAL_CONSTITUENTS_COLUMN_TYPES
from .data_frame_utils import DataFrameBuilder
from .hot_start_data_set import HotStartDataSet
from .material_properties import MaterialProperties
from .material_transport_properties import MaterialTransportProperties
from .model_control import ModelControl
from .sediment_constituent_properties import CLAY_COLUMN_TYPES, SAND_COLUMN_TYPES
from .time_series import TimeSeries
from .vessels import Propeller, SEGMENTS_COLUMN_TYPES, Vessel, VesselList

__all__ = ['read_mesh_file', 'write_mesh_file', 'read_hot_start_file', 'write_hot_start_file', 'read_bc_file',
           'write_bc_file', 'read_bt_file', 'write_bt_file']


def read_mesh_file(file_name, mesh, logger=None):
    """
    Reads a *.3dm file that has a mesh definition.

    Args:
        file_name: file name of *.3dm file
        mesh: Mesh class that will hold the mesh information
        logger: The optional logger
    """
    if logger is not None:
        logger.info('Reading mesh file.')
    # initialize the mesh
    mesh.name = ''
    mesh.nodes = None
    mesh.elements = None

    nodes = []
    elements = []
    elements_3d = []
    with open(file_name, "r") as mesh_file:
        for line_number, line in enumerate(mesh_file):
            # skip comment
            if len(line) > 0 and line[0] == '!':
                continue
            # remove new line character
            line = line.rstrip()
            line_split = line.split(' ')
            # remove blank strings
            line_split[:] = (part for part in line_split if part != '')

            # skip blank line
            if len(line_split) == 0 or line_split[0] == '':
                continue

            if line_split[0] == 'ND':
                try:
                    nd = ['ND', int(line_split[1]), float(line_split[2]), float(line_split[3]), float(line_split[4])]
                    nodes.append(nd)
                except Exception:
                    raise IOError('Error reading node (ND) record from file on line {0}.'.format(line_number + 1))
            elif line_split[0] == 'E3T':
                try:
                    elem = ['E3T', int(line_split[1]), int(line_split[2]), int(line_split[3]),
                            int(line_split[4]), int(line_split[5])]
                    elements.append(elem)
                except Exception:
                    raise IOError('Error reading element (E3T) record from file on line {0}.'.format(line_number + 1))
            elif line_split[0] == 'E4T':
                try:
                    elem = ['E4T', int(line_split[1]), int(line_split[2]), int(line_split[3]),
                            int(line_split[4]), int(line_split[5]), int(line_split[6]), int(line_split[7])]
                    elements_3d.append(elem)
                except Exception:
                    raise IOError('Error reading element (E4T) record from file on line {0}.'.format(line_number + 1))
            elif line_split[0] == 'MESHNAME':
                # read the name a strip the quotes
                name = shlex.split(line, posix="win" not in sys.platform)[1]
                name = name.strip('"').strip("'")
                mesh.name = name
            elif line_split[0] in ['MESH2D', 'MESH3D']:
                continue

    # create the node and element dataframes
    if logger is not None:
        if len(nodes) > 0:
            logger.info(f'Read {len(nodes)} nodes.')
        else:
            logger.warning('No nodes found.')
    labels = ['CARD', 'ID', 'X', 'Y', 'Z']
    mesh.nodes = pd.DataFrame.from_records(nodes, columns=labels)
    if elements:
        labels = ['CARD', 'ID', 'NODE_0', 'NODE_1', 'NODE_2', 'MATERIAL_ID']
        mesh.elements = pd.DataFrame.from_records(elements, columns=labels)
        if logger is not None:
            logger.info(f'Read {len(elements)} 2D elements.')
    if elements_3d:
        labels = ['CARD', 'ID', 'NODE_0', 'NODE_1', 'NODE_2', 'NODE_3', 'MATERIAL_ID', 'ELEMENT_2D']
        mesh.elements_3d = pd.DataFrame.from_records(elements_3d, columns=labels)
        if logger is not None:
            logger.info(f'Read {len(elements_3d)} 3D elements.')
    if logger is not None:
        if not elements and not elements_3d:
            logger.warning(f'Read 0 elements.')


def write_mesh_file(file_name, mesh):
    """
    Writes a *.3dm file that defines a mesh.

    Args:
        file_name: file name of *.3dm file
        mesh: Mesh class with mesh information
    """
    with open(file_name, 'w') as mesh_file:
        mesh_file.write('MESH2D\n')
        if len(mesh.name) > 0:
            mesh_file.write(f'MESHNAME "{mesh.name}"\n')
    if pd.__version__ == '1.4.4':
        mesh.elements.to_csv(file_name, mode='a', sep=' ', index=False, header=False, line_terminator='')
        mesh.nodes.to_csv(file_name, mode='a', sep=' ', index=False, header=False, line_terminator='')
    else:
        mesh.elements.to_csv(file_name, mode='a', sep=' ', index=False, header=False, lineterminator='')
        mesh.nodes.to_csv(file_name, mode='a', sep=' ', index=False, header=False, lineterminator='')


def read_hot_start_file(file_name: str, logger: logging.Logger = None) -> list[HotStartDataSet]:
    """
    Reads a *.hot file with initial conditions for an ADH model.

    Args:
        file_name: file name
        logger: The optional logger

    Returns:
        List of HotStartDataSet classes
    """
    if logger is not None:
        logger.info('Reading hot start file.')
    hot_start_list = []
    with open(file_name, "r") as file:
        reading_data_set = False
        reading_vector = False
        number_nodes = 0
        istat = 0
        data_set_values = []
        activity = []
        data_set = HotStartDataSet()
        for line_number, line in enumerate(file):
            # remove new line character
            line = line.rstrip()
            line_split = line.split(' ')
            # remove blank strings
            line_split[:] = (part for part in line_split if part != '')

            err_str = f'Invalid format on line {line_number + 1}.'
            # skip blank line
            if len(line_split) == 0 or line_split[0] == '':
                continue

            if line_number == 0 and line_split[0] != 'DATASET':
                raise IOError('First line in file must be "DATASET".')

            if line_split[0] == 'DATASET' or line_split[0] == 'OBJTYPE' or \
                    line_split[0] == 'TIMEUNITS' or line_split[0] == 'RT_JULIAN':
                continue
            elif line_split[0] == 'BEGSCL':
                if reading_data_set:
                    raise IOError('{} Found "BEGSCL" but currently reading data set.'.format(err_str))
                reading_data_set = True
            elif line_split[0] == 'BEGVEC':
                if reading_data_set:
                    raise IOError('{} Found "BEGVEC" but currently reading data set.'.format(err_str))
                reading_data_set = True
                reading_vector = True
            elif line_split[0] == 'ND':
                if not reading_data_set:
                    raise IOError('{} Found "ND" but no "BEGSCL" or "BEGVEC" previously read.'.format(err_str))
                try:
                    number_nodes = int(line_split[1])
                except Exception:
                    raise IOError('{} Unable to read number of nodes (ND).'.format(err_str))
            elif line_split[0] == 'NC':
                if not reading_data_set:
                    raise IOError('{} Found "NC" but no "BEGSCL" or "BEGVEC" previously read.'.format(err_str))
                try:
                    data_set.number_of_cells = int(line_split[1])
                except Exception:
                    raise IOError('{} Unable to read number of cells (NC).'.format(err_str))
            elif line_split[0] == 'NAME':
                if not reading_data_set:
                    raise IOError('{} Found "NAME" but no "BEGSCL" or "BEGVEC" previously read.'.format(err_str))
                try:
                    data_set.name = line_split[1].replace('"', '').replace("'", "")
                    if len(data_set.name) < 1:
                        raise Exception
                except Exception:
                    raise IOError('{} Unable to read name.'.format(err_str))
            elif line_split[0] == 'TS':
                if not reading_data_set:
                    raise IOError('{} Found "TS" but no "BEGSCL" or "BEGVEC" previously read.'.format(err_str))
                try:
                    istat = int(line_split[1])
                    time = float(line_split[2])  # noqa F841
                except Exception:
                    raise IOError('{} Unable to read time index and time (TS).'.format(err_str))
            elif line_split[0] == 'ENDDS':
                if not reading_data_set:
                    raise IOError('{} Found "ENDDS" but no "BEGSCL" previously read.'.format(err_str))
                if len(data_set_values) != number_nodes:
                    raise IOError('{} Found "ENDDS" but number of data set values read does not equal number of '
                                  'nodes (ND).'.format(err_str))
                if reading_vector:
                    labels = ['VX', 'VY']
                else:
                    labels = ['S']
                data_set.values = pd.DataFrame.from_records(data_set_values, columns=labels)
                hot_start_list.append(data_set)
                if logger is not None:
                    logger.info(f'Read hot start data set {data_set.name} successfully.')
                data_set = HotStartDataSet()
                reading_data_set = False
                reading_vector = False
                activity = []
                data_set_values = []
            else:
                # read the data set values. There should be number_nodes lines
                if number_nodes < 1:
                    raise IOError('{} Number of nodes not read (ND).'.format(err_str))
                if istat == 1 and len(activity) < number_nodes:
                    try:
                        activity.append(int(line_split[0]))
                    except Exception:
                        raise IOError('{} Unable to read data set activity value.'.format(err_str))
                try:
                    if not reading_vector:
                        data_set_values.append([float(line_split[0])])
                    else:
                        data_set_values.append([float(line_split[0]), float(line_split[1])])
                except Exception:
                    raise IOError('{} Unable to read data set value.'.format(err_str))
    if logger is not None and len(hot_start_list) == 0:
        logger.warning('No hot start data sets found.')
    return hot_start_list


def write_hot_start_file(file_name, hot_start_list):
    """
    Writes the ADH hot start file from a list of hot start data sets.

    Args:
        file_name: file name for *.hot file
        hot_start_list: list of HotStartDataSet classes
    """
    with open(file_name, 'w') as mesh_file:
        for ht in hot_start_list:
            mesh_file.write('DATASET\nOBJTYPE "mesh2d"\n')
            if len(ht.values.columns) > 1:
                mesh_file.write('BEGVEC\n')
                if len(ht.values.columns) < 3:
                    ht.values['z'] = 0.0
            else:
                mesh_file.write('BEGSCL\n')
            mesh_file.write(f'ND {len(ht.values)}\n')
            mesh_file.write(f'NC {ht.number_of_cells}\n')
            mesh_file.write(f'NAME "{ht.name}"\n')
            mesh_file.write('TS 0 0\n')
            mesh_file.write(ht.values.to_csv(sep=' ', index=False, header=False).replace('\r\n', '\n'))
            mesh_file.write('ENDDS\n')


def _format_missing_cards(missing_cards: list[str]) -> str:
    """
    This method takes in a list of missing cards and formats them into a string.

    Args:
        missing_cards: A list of strings representing the missing cards.

    Returns:
        A string representing the formatted missing cards.
    """
    if len(missing_cards) > 0:
        if len(missing_cards) > 1:
            missing_cards.sort()
            last_missing_card = missing_cards.pop()
            return ', '.join(missing_cards) + ', and ' + last_missing_card
        else:
            return missing_cards[0]
    else:
        return ''


def _log_missing_cards(required_cards_found: set[str], logger):
    """Log required BC file cards that were missing.

    Args:
        required_cards_found: set of cards to check
        logger: logger object
    """
    required_cards = {
        'OP SW2', 'OP INC', 'OP TRN', 'OP BLK',
        'OP PRE', 'IP NIT', 'IP MIT', 'MP MU',
        'MP G', 'MP MUC', 'MP RHO', 'MP ML',
        'MP SRT', 'MTS', 'XY1', 'TC T0',
        'TC TF', 'END'
    }
    missing_cards = list(required_cards.difference(required_cards_found))
    formatted_missing_cards = _format_missing_cards(missing_cards)
    if formatted_missing_cards:
        logger.warning(f'The following required cards were missing from the BC file:')
        logger.warning(formatted_missing_cards + '.')
        logger.warning('Default values may be applied.')


def _extract_constituent_ids(sediment) -> list[int]:
    """
    Extracts integer constituent IDs from a DataFrame.

    Args:
        sediment: A pandas DataFrame containing a column named 'ID', which stores values
            intended to be extracted and converted to integers.

    Returns:
        A list of constituent IDs.
    """
    return [int(row['ID']) for _, row in sediment.iterrows()] if not sediment.empty else []


def read_bc_file(file_name, bc_class, model_class, logger=None):
    """
    Reads the *.bc file and fills the AdhModel class.

    Args:
        file_name: File name of the *.bc file
        bc_class: Boundary Condition class
        model_class: Model Control class
        logger: Logger instance
    """
    required_cards_found = set()
    if logger is not None:
        logger.info('Reading BC file.')
    # RegEx code to split on whitespace and preserve quoted strings
    comment_pattern = re.compile(r'!.*|[^"\s]\S*|".+?"')

    # set all not required to off
    model_class.operation_parameters.set_not_required(False)
    model_class.constituent_properties.set_not_required(False)
    model_class.model_constants.set_not_required(False)

    bc_string_cards = {'NDS', 'EGS', 'MDS', 'MTS'}
    bc_cards = {'NB', 'DB', 'BR', 'OB', 'OFF', 'WER', 'WRS', 'FLP', 'FGT', 'SLUICE', 'SLS', 'EQ'}
    xy_series_cards = {'XY1', 'XY2', 'OS', 'SERIES'}
    pc_cards = {'PC', 'OC', 'FLX', 'SOUT', 'FOUT', 'PRN'}
    temp_data = {}
    xy_data_list = []
    with open(file_name, "r") as file:
        xy_by_id = {}
        for line_number, line in enumerate(file):
            line = line.rstrip()
            # Split off comments and set comment value if no comment (requested to export string if we renumber items)
            tokens = comment_pattern.findall(line)
            line_split = [token for token in tokens if not token.startswith(tuple('!"'))]
            comment = ' '.join(tokens[len(line_split):])
            if not comment:
                comment = f' ! Original: {line}'

            # skip blank line, comment line
            if len(line_split) == 0 or line_split[0] == '' or line_split[0][0] == '!':
                continue

            try:
                card = ''
                if line_split[0] == 'OP':
                    card = read_op_cards(line_split, model_class, temp_data, line)
                elif line_split[0] == 'IP':
                    card = read_ip_cards(line_split, model_class, temp_data, line)
                elif line_split[0] == 'CN':
                    read_cn_cards(line_split, model_class, temp_data, line, comment)
                elif line_split[0] == 'MP':
                    card = read_mp_cards(line_split, bc_class, model_class, temp_data, line, comment)
                elif line_split[0] == 'SP':
                    read_sp_cards(line_split, model_class, temp_data, line)
                elif line_split[0] == 'SDV':
                    read_sdv_cards(line_split, model_class, temp_data, line)
                elif line_split[0] in bc_string_cards:
                    card = read_bc_string_cards(line_split, temp_data)
                elif line_split[0] in xy_series_cards:
                    card = read_xy_cards(line_split, temp_data, comment)
                elif line_split[0] == 'FR':
                    read_fr_cards(line_split, temp_data, line)
                elif line_split[0] in pc_cards:
                    read_pc_cards(line_split, model_class, temp_data, line)
                elif line_split[0] in bc_cards:
                    read_bc_cards(line_split, bc_class, model_class, temp_data, line)
                elif line_split[0] == 'TC':
                    card = read_tc_cards(line_split, bc_class, model_class, temp_data, line)
                elif line_split[0] == 'XYC':
                    xy_data_list = []
                    ts_id = int(line_split[1])
                    # if this time-series exists, then set the type to SERIES WIND
                    if ts_id in bc_class.time_series:
                        bc_class.time_series[ts_id].series_type = 'SERIES WIND'
                        bc_class.time_series[ts_id].x_location = float(line_split[2])
                        bc_class.time_series[ts_id].y_location = float(line_split[3])
                    else:
                        xy_by_id[ts_id] = (float(line_split[2]), float(line_split[3]))
                elif 'xy_type' in temp_data:
                    xyt = temp_data['xy_type']
                    if xyt == 'SERIES AWRITE':
                        labels = ['START_TIME', 'END_TIME', 'TIME_STEP_SIZE', 'UNITS']
                        xy_data_list.append([float(line_split[0]), float(line_split[1]), float(line_split[2]),
                                             int(line_split[3])])
                    elif xyt == 'SERIES WIND' or xyt == 'SERIES WAVE':
                        labels = ['X', 'Y', 'Z']
                        xy_data_list.append([float(line_split[0]), float(line_split[1]), float(line_split[2])])
                    else:
                        labels = ['X', 'Y']
                        xy_data_list.append([float(line_split[0]), float(line_split[1])])

                    # set the time step option in the output control if we read 'SERIES DT'
                    if xyt == 'SERIES DT':
                        model_class.time_control.time_step_option = 'Time step series (SERIES DT)'
                        model_class.time_control.max_time_step_size_time_series = temp_data['xy_id']
                    if len(xy_data_list) == temp_data['xy_number_points']:
                        ts = TimeSeries()
                        ts.xy_name = temp_data['xy_name']
                        ts.series_type = xyt
                        if xyt == 'SERIES AWRITE':
                            # objs = list(bc_class.output_control.param.output_control_option.get_range())
                            model_class.output_control.output_control_option = 'Specify autobuild (SERIES AWRITE)'
                        ts.units = temp_data['xy_units']
                        ts.output_units = temp_data['xy_output_units']
                        ts.time_series = pd.DataFrame.from_records(xy_data_list, columns=labels)
                        if 'xy_x_location' in temp_data:
                            ts.x_location = temp_data['xy_x_location']
                            ts.y_location = temp_data['xy_y_location']
                            temp_data.pop('xy_x_location')
                            temp_data.pop('xy_y_location')
                        xy_data_list = []
                        # set time series ID as both the key and in the ID column
                        ts.series_id = temp_data['xy_id']
                        if ts.series_id in xy_by_id:
                            ts.x_location, ts.y_location = xy_by_id[ts_id]
                        bc_class.time_series[temp_data['xy_id']] = ts
                        # empty out temp_data  #todo poor practice
                        temp_data.pop('xy_number_points')
                        temp_data.pop('xy_id')
                        temp_data.pop('xy_type')
                        temp_data.pop('xy_units')
                        temp_data.pop('xy_output_units')
                elif line_split[0] == 'END':
                    card = 'END'
                else:
                    advanced_list = temp_data.setdefault('advanced_list', [])
                    advanced_list.append(line)
                if card:
                    required_cards_found.add(card)
            except Exception:
                msg = 'Error reading line {} of file: {}.\nLine: {}'.format(line_number + 1,
                                                                            os.path.basename(file_name), line)
                raise IOError(msg)

    if logger is not None:
        _log_missing_cards(required_cards_found, logger)
    lists_to_data_frames(bc_class, model_class, temp_data)


def lists_to_data_frames(bc_class, model_class, temp_data):
    """
    Converts temporary lists to DataFrames in the AdhModel class.

    Args:
        bc_class: The ADH boundary condition class that holds the data
        model_class: The ADH model control class that holds the data
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
    """
    if 'bc_string_list' in temp_data:
        labels = ['CARD', 'ID', 'ID_0', 'ID_1']
        df = pd.DataFrame.from_records(temp_data['bc_string_list'], columns=labels)
        for x in range(1, len(labels)):
            df[labels[x]] = df[labels[x]].astype(dtype='Int64')
        bc_class.boundary_strings = df
    if 'bc_list' in temp_data:
        labels = ['CARD', 'CARD_2', 'STRING_ID', 'XY_ID_0', 'XY_ID_1', 'XY_ID_2']
        df = pd.DataFrame.from_records(temp_data['bc_list'], columns=labels)
        for x in range(2, len(labels)):
            df[labels[x]] = df[labels[x]].astype(dtype='Int64')
        bc_class.solution_controls = df
    if 'cn_con_df' in temp_data:
        model_class.constituent_properties.general_constituents = temp_data['cn_con_df'].build()
    if 'cn_cla_df' in temp_data:
        model_class.sediment_constituent_properties.clay = temp_data['cn_cla_df'].build()
    if 'cn_snd_df' in temp_data:
        model_class.sediment_constituent_properties.sand = temp_data['cn_snd_df'].build()
    if 'nb_sdr_list' in temp_data:
        labels = ['CARD', 'CARD_1', 'S_ID', 'COEF_A', 'COEF_B', 'COEF_C', 'COEF_D', 'COEF_E']
        df = pd.DataFrame.from_records(temp_data['nb_sdr_list'], columns=labels)
        bc_class.stage_discharge_boundary = df
    if 'fr_list' in temp_data:
        labels = ['CARD', 'CARD_2', 'STRING_ID', 'REAL_01', 'REAL_02', 'REAL_03', 'REAL_04', 'REAL_05']
        df = pd.DataFrame.from_records(temp_data['fr_list'], columns=labels)
        bc_class.friction_controls = df
    if 'br_list' in temp_data:
        labels = ['CARD', 'CARD_1', 'C_0', 'C_1', 'C_2', 'C_3', 'C_4', 'C_5', 'C_6', 'C_7', 'C_8']
        df = pd.DataFrame.from_records(temp_data['br_list'], columns=labels)
        bc_class.breach_controls = df
    if 'wrs_list' in temp_data:
        labels = ['CARD', 'WRS_NUMBER', 'S_UPSTREAM', 'S_DOWNSTREAM', 'WS_UPSTREAM', 'WS_DOWNSTREAM', 'LENGTH',
                  'CREST_ELEV', 'HEIGHT']
        df = pd.DataFrame.from_records(temp_data['wrs_list'], columns=labels)
        bc_class.weirs = df
    if 'fgt_list' in temp_data:
        labels = ['CARD', 'FGT_NUMBER', 'USER', 'S_UPSTREAM', 'S_DOWNSTREAM', 'FS_UPSTREAM', 'FS_DOWNSTREAM', 'COEF_A',
                  'COEF_B', 'COEF_C', 'COEF_D', 'COEF_E', 'COEF_F', 'LENGTH']
        df = pd.DataFrame.from_records(temp_data['fgt_list'], columns=labels)
        bc_class.flap_gates = df
    if 'sls_list' in temp_data:
        labels = ['CARD', 'SLS_NUMBER', 'S_UPSTREAM', 'S_DOWNSTREAM', 'SS_UPSTREAM', 'SS_DOWNSTREAM', 'LENGTH',
                  'TS_OPENING']
        df = pd.DataFrame.from_records(temp_data['sls_list'], columns=labels)
        bc_class.sluice_gates = df
    if 'flx_list' in temp_data:
        labels = ['CARD', 'S_ID']
        df = pd.DataFrame.from_records(temp_data['flx_list'], columns=labels)
        model_class.output_control.output_flow_strings = df
    if 'prn_list' in temp_data:
        labels = ['CARD', 'NODE', 'POINT_ID']
        df = pd.DataFrame.from_records(temp_data['prn_list'], columns=labels)
        model_class.output_control.nodal_output = df
    if 'advanced_list' in temp_data:
        labels = ['CARD_LINE']
        df = pd.DataFrame(temp_data['advanced_list'], columns=labels)
        model_class.advanced_cards.advanced_cards = df
    if 'global_bed_layers' in temp_data:
        labels = ['CARD', 'BED_LAYER_ID', 'THICKNESS']
        df = pd.DataFrame(temp_data['global_bed_layers'], columns=labels)
        model_class.sediment_properties.number_bed_layers = len(temp_data['global_bed_layers'])
        model_class.sediment_properties.global_bed_layers = df
    if 'material_bed_layers' in temp_data:
        labels = ['CARD', 'BED_LAYER_ID', 'MATERIAL_ID', 'THICKNESS']
        df = pd.DataFrame(temp_data['material_bed_layers'], columns=labels)
        model_class.sediment_properties.material_bed_layers = df
    if 'bed_layer_grain' in temp_data:
        labels = ['MATERIAL_ID', 'BED_LAYER_ID', 'CONSTITUENT_ID', 'FRACTION']
        df = pd.DataFrame(temp_data['bed_layer_grain'], columns=labels)
        model_class.sediment_properties.bed_layer_grain_fractions = df
    if 'global_bed_cohesive' in temp_data:
        labels = ['MP', 'CARD', 'BED_LAYER_ID', 'POROSITY', 'CRITICAL_SHEAR',
                  'EROSION_CONSTANT', 'EROSION_EXPONENT']
        df = pd.DataFrame(temp_data['global_bed_cohesive'], columns=labels)
        model_class.sediment_properties.global_cohesive_bed = df
    if 'material_bed_cohesive' in temp_data:
        labels = ['MP', 'CARD', 'BED_LAYER_ID', 'MATERIAL_ID', 'POROSITY', 'CRITICAL_SHEAR',
                  'EROSION_CONSTANT', 'EROSION_EXPONENT']
        df = pd.DataFrame(temp_data['material_bed_cohesive'], columns=labels)
        model_class.sediment_properties.material_cohesive_bed = df
    if 'global_consolidation' in temp_data:
        labels = ['MP', 'CARD', 'TIME_ID', 'ELAPSED_TIME', 'POROSITY', 'CRITICAL_SHEAR',
                  'EROSION_CONSTANT', 'EROSION_EXPONENT']
        df = pd.DataFrame(temp_data['global_consolidation'], columns=labels)
        model_class.sediment_properties.number_consolidation_times = len(temp_data['global_consolidation'])
        model_class.sediment_properties.global_consolidation = df
    if 'material_consolidation' in temp_data:
        labels = ['MP', 'CARD', 'MATERIAL_ID', 'TIME_ID', 'ELAPSED_TIME', 'POROSITY',
                  'CRITICAL_SHEAR', 'EROSION_CONSTANT', 'EROSION_EXPONENT']
        df = pd.DataFrame(temp_data['material_consolidation'], columns=labels)
        model_class.sediment_properties.material_consolidation = df
    if 'material_diffusion' in temp_data:
        labels = ['CARD', 'MATERIAL_ID', 'DIFFUSION']
        df = pd.DataFrame(temp_data['material_diffusion'], columns=labels)
        model_class.sediment_properties.material_diffusion = df
    if 'material_displacement_off' in temp_data:
        labels = ['CARD', 'MATERIAL_ID']
        df = pd.DataFrame(temp_data['material_displacement_off'], columns=labels)
        model_class.sediment_properties.material_displacement_off = df
    if 'material_local_scour' in temp_data:
        labels = ['CARD', 'MATERIAL_ID']
        df = pd.DataFrame(temp_data['material_local_scour'], columns=labels)
        model_class.sediment_properties.material_local_scour = df
    if 'sediment_diversion' in temp_data:
        labels = ['CARD', 'S_ID', 'TOP', 'BOTTOM', 'BOTTOM_MAIN']
        df = pd.DataFrame(temp_data['sediment_diversion'], columns=labels)
        model_class.sediment_properties.sediment_diversion = df


def read_op_cards(line_split, model_class, temp_data, line) -> str:
    """
    Reads the OP cards from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        model_class: The ADH simulation class that will hold this information
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
        line: The original line from the file, trimmed of endline and ending comments

    Returns:
        The first two card items. For example "OP SW2."
    """
    try:
        part_1 = line_split[1]
        card = f'OP {part_1}'
        if part_1 == 'SW2':
            model_class.operation_parameters.physics = 'SW2'
        elif part_1 == 'SW3':
            model_class.operation_parameters.physics = 'SW3'
        elif part_1 == 'INC':
            model_class.operation_parameters.incremental_memory = int(line_split[2])
        elif part_1 == 'TRN':
            model_class.operation_parameters.transport = int(line_split[2])
        elif part_1 == 'BLK':
            model_class.operation_parameters.blocks_per_processor = int(line_split[2])
        elif part_1 == 'PRE':
            model_class.operation_parameters.preconditioner_type = int(line_split[2])
        elif part_1 == 'BT':
            model_class.operation_parameters.vessel = True
        elif part_1 == 'BTS':
            model_class.operation_parameters.vessel_entrainment = True
        elif part_1 == 'TEM':
            model_class.operation_parameters.second_order_temporal_coefficient_active = True
            model_class.operation_parameters.second_order_temporal_coefficient = float(line_split[2])
        elif part_1 == 'TPG':
            model_class.operation_parameters.petrov_galerkin_coefficient_active = True
            model_class.operation_parameters.petrov_galerkin_coefficient = float(line_split[2])
        elif part_1 == 'NF2':
            model_class.operation_parameters.velocity_gradient = True
        elif part_1 == 'WND':
            model_class.operation_parameters.wind = True
        elif part_1 == 'WAV':
            model_class.operation_parameters.wave = True
        elif part_1 == 'DAM':
            model_class.operation_parameters.dam = True
        elif part_1 == 'DIF':
            model_class.operation_parameters.diffusive_wave = True
        else:
            advanced_list = temp_data.setdefault('advanced_list', [])
            advanced_list.append(line)
    except Exception:
        raise IOError("Error reading OP card from *.bc file.")
    return card


def read_ip_cards(line_split, model_class, temp_data, line) -> str:
    """
    Reads the IP cards from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        model_class: The ADH simulation class that will hold this information
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
        line: The original line from the file, trimmed of endline and ending comments

    Returns:
        The first two card items. For example "IP NIT."
    """
    try:
        split_1 = line_split[1]
        card = f'IP {split_1}'
        if split_1 == 'NIT':
            model_class.iteration_parameters.non_linear_iterations = int(line_split[2])
        elif split_1 == 'NTL':
            model_class.iteration_parameters.non_linear_residual_tolerance = float(line_split[2])
            temp_data['IP NTL'] = True
            objects = list(model_class.iteration_parameters.param.non_linear_tolerance_option.get_range())
            if 'IP ITL' in temp_data:
                model_class.iteration_parameters.non_linear_tolerance_option = objects[0]
            else:
                model_class.iteration_parameters.non_linear_tolerance_option = objects[1]
        elif split_1 == 'ITL':
            model_class.iteration_parameters.non_linear_incremental_tolerance = float(line_split[2])
            temp_data['IP ITL'] = True
            objects = list(model_class.iteration_parameters.param.non_linear_tolerance_option.get_range())
            if 'IP NTL' in temp_data:
                model_class.iteration_parameters.non_linear_tolerance_option = objects[0]
            else:
                model_class.iteration_parameters.non_linear_tolerance_option = objects[2]
        elif split_1 == 'MIT':
            model_class.iteration_parameters.linear_iterations = int(line_split[2])
        else:
            advanced_list = temp_data.setdefault('advanced_list', [])
            advanced_list.append(line)
    except Exception:
        raise IOError("Error reading IP card from *.bc file.")
    return card


def read_cn_cards(line_split, model_class, temp_data, line, comment):
    """
    Reads the CN cards from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        model_class: The ADH simulation class that will hold this information
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
        line: The original line from the file, trimmed of endline and ending comments
        comment: The trailing comment from the original line
    """
    try:
        card = line_split[1]
        # Extract the constituent name if provided
        con_name = comment.removeprefix("! Name: ") if comment.startswith("! Name: ") else ""

        if card == 'CON':
            schema = GENERAL_CONSTITUENTS_COLUMN_TYPES
            builder = temp_data.setdefault('cn_con_df', DataFrameBuilder(schema))
            builder.add(ID=int(line_split[2]), NAME=con_name, CONC=float(line_split[3]))

        elif card == 'SND':
            schema = SAND_COLUMN_TYPES
            builder = temp_data.setdefault('cn_snd_df', DataFrameBuilder(schema))
            builder.add(
                ID=int(line_split[2]), NAME=con_name, CONCENTRATION=float(line_split[3]),
                GRAIN_DIAMETER=float(line_split[4]), SPECIFIC_GRAVITY=float(line_split[5]),
                POROSITY=float(line_split[6])
            )

        elif card == 'CLA':
            schema = CLAY_COLUMN_TYPES
            builder = temp_data.setdefault('cn_cla_df', DataFrameBuilder(schema))
            builder.add(
                ID=int(line_split[2]), NAME=con_name, CONCENTRATION=float(line_split[3]),
                GRAIN_DIAMETER=float(line_split[4]), SPECIFIC_GRAVITY=float(line_split[5]),
                POROSITY=float(line_split[6]), CRITICAL_SHEAR_EROSION=float(line_split[7]),
                EROSION_RATE=float(line_split[8]), CRITICAL_SHEAR_DEPOSITION=float(line_split[9]),
                FREE_SETTLING_VELOCITY=float(line_split[10])
            )

        elif card == 'SAL':
            model_class.constituent_properties.salinity = True
            model_class.constituent_properties.salinity_id = int(line_split[2])
            model_class.constituent_properties.reference_concentration = float(line_split[3])

        elif card == 'TMP':
            model_class.constituent_properties.temperature = True
            model_class.constituent_properties.temperature_id = int(line_split[2])
            model_class.constituent_properties.reference_temperature = float(line_split[3])
            val = int(line_split[4])
            if val:
                model_class.constituent_properties.air_water_heat_transfer = True

        elif card == 'VOR':
            model_class.constituent_properties.vorticity = True
            model_class.constituent_properties.vorticity_id = int(line_split[2])
            model_class.constituent_properties.vorticity_normalization = float(line_split[3])
            model_class.constituent_properties.vorticity_as_term = float(line_split[4])
            model_class.constituent_properties.vorticity_ds_term = float(line_split[5])

        else:
            # Handle advanced cards
            advanced_list = temp_data.setdefault('advanced_list', [])
            advanced_list.append(line)

    except Exception:
        raise IOError("Error reading CN card from *.bc file.")


def read_mp_cards(line_split, bc_class, model_class, temp_data, line, comment: str) -> str | None:
    """
    Reads the MP cards from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        bc_class: The ADH simulation boundary condition class that will hold this information
        model_class: The ADH simulation class that will hold this information
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
        line: The original line from the file, trimmed of endline and ending comments
        comment: The trailing comment from the original line

    Returns:
        The first two card items. For example "MP MU."
    """
    try:
        split_1 = line_split[1]
        card = f'MP {split_1}'
        if split_1 == 'MU':
            model_class.model_constants.kinematic_viscosity = float(line_split[2])
        elif split_1 == 'G':
            model_class.model_constants.gravity = float(line_split[2])
        elif split_1 == 'MUC':
            model_class.model_constants.mannings_unit_constant = float(line_split[2])
        elif split_1 == 'RHO':
            model_class.model_constants.density = float(line_split[2])
        elif split_1 == 'DTL':
            model_class.model_constants.enable_wet_dry_stabilization = True
            model_class.model_constants.wet_dry_stabilization_length = float(line_split[2])
        elif split_1 == 'SBA':
            bed_layer_id = int(line_split[2])
            sba_list = temp_data.setdefault('global_bed_layers', [])
            sba_list.append(['MP SBA', bed_layer_id, float(line_split[3])])
            grain_list = temp_data.setdefault('bed_layer_grain', [])
            fraction_list = []
            for idx in range(4, len(line_split)):
                fraction_list.append([0, bed_layer_id, idx - 3, float(line_split[idx])])
            grain_list.extend(fraction_list)
        elif split_1 == 'CBA':
            bed_layer_id = int(line_split[2])
            cba_list = temp_data.setdefault('global_bed_cohesive', [])
            cba_list.append(['MP', 'CBA', bed_layer_id, float(line_split[3]), float(line_split[4]),
                             float(line_split[5]), float(line_split[6])])
        elif split_1 == 'CPA':
            cpa_list = temp_data.setdefault('global_consolidation', [])
            cpa_list.append(['MP', 'CPA', int(line_split[2]), float(line_split[3]), float(line_split[4]),
                             float(line_split[5]), float(line_split[6]), float(line_split[7])])
        elif split_1 in ['SBN', 'CBN', 'CPN']:
            # Node based versions of these cards will be put into advanced cards
            advanced_list = temp_data.setdefault('advanced_list', [])
            advanced_list.append(line)
        elif split_1 in ['NBL', 'NCP']:
            pass  # Sediment cards that do not need to be read in
        elif split_1 in ['BLD', 'NDM', 'LSM', 'SBM', 'CBM', 'CPM']:
            # These cards have a sediment material id.
            if split_1 == 'BLD':
                material_id = int(line_split[2])
                bld_list = temp_data.setdefault('material_diffusion', [])
                bld_list.append(['MP BLD', material_id, float(line_split[4])])
            elif split_1 == 'NDM':
                material_id = int(line_split[2])
                ndm_list = temp_data.setdefault('material_displacement_off', [])
                ndm_list.append(['MP NDM', material_id])
            elif split_1 == 'LSM':
                material_id = int(line_split[2])
                lsm_list = temp_data.setdefault('material_local_scour', [])
                lsm_list.append(['MP LSM', material_id])
            elif split_1 == 'SBM':
                bed_layer_id = int(line_split[2])
                material_id = int(line_split[3])
                sbm_list = temp_data.setdefault('material_bed_layers', [])
                sbm_list.append(['MP SBM', bed_layer_id, material_id, float(line_split[4])])
                grain_list = temp_data.setdefault('bed_layer_grain', [])
                fraction_list = []
                for idx in range(5, len(line_split)):
                    fraction_list.append([material_id, bed_layer_id, idx - 3, float(line_split[idx])])
                grain_list.extend(fraction_list)
            elif split_1 == 'CBM':
                bed_layer_id = int(line_split[2])
                material_id = int(line_split[3])
                cbm_list = temp_data.setdefault('material_bed_cohesive', [])
                cbm_list.append(['MP', 'CBM', bed_layer_id, material_id, float(line_split[4]), float(line_split[5]),
                                 float(line_split[6]), float(line_split[7])])
            elif split_1 == 'CPM':
                cpm_list = temp_data.setdefault('material_consolidation', [])
                cpm_list.append(['MP', 'CPM', int(line_split[2]), int(line_split[3]), float(line_split[4]),
                                 float(line_split[5]), float(line_split[6]), float(line_split[7]),
                                 float(line_split[8])])
        else:
            # check to see if there are enough values
            if len(line_split) <= 2:
                advanced_list = temp_data.setdefault('advanced_list', [])
                advanced_list.append(line)
                return None
            # get material ID
            if split_1 != 'WND':
                material_id = int(line_split[2])
            else:
                material_id = int(line_split[3])

            if material_id in bc_class.material_properties:
                material_property = bc_class.material_properties[material_id]
            else:
                # get material property class
                material_property = MaterialProperties()
                # set "not required" to "deactivated"
                material_property.set_not_required(False)
                bc_class.material_properties[material_id] = material_property

            # set material name if available
            if comment.startswith("! Name: "):
                material_name = comment.removeprefix("! Name: ")
                material_property.material_name = material_name

            if split_1 == 'EVS':
                material_property.eddy_viscosity_method = 'Constant (EVS)'
                material_property.constant_eddy_viscosity = float(line_split[3])
                # material_property.vyy_eddy_viscosity = float(line_split[4])
                # material_property.vxy_eddy_viscosity = float(line_split[5])
            elif split_1 == 'EEV':
                material_property.eddy_viscosity_method = 'Estimated (EEV)'
                material_property.estimated_eddy_viscosity_method = int(line_split[3])
                material_property.estimated_eddy_viscosity_weighting_factor = float(line_split[4])
                material_property.estimated_eddy_viscosity_minimum = float(line_split[5])
            elif split_1 == 'COR':
                material_property.coriolis = True
                material_property.coriolis_latitude = float(line_split[3])
            elif split_1 == 'ML':
                material_property.max_refinement_level = int(line_split[3])
            elif split_1 == 'SRT':
                material_property.refinement_tolerance = float(line_split[3])
            elif split_1 == 'DF' or split_1 == 'TRT':
                constituent_id = int(line_split[3])
                transport_property = material_property.transport_properties.setdefault(constituent_id,
                                                                                       MaterialTransportProperties())
                if split_1 == 'TRT':
                    transport_property.refinement_tolerance = float(line_split[4])
                else:
                    transport_property.turbulent_diffusion_rate = float(line_split[4])
                    transport_property.use_diffusion_coefficient = True
            elif split_1 == 'WND':
                material_property.wind_properties = True
                if line_split[2] == 'STR':
                    material_property.stress_formulation = int(line_split[4])
                elif line_split[2] == 'ATT':
                    material_property.attenuation = float(line_split[4])
            else:
                advanced_list = temp_data.setdefault('advanced_list', [])
                advanced_list.append(line)
    except Exception:
        raise IOError("Error reading MP card from *.bc file.")
    return card


def read_sp_cards(line_split, model_class, temp_data, line):
    """
    Reads the SP cards from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        model_class: The ADH simulation class that will hold this information
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
        line: The original line from the file, trimmed of endline and ending comments
    """
    try:
        card = line_split[1]
        if card == 'CSV':
            model_class.sediment_properties.cohesive_settling = int(line_split[2])
            if len(line_split) == 7 and model_class.sediment_properties.cohesive_settling == 1:
                model_class.sediment_properties.cohesive_settling_a = float(line_split[3])
                model_class.sediment_properties.cohesive_settling_b = float(line_split[4])
                model_class.sediment_properties.cohesive_settling_m = float(line_split[5])
                model_class.sediment_properties.cohesive_settling_n = float(line_split[6])
        elif card == 'WWS':
            model_class.sediment_properties.wind_wave_stress = int(line_split[2])
        elif card == 'NSE':
            model_class.sediment_properties.suspended_entrainment = int(line_split[2])
        elif card == 'NBE':
            model_class.sediment_properties.bedload_entrainment = int(line_split[2])
            if len(line_split) == 5 and model_class.sediment_properties.bedload_entrainment == 3:
                model_class.sediment_properties.critical_shear_sand = float(line_split[3])
                model_class.sediment_properties.critical_shear_clay = float(line_split[4])
        elif card == 'HID':
            model_class.sediment_properties.hiding_factor = int(line_split[2])
            if len(line_split) == 4 and model_class.sediment_properties.hiding_factor == 3:
                model_class.sediment_properties.hiding_factor_exponent = float(line_split[3])
        elif card == 'SIF':
            model_class.sediment_properties.use_infiltration_factor = 1
            model_class.sediment_properties.infiltration_factor = int(line_split[2])
        else:
            advanced_list = temp_data.setdefault('advanced_list', [])
            advanced_list.append(line)
    except Exception:
        raise IOError("Error reading SP card from *.bc file.")


def read_sdv_cards(line_split, model_class, temp_data, line):
    """
    Reads the SDV cards from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        model_class: Model Control class
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
        line: The original line from the file, trimmed of endline and ending comments
    """
    try:
        diversion = temp_data.setdefault('sediment_diversion', [])
        diversion.append(['SDV', int(line_split[1]), float(line_split[2]), float(line_split[3]), float(line_split[4])])
    except Exception:
        raise IOError("Error reading SDV card from *.bc file.")


def read_xy_cards(line_split, temp_data, comment="") -> str:
    """
    Reads the xy series card from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
        comment: The XY card comment

    Returns:
        The card name. For example "XY1."
    """
    try:
        card = line_split[0]
        # ts = TimeSeries()
        if card == 'XY1':
            temp_data['xy_type'] = 'SERIES BC'
            temp_data['xy_id'] = int(line_split[1])
            temp_data['xy_number_points'] = int(line_split[2])
            temp_data['xy_units'] = int(line_split[3])
            temp_data['xy_output_units'] = int(line_split[4])
        elif card == 'XY2':
            temp_data['xy_type'] = 'SERIES WIND'
            temp_data['xy_id'] = int(line_split[1])
            temp_data['xy_number_points'] = int(line_split[2])
            temp_data['xy_units'] = int(line_split[3])
            temp_data['xy_output_units'] = int(line_split[4])
        elif card == 'SERIES':
            temp_data['xy_type'] = f'{card} {line_split[1]}'
            temp_data['xy_id'] = int(line_split[2])
            temp_data['xy_number_points'] = int(line_split[3])
            if line_split[1] == 'AWRITE':
                temp_data['xy_units'] = temp_data['xy_output_units'] = int(line_split[4])
            elif line_split[1] == 'WIND' or line_split[1] == 'WAVE':
                temp_data['xy_x_location'] = float(line_split[4])
                temp_data['xy_y_location'] = float(line_split[5])
                temp_data['xy_units'] = int(line_split[6])
                temp_data['xy_output_units'] = int(line_split[7])
            else:
                temp_data['xy_units'] = int(line_split[4])
                temp_data['xy_output_units'] = int(line_split[5])
        elif card == 'OS':
            temp_data['xy_type'] = 'SERIES AWRITE'
            temp_data['xy_id'] = int(line_split[1])
            temp_data['xy_number_points'] = int(line_split[2])
            temp_data['xy_units'] = temp_data['xy_output_units'] = int(line_split[3])
        temp_data['xy_name'] = comment
    except Exception:
        raise IOError("Error reading XY card from *.bc file.")
    return card


def read_bc_string_cards(line_split, temp_data) -> str:
    """
    Reads the NDS, EGS, MDS, MTS cards from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file

    Returns:
        The first card item. For example "MTS."
    """
    bc_string_list = temp_data.setdefault('bc_string_list', [])
    record = [None, None, None, None]
    record[0] = line_split[0]  # card NDS, EGS, MDS, MTS
    try:
        record[1] = int(line_split[-1])  # string id
        record[2] = int(line_split[1])  # node id, cell id, material id
        if line_split[0] == 'EGS' or line_split[0] == 'MDS':
            record[3] = int(line_split[2])  # node id, cell id
        bc_string_list.append(record)
    except Exception:
        raise IOError("Error reading boundary string from *.bc file.")
    return record[0]


def read_fr_cards(line_split, temp_data, line):
    """
    Reads the FR cards from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
        line: The original line from the file, trimmed of endline and ending comments
    """
    fr_list = temp_data.setdefault('fr_list', [])
    try:
        if line_split[1] not in ['MNG', 'MNC', 'ERH', 'SAV', 'URV', 'EDO', 'ICE', 'DUN', 'SDK', 'BRD']:
            advanced_list = temp_data.setdefault('advanced_list', [])
            advanced_list.append(line)
            return
        num_vals_dict = {'SAV': 1, 'URV': 2, 'EDO': 4, 'ICE': 4, 'DUN': 3, 'BRD': 1}  # number of parameters minus 1!
        record = [float('NaN') for _ in range(8)]
        record[0] = line_split[0]
        record[1] = line_split[1]
        record[2] = int(line_split[2])
        record[3] = float(line_split[3])
        if record[1] in num_vals_dict:
            for x in range(num_vals_dict[record[1]]):
                record[4 + x] = float(line_split[4 + x])
    except Exception:
        raise IOError("Error reading friction control from *.bc file.")

    fr_list.append(record)


def read_pc_cards(line_split, model_class, temp_data, line):
    """
    Reads the PC cards from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        model_class: The ADH simulation class that will hold this information
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
        line: The original line from the file, trimmed of endline and ending comments
    """
    try:
        card = line_split[1]
        oc = model_class.output_control
        if card == 'ADP':
            oc.print_adaptive_mesh = True
        elif card == 'ELM':
            oc.print_numerical_fish_surrogate = True
        elif card == 'LVL':
            option = int(line_split[2])
            if option == 0:
                oc.screen_output_residual = True
            elif option == 1:
                oc.screen_output_all = True
        elif card == 'MEO':
            oc.screen_output_mass_error = True
        elif line_split[0] == 'OC':
            oc.oc_time_series_id = int(line_split[1])
            objects = list(oc.param.output_control_option.get_range())
            oc.output_control_option = objects[0]
        elif line_split[0] == 'FLX':
            flx_list = temp_data.setdefault('flx_list', [])
            flx_list.append(['FLX', int(line_split[1])])
        elif line_split[0] == 'PRN':
            nodal_list = temp_data.setdefault('prn_list', [])
            nodal_list.append(['PRN', int(line_split[1]), -1])
        elif line_split[0] == 'SOUT':
            if line_split[1] == 'RESID':
                oc.screen_output_residual = True
            elif line_split[1] == 'ALL':
                oc.screen_output_all = True
            elif line_split[1] == 'MERROR':
                oc.screen_output_mass_error = True
            elif line_split[1] == 'NLNODE':
                oc.screen_output_worst_nonlinear_node = True
            elif line_split[1] == 'LNODE':
                oc.screen_output_worst_linear_node = True
        elif line_split[0] == 'FOUT':
            if line_split[1] == 'WIND':
                oc.file_output_wind = True
            elif line_split[1] == 'WAVE':
                oc.file_output_wave = True
            if line_split[1] == "ADAPT":
                if line_split[2] == "GRID":
                    oc.file_output_adapted_grid = True
                if line_split[2] == "SW":
                    oc.file_output_adapted_solution = True
                if line_split[2] == "CON":
                    oc.file_output_adapted_transport = True
            if line_split[1] == "SED":
                oc.file_output_sediment = True
        else:
            advanced_list = temp_data.setdefault('advanced_list', [])
            advanced_list.append(line)

    except Exception:
        raise IOError("Error reading CN card from *.bc file.")


def read_bc_cards(line_split, bc_class, model_class, temp_data, line):
    """
    Reads the solution controls from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        bc_class: The ADH simulation boundary conditions class that will hold this information
        model_class: The ADH simulation class that will hold this information
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
        line: The original line from the file, trimmed of endline and ending comments
    """
    # these items are NOT stored in the AdhModel.solution_controls DataFrame
    try:
        cards = {'BR', 'WER', 'FLP', 'SLUICE'}
        structures = {'WRS', 'FGT', 'SLS'}
        if line_split[0] in cards:
            if line_split[0] == 'BR':
                br_list = temp_data.setdefault('br_list', [])
                if len(line_split) < 11:
                    line_split.extend([float('NaN') for _ in range(len(line_split), 11)])
                br_list.append(line_split[:11])
            return
        if line_split[0] == 'NB' and line_split[1] in structures:
            read_wrs_fgt_sls_cards(line_split, temp_data)
            return
        if line_split[0] == 'NB' and line_split[1] == 'SDR':
            nb_sdr = temp_data.setdefault('nb_sdr_list', [])
            nb_sdr.append(['NB', 'SDR', int(line_split[2]), float(line_split[3]), float(line_split[4]),
                           float(line_split[5]), float(line_split[6]), float(line_split[7])])
            return
        if line_split[0] == 'DB' and line_split[1] == 'RAD':
            model_class.constituent_properties.short_wave_radiation_series = int(line_split[2])
            model_class.constituent_properties.dew_point_temperature_series = int(line_split[3])
            return

        num_vals_dict = {'NB TRN': 1, 'NB OUT': 1, 'DB OVL': 1, 'DB OVH': 2, 'DB TRN': 1, 'EQ TRN': 1}
        nb_cards = ['DIS', 'OVL', 'VEL', 'OTW', 'TRN', 'SPL', 'OUT', 'TID', 'SDR', 'SOURCE']
        db_cards = ['OVL', 'OVH', 'TRN', 'LDE', 'LDH', 'LID']
        eq_cards = ['TRN']
        bc_list = temp_data.setdefault('bc_list', [])
        record = [float('NaN') for _ in range(6)]
        record[0] = line_split[0]
        if record[0] == 'OFF':
            record[1] = int(line_split[1])
        elif record[0] == 'OB':
            record[1] = line_split[1]
            record[2] = int(line_split[2])
        elif (line_split[0] == 'NB' and line_split[1] in nb_cards) or \
                (line_split[0] == 'DB' and line_split[1] in db_cards) or \
                (line_split[0] == 'EQ' and line_split[1] in eq_cards):
            record[1] = line_split[1]
            card = f'{record[0]} {record[1]}'
            record[2] = int(line_split[2])
            if len(line_split) > 3:
                record[3] = int(line_split[3])
                if card in num_vals_dict:
                    for x in range(num_vals_dict[card]):
                        record[4 + x] = int(line_split[4 + x])
        else:
            advanced_list = temp_data.setdefault('advanced_list', [])
            advanced_list.append(line)
            return
    except Exception:
        raise IOError("Error reading solution control from *.bc file.")

    bc_list.append(record)


def read_wrs_fgt_sls_cards(line_split, temp_data):
    """
    Reads the WRS, FGT, and SLS cards from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
    """
    try:
        if line_split[1] == 'WRS':
            wrs_list = temp_data.setdefault('wrs_list', [])
            wrs_list.append(['NB WRS', int(line_split[2]), int(line_split[3]), int(line_split[4]), int(line_split[5]),
                             int(line_split[6]), float(line_split[7]), float(line_split[8]), float(line_split[9])])
        elif line_split[1] == 'FGT':
            fgt_list = temp_data.setdefault('fgt_list', [])
            fgt_list.append(['NB FGT', int(line_split[2]), int(line_split[3]), int(line_split[4]), int(line_split[5]),
                             int(line_split[6]), int(line_split[7]), float(line_split[8]), float(line_split[9]),
                             float(line_split[10]), float(line_split[11]), float(line_split[12]), float(line_split[13]),
                             float(line_split[14])])
        if line_split[1] == 'SLS':
            sls_list = temp_data.setdefault('sls_list', [])
            sls_list.append(['NB SLS', int(line_split[2]), int(line_split[3]), int(line_split[4]), int(line_split[5]),
                             int(line_split[6]), float(line_split[7]), int(line_split[8])])
    except Exception:
        raise IOError("Error reading NB {} card from *.bc file.".format(line_split[1]))


def read_tc_cards(line_split, bc_class, model_class, temp_data, line) -> str:
    """
    Reads the TC cards from the *.bc file.

    Args:
        line_split: list of strings from a parsed line of a *.bc file
        bc_class: The ADH simulation boundary conditions class that will hold this information
        model_class: The ADH simulation class that will hold this information
        temp_data: Dictionary of data that is not stored in the ADH simulation but is needed while reading the file
        line: The original line from the file, trimmed of endline and ending comments

    Returns:
        The first two card items. For example "TC T0."
    """
    try:
        split_1 = line_split[1]
        card = f'TC {split_1}'
        if split_1 == 'T0':
            model_class.time_control.start_time = float(line_split[2])
            if len(line_split) > 3:
                option = int(line_split[3])
                objects = list(model_class.time_control.param.start_time_units.get_range())
                model_class.time_control.start_time_units = objects[option]
        elif split_1 == 'IDT':
            objects = list(model_class.time_control.param.time_step_option.get_range())
            model_class.time_control.time_step_option = objects[1]
            ts_id = int(line_split[2])
            model_class.time_control.max_time_step_size_time_series = ts_id
            # if this time series exists then set the type to SERIES DT
            if ts_id in bc_class.time_series:
                bc_class.time_series[ts_id].series_type = 'SERIES DT'
        elif split_1 == 'TF':
            model_class.time_control.end_time = float(line_split[2])
            if len(line_split) > 3:
                option = int(line_split[3])
                objects = list(model_class.time_control.param.end_time_units.get_range())
                model_class.time_control.end_time_units = objects[option]
        elif split_1 == 'ATF':
            objects = list(model_class.time_control.param.time_step_option.get_range())
            model_class.time_control.time_step_option = objects[2]
            model_class.time_control.auto_time_step_find_min_time_step_size = float(line_split[2])
            model_class.time_control.auto_time_step_find_max_time_step_size_series = int(line_split[3])
        elif split_1 == 'STD':
            objects = list(model_class.time_control.param.time_step_option.get_range())
            model_class.time_control.time_step_option = objects[0]
            model_class.time_control.steady_state_min_time_step_size = float(line_split[2])
            model_class.time_control.steady_state_max_time_step_size = float(line_split[3])
        else:
            advanced_list = temp_data.setdefault('advanced_list', [])
            advanced_list.append(line)
    except Exception:
        raise IOError("Error reading TC card from *.bc file.")
    return card


def write_bc_file(file_name, bc_class, model_class, is_46=False):
    """
    Writes a *.bc simulation give the information in the AdhModel class.

    Args:
        file_name: File name
        bc_class: The ADH boundary conditions simulation class that holds simulation information
        model_class: The ADH simulation class that holds simulation information
        is_46 (bool): True if writing the cards as AdH 4.6 compatible cards.
    """
    with open(file_name, "w") as bc_file:
        bc_file.write('! ADH BC File - written by adhparam\n\n')
        write_op_cards(bc_file, model_class)
        write_ip_cards(bc_file, model_class)
        write_cn_cards(bc_file, model_class)
        write_sp_cards(bc_file, model_class)
        write_mp_cards(bc_file, bc_class, model_class)
        write_boundary_string_cards(bc_file, bc_class)
        write_time_series_cards(bc_file, bc_class, is_46)
        write_fr_cards(bc_file, bc_class)
        write_sdv_cards(bc_file, model_class)
        write_pc_cards(bc_file, model_class, is_46)
        write_solution_control_cards(bc_file, bc_class, model_class)
        write_tc_cards(bc_file, model_class, is_46)
        write_advanced_cards(bc_file, model_class)
        bc_file.write('END\n')


def write_nodal_output_file(file_name, model_class):
    """
    Writes the nodal output mapping file from mesh nodes to map IDs.

    Args:
        file_name: The name of the file where the nodal mapping will be written.
        model_class: An instance of the model containing the output control data.
    """
    nodal_output = model_class.output_control.nodal_output
    if not nodal_output.empty:
        nodal_output.to_csv(file_name)
    elif os.path.isfile(file_name):
        os.remove(file_name)


def number_of_constituents(sim_class):
    """
    Calculates the number of constituents.

    Args:
        sim_class: The ADH simulation class that holds all simulation information

    Returns:
        The number of transport constituents

    """
    num_trn = 0
    cn = sim_class.constituent_properties
    sed_cn = sim_class.sediment_constituent_properties
    if cn.salinity:
        num_trn += 1
    if cn.temperature:
        num_trn += 1
    if cn.vorticity:
        num_trn += 1
    if not cn.general_constituents.empty:
        num_trn += len(cn.general_constituents.index)
    if not sed_cn.sand.empty:
        num_trn += len(sed_cn.sand.index)
    if not sed_cn.clay.empty:
        num_trn += len(sed_cn.clay.index)
    return num_trn


def write_op_cards(bc_file, model_class):
    """
    Writes the OP cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        model_class: The ADH simulation class that holds all simulation information
    """
    op = model_class.operation_parameters
    bc_file.write('! Operation Parameters\n')
    bc_file.write(f'OP {op.physics}\n')
    bc_file.write(f'OP INC {op.incremental_memory}\n')
    bc_file.write(f'OP TRN {op.transport}\n')
    bc_file.write(f'OP BLK {op.blocks_per_processor}\n')
    bc_file.write(f'OP PRE {op.preconditioner_type}\n')
    if op.vessel:
        bc_file.write('OP BT\n')
        if op.vessel_entrainment:
            bc_file.write('OP BTS\n')
    if op.second_order_temporal_coefficient_active:
        bc_file.write(f'OP TEM {op.second_order_temporal_coefficient}\n')
    if op.petrov_galerkin_coefficient_active:
        bc_file.write(f'OP TPG {op.petrov_galerkin_coefficient}\n')
    if op.velocity_gradient:
        bc_file.write('OP NF2\n')
    if op.wind:
        bc_file.write('OP WND\n')
    if op.wave:
        bc_file.write('OP WAV\n')
    if op.dam:
        bc_file.write('OP DAM\n')
    if op.diffusive_wave:
        bc_file.write('OP DIF\n')

    bc_file.write('\n')  # blank line at the end of the Operation Parameters


def write_ip_cards(bc_file, model_class):
    """
    Writes the IP cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        model_class: The ADH simulation class that holds all simulation information
    """
    ip = model_class.iteration_parameters
    bc_file.write('! Iteration Parameters\n')
    bc_file.write(f'IP NIT {ip.non_linear_iterations}\n')
    index = list(ip.param.non_linear_tolerance_option.get_range()).index(ip.non_linear_tolerance_option)
    if index == 0 or index == 1:
        bc_file.write(f'IP NTL {ip.non_linear_residual_tolerance}\n')
    if index == 0 or index == 2:
        bc_file.write(f'IP ITL {ip.non_linear_incremental_tolerance}\n')
    bc_file.write(f'IP MIT {ip.linear_iterations}\n')

    bc_file.write('\n')  # blank line at the end of the Iteration Parameters


def write_cn_cards(bc_file, model_class):
    """
    Writes the CN cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        model_class: The ADH simulation class that holds all simulation information
    """
    cn = model_class.constituent_properties
    sed_cn = model_class.sediment_constituent_properties
    bc_file.write('! Constituent Properties\n')
    if not cn.general_constituents.empty:
        # bc_file.write(cn.general_constituents.to_csv(sep=' ', index=False, header=False).replace('\r\n', '\n'))
        for _, row in model_class.constituent_properties.general_constituents.iterrows():
            gc_line = f"CN CON {int(row['ID'])} {row['CONC']}"
            if row['NAME']:
                gc_line += f" ! Name: {row['NAME']}"
            bc_file.write(f"{gc_line}\n")
    if not sed_cn.sand.empty:
        for _, row in sed_cn.sand.iterrows():
            sand_line = (
                f"CN SND {int(row['ID'])} {row['CONCENTRATION']} {row['GRAIN_DIAMETER']} "
                f"{row['SPECIFIC_GRAVITY']} {row['POROSITY']}"
            )
            if row['NAME']:
                sand_line += f" ! Name: {row['NAME']}"
            bc_file.write(f"{sand_line}\n")
    if not sed_cn.clay.empty:
        for _, row in sed_cn.clay.iterrows():
            clay_line = (
                f"CN CLA {int(row['ID'])} {row['CONCENTRATION']} {row['GRAIN_DIAMETER']} "
                f"{row['SPECIFIC_GRAVITY']} {row['POROSITY']} {row['CRITICAL_SHEAR_EROSION']} "
                f"{row['EROSION_RATE']} {row['CRITICAL_SHEAR_DEPOSITION']} {row['FREE_SETTLING_VELOCITY']}")
            if row['NAME']:
                clay_line += f" ! Name: {row['NAME']}"
            bc_file.write(f"{clay_line}\n")
    if cn.salinity:
        bc_file.write(f'CN SAL {cn.salinity_id} {cn.reference_concentration}\n')
    if cn.temperature:
        bc_file.write(f'CN TMP {cn.temperature_id} {cn.reference_temperature}\n')
    if cn.vorticity:
        bc_file.write(f'CN VOR {cn.vorticity_id} {cn.vorticity_normalization} {cn.vorticity_as_term} '
                      f'{cn.vorticity_ds_term}\n')

    bc_file.write('\n')  # blank line at the end of the Constituent Properties


def write_sp_cards(bc_file, model_class):
    """
    Writes the SP cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        model_class: The ADH simulation class that holds simulation information
    """
    sed_cn = model_class.sediment_constituent_properties
    if sed_cn.sand.empty and sed_cn.clay.empty:
        return
    sed = model_class.sediment_properties
    bc_file.write('! Sediment Process\n')

    # Currently, we only right the SP CSV card if 'Hwang and Mehta'.
    if sed.cohesive_settling == 1:
        bc_file.write(f'SP CSV 1 {sed.cohesive_settling_a} {sed.cohesive_settling_b} '
                      f'{sed.cohesive_settling_m} {sed.cohesive_settling_n}\n')

    if sed.wind_wave_stress != 0:
        bc_file.write(f'SP WWS {sed.wind_wave_stress}\n')

    if sed.suspended_entrainment != -1:
        bc_file.write(f'SP NSE {sed.suspended_entrainment}\n')

    if sed.bedload_entrainment == 3:
        bc_file.write(f'SP NBE {sed.bedload_entrainment} {sed.critical_shear_sand} {sed.critical_shear_clay}\n')
    elif sed.bedload_entrainment != -1:
        bc_file.write(f'SP NBE {sed.bedload_entrainment}\n')

    # Always write SP HID.
    if sed.hiding_factor == 3:
        bc_file.write(f'SP HID {sed.hiding_factor} {sed.hiding_factor_exponent}\n')
    else:
        bc_file.write(f'SP HID {sed.hiding_factor}\n')

    if sed.use_infiltration_factor:
        bc_file.write(f'SP SIF {sed.infiltration_factor}\n')
    bc_file.write('\n')


def write_mp_cards(bc_file: TextIO, bc_class: 'BoundaryConditions', model_class: ModelControl) -> None:
    """
    Writes the MP cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        bc_class: The ADH simulation boundary condition class that holds simulation information
        model_class: The ADH simulation class that holds simulation information
    """
    sed_cn = model_class.sediment_constituent_properties
    has_sed = not sed_cn.sand.empty or not sed_cn.clay.empty
    num_sed_constituents = len(sed_cn.sand.index) + len(sed_cn.clay.index)
    sed = model_class.sediment_properties
    mc = model_class.model_constants
    bc_file.write('! Global Material Properties\n')
    bc_file.write(f'MP MU {mc.kinematic_viscosity}\n')
    bc_file.write(f'MP G {mc.gravity}\n')
    bc_file.write(f'MP RHO {mc.density}\n')
    if mc.enable_wet_dry_stabilization:
        bc_file.write(f'MP DTL {mc.wet_dry_stabilization_length}\n')
    bc_file.write(f'MP MUC {mc.mannings_unit_constant}\n')

    if has_sed:
        bc_file.write(f'MP NBL {sed.number_bed_layers} {sed.bed_layer_thickness_protocol}\n')
        num_fractions = num_sed_constituents * sed.number_bed_layers
        if num_fractions > 0:
            try:
                for idx, row in sed.global_bed_layers.iterrows():
                    # Print the beginning of each line: 'CARD', 'BED_LAYER_ID', and 'THICKNESS'
                    bc_file.write(f"{row['CARD']} {row['BED_LAYER_ID']} {row['THICKNESS']}")

                    # Get grain fractions where both MATERIAL_ID and BED_LAYER_ID match
                    # Global is MATERIAL_ID 0
                    matching_grain_fractions = sed.bed_layer_grain_fractions[
                        (sed.bed_layer_grain_fractions['MATERIAL_ID'] == 0) &
                        (sed.bed_layer_grain_fractions['BED_LAYER_ID'] == row['BED_LAYER_ID'])
                        ]

                    for _, grain_row in matching_grain_fractions.iterrows():
                        grain_fraction = grain_row.iloc[3]
                        # Append the grain fraction to the line
                        bc_file.write(f" {grain_fraction}")
                    # Add a new line at the end of the card
                    bc_file.write("\n")

                if not sed.global_cohesive_bed.empty:
                    bc_file.write(
                        sed.global_cohesive_bed.to_csv(sep=' ', index=False, header=False).replace('\r\n', '\n'))
            except Exception:
                raise IOError('Error writing bed layers card.')
        if not sed.global_consolidation.empty:
            bc_file.write(f'MP NCP {sed.number_consolidation_times}\n')
            bc_file.write(sed.global_consolidation.to_csv(sep=' ', index=False, header=False, ).replace('\r\n', '\n'))
    bc_file.write('\n')

    bc_file.write('! Material Properties\n')
    for mat_id, mat_prop in bc_class.material_properties.items():
        if mat_prop.eddy_viscosity_method == 'Constant (EVS)':
            evs_line = f'MP EVS {mat_id} {mat_prop.constant_eddy_viscosity}'
            if mat_prop.material_name != "":
                evs_line += f' ! Name: {mat_prop.material_name}'
            bc_file.write(f'{evs_line}\n')

        elif mat_prop.eddy_viscosity_method == 'Estimated (EEV)':
            eev_line = (
                f'MP EEV {mat_id} {mat_prop.estimated_eddy_viscosity_method} '
                f'{mat_prop.estimated_eddy_viscosity_weighting_factor} '
                f'{mat_prop.estimated_eddy_viscosity_minimum}')
            if mat_prop.material_name != "":
                eev_line += f' ! Name: {mat_prop.material_name}'
            bc_file.write(f'{eev_line}\n')

        if mat_prop.coriolis:
            bc_file.write(f'MP COR {mat_id} {mat_prop.coriolis_latitude}\n')
        bc_file.write(f'MP SRT {mat_id} {mat_prop.refinement_tolerance}\n')
        bc_file.write(f'MP ML {mat_id} {mat_prop.max_refinement_level}\n')
        if model_class.operation_parameters.transport != 0:
            for id1, tran_prop in mat_prop.transport_properties.items():
                bc_file.write(f'MP TRT {mat_id} {id1} {tran_prop.refinement_tolerance}\n')
                if (tran_prop.refinement_tolerance > 0 and mat_prop.eddy_viscosity_method == 'Constant (EVS)'
                        and tran_prop.use_diffusion_coefficient):
                    bc_file.write(f'MP DF {mat_id} {id1} {tran_prop.turbulent_diffusion_rate}\n')
        if mat_prop.wind_properties:
            "wnd = mat_prop.wind_properties"
            wnd = mat_prop
            bc_file.write(f'MP WND STR {mat_id} {wnd.stress_formulation}\n')
            bc_file.write(f'MP WND ATT {mat_id} {wnd.attenuation}\n')
        if has_sed:
            displacement_off = sed.material_displacement_off.loc[sed.material_displacement_off['MATERIAL_ID'] == mat_id]
            if not displacement_off.empty:
                bc_file.write(f'MP NDM {mat_id} 0\n')
                # TODO The zero should be updated to type of NDM implementation
                #  0 = bed sediment mass is preserved, bed surface elevation is fixed, solid bottom changes elevation
                #  1 = bed sediment mass is not preserved, bed surface elevation is fixed, solid bottom elevation is
                #      fixed
            local_scour = sed.material_local_scour.loc[sed.material_local_scour['MATERIAL_ID'] == mat_id]
            if not local_scour.empty:
                bc_file.write(f'MP LSM {mat_id}\n')
            diffusion = sed.material_diffusion.loc[sed.material_diffusion['MATERIAL_ID'] == mat_id]
            if not diffusion.empty:
                bc_file.write(f"MP BLD {mat_id} {diffusion['DIFFUSION'][0]}\n")
            try:
                if mat_id in sed.material_bed_layers['MATERIAL_ID'].values:
                        bed_layers = sed.material_bed_layers.loc[sed.material_bed_layers['MATERIAL_ID'] == mat_id]
                        for _, row in bed_layers.iterrows():
                            bed_layer_id = int(row['BED_LAYER_ID'])
                            bc_file.write(f"{row['CARD']} {bed_layer_id} {mat_id} {row['THICKNESS']}")
                            matching_grain_fractions = sed.bed_layer_grain_fractions[
                                (sed.bed_layer_grain_fractions['MATERIAL_ID'] == mat_id) &
                                (sed.bed_layer_grain_fractions['BED_LAYER_ID'] == bed_layer_id)
                                ]
                            for _, row in matching_grain_fractions.iterrows():
                                bc_file.write(f" {row['FRACTION']}")
                            bc_file.write('\n')
                cohesive = sed.material_cohesive_bed.loc[sed.material_cohesive_bed['MATERIAL_ID'] == mat_id]
                if not cohesive.empty:
                    bc_file.write(cohesive.to_csv(sep=' ', index=False, header=False, ).replace('\r\n', '\n'))
            except Exception:
                raise IOError('Error writing bed layers card.')
            consolidation = sed.material_consolidation.loc[sed.material_consolidation['MATERIAL_ID'] == mat_id]
            if not consolidation.empty:
                bc_file.write(consolidation.to_csv(sep=' ', index=False, header=False, ).replace('\r\n', '\n'))

    con_ids = []
    con_ids.extend(_extract_constituent_ids(sed_cn.sand))
    con_ids.extend(_extract_constituent_ids(sed_cn.clay))

    for mat_id, mat_sed in bc_class.mat_id_to_mat_sed.items():
        _, old_sed_id = mat_sed
        for index, con_id in enumerate(con_ids):
            sed_mat_props = bc_class.sediment_material_properties
            if old_sed_id not in sed_mat_props or len(sed_mat_props[old_sed_id]) < index + 1:
                # default for missing data
                tolerance = 1.0
                diffusion = 0.0
            else:
                tolerance = bc_class.sediment_material_properties[old_sed_id][index].refinement_tolerance
                diffusion = sed_mat_props[old_sed_id][index].turbulent_diffusion_rate
            bc_file.write(f'MP TRT {int(mat_id)} {con_id} {tolerance}\n')
            bc_file.write(f'MP DF {mat_id} {con_id} {diffusion}\n')
    bc_file.write('\n')  # blank line at the end of the Material Properties


def write_boundary_string_cards(bc_file, bc_class):
    """
    Writes the NDS, EGS, MDS, MTS cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        bc_class: The ADH simulation class that holds all simulation information
    """

    def _swap_id_and_id_0(row):
        """Swap 'ID' and 'ID0' for 'MTS', NDS, 'EGS', and 'MDS' cards."""
        if row['CARD'] == 'MTS' or row['CARD'] == 'NDS' or row['CARD'] == 'EGS' or row['CARD'] == 'MDS':
            return row['ID_0'], row['ID']
        return row['ID'], row['ID_0']

    def _swap_id_0_and_id_1(row):
        """Swap 'ID_0' and 'ID1' for 'EGS' and 'MDS' cards."""
        if row['CARD'] == 'EGS' or row['CARD'] == 'MDS':
            return row['ID_1'], row['ID_0']
        return row['ID_0'], row['ID_1']

    bs = copy.deepcopy(bc_class.boundary_strings)
    if not bs.empty:
        bc_file.write('! Boundary Strings\n')
        # Swap the ids so the string id is at the end.
        # We keep the string id in the front for ease of querying. However, AdH always has the string ids at the end.
        bs[['ID', 'ID_0']] = bs.apply(_swap_id_and_id_0, axis=1, result_type='expand')
        bs[['ID_0', 'ID_1']] = bs.apply(_swap_id_0_and_id_1, axis=1, result_type='expand')
        write_str = bs.to_csv(sep=' ', na_rep='', index=False, header=False, )
        bc_file.write(re.sub(r'\s*\r?\n', '\n', write_str))
        bc_file.write('\n')  # blank line after Boundary Strings


def write_time_series(bc_file, ts, series_id, is_46):
    """
    Writes a time series to the *.bc file.

    Args:
        bc_file: the *.bc file
        ts: TimeSeries class that is written to the file
        series_id: ID of the time series
        is_46 (bool): True if writing the series as an AdH 4.6 compatible card.
    """
    # construct the card line with series type, series number, and number of entries
    _series_type = ts.series_type
    if is_46:
        if _series_type == 'SERIES WIND':
            _series_type = 'XY2'
        elif _series_type == 'SERIES AWRITE':
            _series_type = 'OS'
        else:
            _series_type = 'XY1'
    line = f'{_series_type} {series_id} {len(ts.time_series.index)}'
    # for output series, add the output units
    if ts.series_type == 'SERIES AWRITE':
        if is_46:
            line = f'{line} {ts.output_units}'
        else:
            line = f'{line} {ts.output_units} {0}'  # todo this looks wrong
    # for wind and wave series, add the location and input/output units
    elif ts.series_type == 'SERIES WIND' or ts.series_type == 'SERIES WAVE':
        line = f'{line} {ts.units} {ts.output_units}'
        bc_file.write(f'XYC {series_id} {ts.x_location} {ts.y_location} \n')
    else:
        # for all other series types, add input/output units
        line = f'{line} {ts.units} {ts.output_units}'
    # write the constructed line
    bc_file.write(f'{line} {ts.xy_name}\n')
    bc_file.write(ts.time_series.to_csv(sep=' ', index=False, header=False, ).replace('\r\n', '\n'))
    bc_file.write('\n')


def write_time_series_cards(bc_file, bc_class, is_46=False):
    """
    Writes the time series cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        bc_class: The ADH simulation class that holds all simulation information
        is_46 (bool): True if writing the series as an AdH 4.6 compatible card.
    """
    # add header for the time series section
    bc_file.write('! Time Series\n')
    awrite_key = -1
    dt_key = -1

    for key, ts in bc_class.time_series.items():
        if ts.series_type == 'SERIES AWRITE':
            # store the output series number
            awrite_key = key
        elif ts.series_type == 'SERIES DT':
            # store the timestep series number
            dt_key = key
        else:
            # write all other series
            write_time_series(bc_file, ts, key, is_46)
    bc_file.write('\n')  # blank line after Time Series

    # write the time step series
    if dt_key != -1:
        # write header for time step series section
        bc_file.write('! Time step time series\n')
        write_time_series(bc_file, bc_class.time_series[dt_key], dt_key, is_46)
        bc_file.write('\n')  # blank line after Time step time series

    # write the output series
    if awrite_key != -1:
        # write header for time step series
        bc_file.write('! Output series\n')
        write_time_series(bc_file, bc_class.time_series[awrite_key], awrite_key, is_46)
        bc_file.write('\n')  # blank line after Output series


def write_fr_cards(bc_file, bc_class):
    """
    Writes the FR cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        bc_class: The ADH simulation class that holds all simulation information
    """
    fr = bc_class.friction_controls
    if not fr.empty:
        bc_file.write('! Friction Controls\n')
        write_str = fr.to_csv(sep=' ', na_rep='', index=False, header=False, )
        bc_file.write(re.sub(r'\s*\r?\n', '\n', write_str))
        bc_file.write('\n')  # blank line after Friction Controls


def write_sdv_cards(bc_file, model_class):
    """
    Writes the SDV cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        model_class: The ADH simulation class that holds all simulation information
    """
    sdv = model_class.sediment_properties.sediment_diversion
    if not sdv.empty:
        bc_file.write('! Sediment Diversions\n')
        bc_file.write(sdv.to_csv(sep=' ', na_rep='', index=False, header=False).replace('\r\n', '\n'))
        bc_file.write('\n')


def write_pc_cards(bc_file, model_class, is_46):
    """
    Writes the PC cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        model_class: The ADH simulation class that holds all simulation information
        is_46 (bool): True if writing the cards as AdH 4.6 compatible cards.
    """
    bc_file.write('! Output Control\n')
    oc = model_class.output_control
    # OC card
    objects = list(oc.param.output_control_option.get_range())
    if oc.output_control_option == objects[0]:
        bc_file.write(f'OC {oc.oc_time_series_id}\n')
    # FLX cards
    df = oc.output_flow_strings
    if not df.empty:
        bc_file.write(df.to_csv(sep=' ', na_rep='', index=False, header=False).replace('\r\n', '\n'))
    # PRN cards
    df = oc.nodal_output
    if not df.empty:
        df = df.drop('POINT_ID', axis=1)
        bc_file.write(df.to_csv(sep=' ', na_rep='', index=False, header=False).replace('\r\n', '\n'))
    # Other cards
    if oc.print_adaptive_mesh:
        bc_file.write('PC ADP\n')
    if oc.print_numerical_fish_surrogate:
        bc_file.write('PC ELM\n')
    if oc.screen_output_residual:
        if is_46:
            bc_file.write('PC LVL 0\n')
        else:
            bc_file.write('SOUT RESID\n')
    if oc.screen_output_all:
        if is_46:
            bc_file.write('PC LVL 1\n')
        else:
            bc_file.write('SOUT ALL\n')
    if oc.screen_output_mass_error:
        if is_46:
            bc_file.write('PC MEO 1\n')
        else:
            bc_file.write('SOUT MERROR\n')
    if oc.screen_output_worst_nonlinear_node:
        bc_file.write('SOUT NLNODE\n')
    if oc.screen_output_worst_linear_node:
        bc_file.write('SOUT LNODE\n')
    if oc.file_output_wind:
        bc_file.write('FOUT WIND\n')
    if oc.file_output_wave:
        bc_file.write('FOUT WAVE\n')
    if oc.file_output_adapted_grid:
        bc_file.write('FOUT ADAPT GRID\n')
    if oc.file_output_adapted_solution:
        bc_file.write('FOUT ADAPT SW\n')
    if oc.file_output_adapted_transport:
        bc_file.write('FOUT ADAPT CON\n')
    if oc.file_output_sediment:
        bc_file.write('FOUT SED\n')

    bc_file.write('\n')  # blank line after Output Control


def write_solution_control_cards(bc_file, bc_class, model_class):
    """
    Writes the NB, DB, WER, FLP, SLUICE cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        bc_class: The ADH simulation boundary conditions class that holds simulation information
        model_class: The ADH simulation class that holds simulation information
    """
    bc_file.write('! Solution Controls\n')
    sc = bc_class.solution_controls
    if not sc.empty:
        write_str = sc.to_csv(sep=' ', na_rep='', index=False, header=False)
        bc_file.write(re.sub(r'\s*\r?\n', '\n', write_str))

    nb_sdr = bc_class.stage_discharge_boundary
    if not nb_sdr.empty:
        bc_file.write(nb_sdr.to_csv(sep=' ', na_rep='', index=False, header=False, ).replace('\r\n', '\n'))

    if model_class.constituent_properties.temperature:
        cp = model_class.constituent_properties
        bc_file.write(f'DB RAD {cp.short_wave_radiation_series} {cp.dew_point_temperature_series}\n')

    write_breach_control_cards(bc_file, bc_class)

    write_structure_cards(bc_file, bc_class)

    bc_file.write('\n')  # blank line after Solution Controls


def write_structure_cards(bc_file, bc_class):
    """Writes out cards for weirs, flap gates, and sluice gates.

    Args:
        bc_file: the *.bc file
        bc_class: The ADH simulation class that holds all simulation information
    """
    wer = bc_class.weirs
    if not wer.empty:
        bc_file.write(f'WER {len(wer.index)}\n')
        bc_file.write(
            wer.to_csv(sep=' ', na_rep='', index=False, header=False, ).replace('\r\n', '\n').replace('"', ''))
    flp = bc_class.flap_gates
    if not flp.empty:
        bc_file.write(f'FLP {len(flp.index)}\n')
        bc_file.write(
            flp.to_csv(sep=' ', na_rep='', index=False, header=False, ).replace('\r\n', '\n').replace('"', ''))
    sls = bc_class.sluice_gates
    if not sls.empty:
        bc_file.write(f'SLUICE {len(sls.index)}\n')
        bc_file.write(
            sls.to_csv(sep=' ', na_rep='', index=False, header=False, ).replace('\r\n', '\n').replace('"', ''))


def write_breach_control_cards(bc_file, bc_class):
    """
    Writes the BR cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        bc_class: The ADH simulation class that holds all simulation information
    """
    br = bc_class.breach_controls
    if br.empty:
        return
    format_dict = {'JAI': [str, str, int, int, float, float, float, float, int, int],  # noqa E241
                   'SAS': [str, str, int, int, float, float, float, float, int, int],  # noqa E241
                   'MLM': [str, str, float, float, float, float],  # noqa E241
                   'FRO': [str, str, int, int, float, float, float, float, float, int, int],  # noqa E241
                   'BRC': [str, str, int, int, float, float, float, float, int, int],  # noqa E241
                   'VTG': [str, str, int, int, float, float, float, int, float, int, int],  # noqa E241
                   'FER': [str, str, int, int, float, float, float, int, float, int, int],  # noqa E241
                   'USR': [str, str, int, int]  # noqa E241
                   }
    try:
        col_list = ['CARD', 'CARD_1', 'C_0', 'C_1', 'C_2', 'C_3', 'C_4', 'C_5', 'C_6', 'C_7', 'C_8']
        for _, row in br.iterrows():
            card = row['CARD_1']
            format_list = format_dict[card]
            for index, item in enumerate(format_list):
                record = row[col_list[index]]
                mystr = f'{item(record)} '
                bc_file.write(mystr)
            bc_file.write('\n')

    except Exception:
        raise IOError('Error writing BR card.')


def write_tc_cards(bc_file, model_class, is_46):
    """
    Writes the TC cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        model_class: The ADH simulation class that holds all simulation information
        is_46 (bool): True if writing the cards as AdH 4.6 compatible cards.
    """
    bc_file.write('! Time Controls\n')
    tc = model_class.time_control
    objects = list(tc.param.start_time_units.get_range())
    bc_file.write(f'TC T0 {tc.start_time} {objects.index(tc.start_time_units)}\n')
    objects = list(tc.param.end_time_units.get_range())
    bc_file.write(f'TC TF {tc.end_time} {objects.index(tc.end_time_units)}\n')
    objects = list(tc.param.time_step_option.get_range())
    if tc.time_step_option == objects[0]:
        bc_file.write(f'TC STD {tc.steady_state_min_time_step_size} {tc.steady_state_max_time_step_size}\n')
    elif tc.time_step_option == objects[2]:
        bc_file.write(f'TC ATF {tc.auto_time_step_find_min_time_step_size} '
                      f'{tc.auto_time_step_find_max_time_step_size_series}\n')
    # TC IDT replaced by SERIES DT in AdH 5.0
    elif is_46:
        bc_file.write(f'TC IDT {tc.max_time_step_size_time_series}\n')

    bc_file.write('\n')  # blank line after Time Controls


def write_advanced_cards(bc_file, model_class):
    """
    Writes the advanced cards to the *.bc file.

    Args:
        bc_file: the *.bc file
        model_class: The ADH simulation class that holds all simulation information
    """
    advanced_cards = model_class.advanced_cards.advanced_cards
    if advanced_cards.empty:
        return
    bc_file.write('! Advanced\n')
    try:
        for _, row in advanced_cards.iterrows():
            card = row['CARD_LINE']
            bc_file.write(f'{card}\n')

    except Exception:
        raise IOError('Error writing advanced card.')

    bc_file.write('\n')  # blank line after Advanced cards


def read_bt_file(file_name: str, vessels: VesselList, logger: logging.Logger = None):
    """
    Reads the *.bt file and fills the vessel section of the Vessels class.

    Args:
        file_name (str): File name of the *.bt file
        vessels (VesselList): Vessels class instance
        logger (Logger, optional): Logger instance
    """
    propeller_type_map = {'1': 'Open wheel', '2': 'Kort nozzle'}

    if logger is not None:
        logger.info('Reading vessel file.')

    with open(file_name, "r") as file:
        vessels_dict: dict[int, Vessel] = defaultdict(lambda: Vessel())
        segment_dict: dict[int, list] = defaultdict(lambda: [])
        for line_number, line in enumerate(file):
            try:
                # Remove the new line character and split the line
                line = line.strip()
                line_split = line.split()

                # Skip blank lines
                if not line_split:
                    continue

                # Parse vessel data
                keyword = line_split[0]

                if keyword == "BOAT":
                    vessels.use_velocity = bool(True if int(line_split[2]) == 0 else False)

                elif keyword == "FDEF":
                    vessel_id = int(line_split[1])
                    vessel = vessels_dict[vessel_id]
                    vessel.initial_x_coordinate = float(line_split[3])
                    vessel.initial_y_coordinate = float(line_split[4])
                    vessel.initial_speed = float(line_split[5])

                    # Store FDEF coordinates for first SDEF validation
                    vessel._fdef_coords = (vessel.initial_x_coordinate, vessel.initial_y_coordinate)
                    vessel._first_sdef = True
                    vessel._skip_first_sdef = False

                elif keyword == "DRFT":
                    vessel_id = int(line_split[1])
                    vessel = vessels_dict[vessel_id]
                    vessel.vessel_draft = float(line_split[2])

                elif keyword == "BLEN":
                    vessel_id = int(line_split[1])
                    vessel = vessels_dict[vessel_id]
                    vessel.vessel_length = float(line_split[2])

                elif keyword == "BWID":
                    vessel_id = int(line_split[1])
                    vessel = vessels_dict[vessel_id]
                    vessel.vessel_width = float(line_split[2])

                elif keyword == "PBOW":
                    vessel_id = int(line_split[1])
                    vessel = vessels_dict[vessel_id]
                    vessel.bow_length_ratio = float(line_split[2])

                elif keyword == "PSTR":
                    vessel_id = int(line_split[1])
                    vessel = vessels_dict[vessel_id]
                    vessel.stern_length_ratio = float(line_split[2])

                elif keyword == "CBOW":
                    vessel_id = int(line_split[1])
                    vessel = vessels_dict[vessel_id]
                    vessel.bow_draft_ratio = float(line_split[2])

                elif keyword == "CSTR":
                    vessel_id = int(line_split[1])
                    vessel = vessels_dict[vessel_id]
                    vessel.stern_draft_ratio = float(line_split[2])

                elif keyword == "PROP":
                    vessel_id = int(line_split[1])
                    vessel = vessels_dict[vessel_id]
                    vessel.propeller = Propeller()
                    vessel.propeller.propeller_type = propeller_type_map[line_split[2]]
                    vessel.propeller.propeller_diameter = float(line_split[3])
                    vessel.propeller.propeller_center_distance = float(line_split[4])
                    vessel.propeller.tow_boat_length = float(line_split[5])
                    vessel.propeller.distance_to_stern = float(line_split[6])

                elif keyword == "SDEF":
                    vessel_id = int(line_split[1])
                    segment_type = int(line_split[3])
                    end_x = float(line_split[4])
                    end_y = float(line_split[5])

                    # Check if this is the first SDEF
                    if vessel._first_sdef:
                        vessel._first_sdef = False
                        fdef_x, fdef_y = vessel._fdef_coords
                        if end_x == fdef_x and end_y == fdef_y:
                            vessel._skip_first_sdef = True
                            continue  # Skip first SDEF if first coords match

                    # Adjust segment number if first SDEF was skipped
                    segment_number = int(line_split[2])
                    if vessel._skip_first_sdef:
                        segment_number -= 1

                    segment = {
                        "SEGMENT_NUMBER": segment_number,
                        "SEGMENT_TYPE": segment_type,
                        "END_X_COORDINATE": float(line_split[4]),
                        "END_Y_COORDINATE": float(line_split[5]),
                        "MOTION_VALUE": float(line_split[6]),
                        "ARC_CENTER_X": float(line_split[7]) if int(line_split[3]) == 1 else None,
                        "ARC_CENTER_Y": float(line_split[8]) if int(line_split[3]) == 1 else None,
                        "TURN_DIRECTION": float(line_split[9]) if int(line_split[3]) == 1 else 0
                    }
                    vessel_segments = segment_dict[vessel_id]
                    vessel_segments.append(segment)

                elif keyword == "ENDD":
                    break  # End of the file

            except Exception as e:
                msg = f'Error reading line {line_number + 1} of file: {os.path.basename(file_name)}.\nLine: {line}\nError: {e}'
                if logger is not None:
                    logger.error(msg)
                else:
                    print(msg)
        for vessel_id, vessel in vessels_dict.items():
            if vessel_id in segment_dict:
                vessel.sailing_segments = pd.DataFrame(segment_dict[vessel_id])
                vessel.sailing_segments.astype(SEGMENTS_COLUMN_TYPES)
            vessels.vessels.append(vessel)


def write_bt_file(file_name: str, vessels: VesselList, logger: logging.Logger = None):
    """
    Writes the vessels to a *.bt file.

    Args:
        file_name (str): Output file name
        vessels (VesselList): Vessels class instance containing vessels and segments
        logger (Logger, optional): Logger instance
    """
    turn_direction_map = {'Left': 1.0, 'Right': -1.0}
    propeller_type_map = {'Open wheel': 1, 'Kort nozzle': 2}

    if logger is not None:
        logger.info('Writing vessel file.')

    try:
        with open(file_name, "w") as file:
            # Write "BOAT" keyword and use_velocity flag at the beginning
            num_vessels = len(vessels.vessels)
            file.write(f"BOAT {num_vessels} {0 if vessels.use_velocity else 1}\n")

            for vessel_id, vessel in enumerate(vessels.vessels, start=1):
                num_segments = len(vessel.sailing_segments)
                file.write(f"FDEF {vessel_id} {num_segments} {vessel.initial_x_coordinate} "
                           f"{vessel.initial_y_coordinate} {vessel.initial_speed}\n")
                file.write(f"DRFT {vessel_id} {vessel.vessel_draft}\n")
                file.write(f"BLEN {vessel_id} {vessel.vessel_length}\n")
                file.write(f"BWID {vessel_id} {vessel.vessel_width}\n")
                file.write(f"PBOW {vessel_id} {vessel.bow_length_ratio}\n")
                file.write(f"PSTR {vessel_id} {vessel.stern_length_ratio}\n")
                file.write(f"CBOW {vessel_id} {vessel.bow_draft_ratio}\n")
                file.write(f"CSTR {vessel_id} {vessel.stern_draft_ratio}\n")

                # Write vessel propeller details (PROP)
                if vessel.propeller is not None:
                    file.write(f"PROP {vessel_id} {propeller_type_map[vessel.propeller.propeller_type]} "
                               f"{vessel.propeller.propeller_diameter} {vessel.propeller.propeller_center_distance} "
                               f"{vessel.propeller.tow_boat_length} {vessel.propeller.distance_to_stern}\n")

                # Write all segments (SDEF)
                for index, segment in vessel.sailing_segments.iterrows():
                    segment_card = (
                        f"SDEF {vessel_id} {segment['SEGMENT_NUMBER']} {segment['SEGMENT_TYPE']} "
                        f"{segment['END_X_COORDINATE']} {segment['END_Y_COORDINATE']}"
                        f" {segment['MOTION_VALUE']}"
                    )
                    if segment['SEGMENT_TYPE'] == 1:
                        segment_card += (f" {segment['ARC_CENTER_X']} {segment['ARC_CENTER_Y']} "
                                         f"{turn_direction_map[segment['TURN_DIRECTION']]}")
                    file.write(segment_card + "\n")

            # Write ENDD to indicate the end of the file
            file.write("ENDD 0 0\n")

    except Exception as e:
        msg = f'Error writing to file: {file_name}.\nError: {e}'
        if logger is not None:
            logger.error(msg)
        else:
            print(msg)
