# -*- coding: utf-8 -*-
import param


class MaterialProperties(param.Parameterized):
    material_name = param.String(
        default='',
        doc='Optional material name.',
        precedence=0,
    )
    eddy_viscosity_method = param.ObjectSelector(
        default='Estimated (EEV)',
        objects=['Constant (EVS)', 'Estimated (EEV)'],
        doc='MP EVS or MP EEV: Method of defining eddy viscosity.',
        precedence=1,
    )
    constant_eddy_viscosity = param.Number(
        default=0.0,
        bounds=(0, None),
        doc='MP EVS: isotropic kinematic eddy viscosity parameter.',
        precedence=2,
    )
    # vxy_eddy_viscosity = param.Number(
    #     default=0.0,
    #     bounds=(0, None),
    #     doc='MP EVS: Vxy term of the kinematic eddy viscosity tensor.',
    #     precedence=3,
    # )
    # vyy_eddy_viscosity = param.Number(
    #     default=0.0,
    #     bounds=(0, None),
    #     doc='MP EVS: Vyy term of the kinematic eddy viscosity tensor.',
    #     precedence=4,
    # )
    estimated_eddy_viscosity_method = param.ObjectSelector(
        default=1,
        objects={'1 - Isotropic': 1, '2 - Anisotropic': 2, '3 - Smagorinsky': 3, '4 - Stansby': 4},
        doc='MP EEV: Method of defining estimated eddy viscosity.',
        precedence=3,
    )
    estimated_eddy_viscosity_weighting_factor = param.Number(
        default=1.0,
        bounds=(0, None),
        softbounds=(0.1, 5.0),
        doc='MP EEV: Weighting factor, method dependent.',
        precedence=4,
    )
    estimated_eddy_viscosity_minimum = param.Number(
        default=0.0,
        bounds=(0, None),
        softbounds=(0.0, 1.0),
        doc='MP EEV: Weighting factor, method dependent.',
        precedence=5,
    )
    refinement_tolerance = param.Number(
        default=1.0,
        bounds=(0, None),
        doc='MP SRT: Mesh refinement tolerance. REQUIRED.',
        precedence=6,
    )
    max_refinement_level = param.Integer(
        default=0,
        bounds=(0, None),
        doc='MP ML: Maximum mesh refinement level. REQUIRED.',
        precedence=7,
    )
    wind_properties = param.Boolean(
        default=False,
        doc='MP WND: Wind parameters active.',
        precedence=8,
    )
    stress_formulation = param.ObjectSelector(
        default=0,
        objects={'0 - no transform': 0, '1 - Wu': 1, '2 - Teeter': 2},
        doc='MP WND STR: Wind stress formulation.',
        precedence=9,
    )
    attenuation = param.Number(
        default=1.0,
        bounds=(0, None),
        doc='MP WND ATT: Wind attenuation factor applied to wind speeds or shears.',
        precedence=10,
    )
    coriolis = param.Boolean(
        default=False,
        doc='MP COR: Coriolis parameter active.',
        precedence=11,
    )
    coriolis_latitude = param.Number(
        default=0.0,
        bounds=(-90,90),
        doc='MP COR: Latitude decimal degrees.',
        precedence=12,
    )


    # TODO TVS, DPL, DPT, RD, TOR - need documentation

    def __init__(self):
        super(MaterialProperties, self).__init__()
        #self.wind_properties = MaterialWindProperties()
        self.transport_properties = {}
        self.sediment_transport_properties = {}

        self._update_eddy_viscosity_method()
        self._update_wind_properties()
        self._update_coriolis()

    @param.depends('eddy_viscosity_method', watch=True)
    def _update_eddy_viscosity_method(self):
        self.param.constant_eddy_viscosity.precedence = -1
        # self.param.vxy_eddy_viscosity.precedence = -1
        # self.param.vyy_eddy_viscosity.precedence = -1
        self.param.estimated_eddy_viscosity_method.precedence = -1
        self.param.estimated_eddy_viscosity_weighting_factor.precedence = -1
        self.param.estimated_eddy_viscosity_minimum.precedence = -1
        objects = list(self.param.eddy_viscosity_method.get_range())
        if self.eddy_viscosity_method == objects[0]:
            self.param.constant_eddy_viscosity.precedence = 2
            # self.param.vxy_eddy_viscosity.precedence = 3
            # self.param.vyy_eddy_viscosity.precedence = 4
        else:
            self.param.estimated_eddy_viscosity_method.precedence = 3
            self.param.estimated_eddy_viscosity_weighting_factor.precedence = 4
            self.param.estimated_eddy_viscosity_minimum.precedence = 5

    @param.depends('wind_properties', watch=True)
    def _update_wind_properties(self):
        self.param.stress_formulation.precedence = -1
        self.param.attenuation.precedence = -1
        if self.wind_properties:
            self.param.stress_formulation.precedence = 9
            self.param.attenuation.precedence = 10

    def set_not_required(self, value=False):
        self.coriolis = value
        self.wind_properties = value

    @param.depends('coriolis', watch=True)
    def _update_coriolis(self):
        self.param.coriolis_latitude.precedence = -1
        if self.coriolis:
            self.param.coriolis_latitude.precedence = 12
