"""ConstantDefinitions Class."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import sys

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.core_data.calculator.setting_group import SettingGroup
from xms.FhwaVariable.core_data.variables.variable import Variable


# @singleton
class ConstantDefinitions(SettingGroup):
    """Provides definitions for the constants used in hydraulic computations."""

    def __init__(self, name: str = 'HydraulicToolbox', version: str = '0.1', agency: str = 'FHWA',
                 developed_by: str = 'Aquaveo'):
        """Initializes the constant values."""
        super().__init__(name, version, agency, developed_by)

        self.name = 'ConstantDefinitions'
        self.type = 'ConstantDefinitions'

        # TODO: Implement Salt/Freshwater options and buttons to compute the constants
        salinity_options = ['custom', 'Freshwater (0 g/kg)', 'Atlantic Ocean (35.4 g/kg)', 'Pacific Ocean (33 g/kg)',
                            'Indian Ocean (35 g/kg)', 'Gulf of Mexico (35 g/kg)', 'Mediterranean Sea (38 g/kg)',
                            'Dead Sea 337 (g/kg)', 'Great Salt Lake (North Arm) (317 g/kg)',
                            'Great Salt Lake (South Arm) (142 g/kg),']

        self.input['Null data'] = Variable('Value that indicates null data', 'float', -9999.0, [], complexity=2,
                                           limits=(-sys.float_info.max, sys.float_info.max))
        self.input['Zero tolerance'] = Variable('Tolerance for numbers to be considered zero', 'float', 0.00001, [],
                                                complexity=2, precision=8)

        self.input['Elevation'] = Variable('Elevation for Settings', 'float', 0.0, [], precision=2,
                                           unit_type=['length'], native_unit='ft', us_units=self.us_mid_length,
                                           si_units=self.si_mid_length, complexity=2)
        # Compute g for elevation
        # Reset g to sea-level default
        self.input['Gravity'] = Variable('Gravity (g)', 'float', 32.17405, [], native_unit='ft/s^2',
                                         us_units=[["ft/s^2"]], si_units=[["m/s^2"]], precision=6,
                                         complexity=2)
        self.input['Pi'] = Variable('Pi (π)', 'float', 3.1415926535, [], precision=10, complexity=2)

        self.input['Manning constant'] = Variable('Manning unit constant (k)', 'float', 1.485977123, [], precision=4,
                                                  complexity=1, native_unit='ft^(1/3)/s', us_units=[["ft^(1/3)/s"]],
                                                  si_units=[["m^(1/3)/s"]])
        self.input['Unit conversion constant (curbs)'] = Variable('Unit conversion constant for curb and gutter (ku)',
                                                                  'float', 0.56, [], precision=4, complexity=1,
                                                                  native_unit='ft^(1/3)/s', us_units=[["ft^(1/3)/s"]],
                                                                  si_units=[["m^(1/3)/s"]])

        self.input['Stream value'] = Variable('Stream value', 'float', 12.0, precision=2, unit_type=["flow"],
                                              native_unit="cfs", us_units=[["cfs"]], si_units=[["cms"]], complexity=2)

        self.input['Gallon'] = Variable('Use US or British gallon', 'list', 0, ['US gallon', 'British gallon'],
                                        complexity=1)

        self.input['Temperature'] = Variable('Temperature of water', 'float', 60.0, precision=2,
                                             unit_type=["temperature"],
                                             native_unit="°F", us_units=self.us_temperature,
                                             si_units=self.si_temperature, complexity=1)
        # Compute pressure for elevation
        self.input['Ambient pressure'] = Variable('Ambient pressure', 'float', 1.0, precision=4,
                                                  unit_type=["pressure"],
                                                  native_unit='atm', us_units=[['atm', 'psi']],
                                                  si_units=[['bar', 'pa']],
                                                  complexity=2)
        self.input['Water salinity options'] = Variable('Water salinity options', 'list', 0, salinity_options)
        self.input['Water salinity'] = Variable("Water salinity", 'float', 1.0, precision=4,
                                                unit_type=["concentration"],
                                                native_unit='parts per thousand',
                                                us_units=[['parts per thousand', 'ppm']],
                                                si_units=[['parts per thousand', 'ppm']],
                                                complexity=2)
        # compute constants based on temperature, pressure, and salinity: compute_constants_from_constant_data
        # Reset constants to freshwater defaults
        self.input['Water density'] = Variable('Water density (ρw)', 'float', 1.94, precision=4, unit_type=["density"],
                                               native_unit="slugs/ft^3", us_units=[["slugs/ft^3"]],
                                               si_units=[["kg/m^3"]], complexity=1)
        self.input['Sediment density'] = Variable('Sediment density (ρs)', 'float', 5.141, precision=4,
                                                  unit_type=["density"], native_unit="slugs/ft^3",
                                                  us_units=[["slugs/ft^3"]],
                                                  si_units=[["kg/m^3"]], complexity=1)
        self.input['Dynamic viscosity of water'] = Variable('Dynamic viscosity of water (η)', 'float', 0.0000234,
                                                            precision=10, unit_type=["viscosity"],
                                                            native_unit='lbf*s/ft^2',
                                                            us_units=[['lbf*s/ft^2']],
                                                            si_units=[['kN*s/m^2', 'N*s/m^2', 'pa*s']],
                                                            complexity=1)
        self.input['Kinematic viscosity of water'] = Variable('Kinematic viscosity of water (ν)', 'float', 0.00001208,
                                                              precision=10, unit_type=["viscosity"],
                                                              native_unit="ft^2/s", us_units=[["ft^2/s"]],
                                                              si_units=[["m^2/s", "mm^2/s"]], complexity=1)
        self.input['Unit weight of water'] = Variable('Unit weight of water (γw)', 'float', 62.4, precision=4,
                                                      unit_type=["specific_weight"], native_unit="lb/ft^3",
                                                      us_units=[["lb/ft^3"]], si_units=[["kN/m^3", "N/m^3"]])
        self.input['Unit weight of soil'] = Variable('Unit weight of soil (γs)', 'float', 165.0, precision=4,
                                                     unit_type=["specific_weight"], native_unit="lb/ft^3",
                                                     us_units=[["lb/ft^3"]], si_units=[["kN/m^3", "N/m^3"]])

        # self.input['Minimum size of gravel'] = Variable("Minimum size of gravel", 'float', 3.0, precision=4,
        #                                                  unit_type=["length"], native_unit="mm",
        #                                                  us_units=self.us_short_length, si_units=self.si_short_length)

    # def compute_water_density_based_on_temperature(self, t):
    #     p0 = 999.83311  # kg.m^3
    #     a1 = 0.0752  # kg/(m³·°C)
    #     a2 = -0.0089  # kg/m3 * C)
    #     a3 = 7.36413 * 10 ** -5  # kg/(m³·°C³)
    #     a4 = 4.74639 * 10 ** -7  # kg/(m³·°C4),
    #     a5 = 1.34888 * 10 ** -9  # kg/(m³·°C5),
    #
    #     rho = p0 + a1 * t + a2 * t ** 2 + a3 * t ** 3 + a4 * t ** 4 + a5 * t ** 5

    def _compute_water_bulk_modulus_based_on_temperature_salinity_and_pressure(self, t, t2, t3, t4, s, s15, p):
        """Computes the water Bulk Modulus based on temperature, salinity, and pressure.

        Args:
            t (float): temperature
            t2 (float): temperature^2
            t3 (float): temperature^3
            t4 (float): temperature^4
            s (float): salinity
            s15 (float): salinity^1.5
            p (float): pressure

        Returns:
            k (float): water Bulk Modulus
        """
        # 'A New high pressure equation of state for seawater
        #   by Frank J Millero, Chen-Tung Chen, Alving Bradshaw, and Karl Schleicher, 1979
        a = 54.6746 - 0.603459 * t + 0.0109987 * t2 - 6.167e-05 * t3
        b = 0.07944 + 0.016483 * t - 5.3009e-04 * t2
        c = 0.0022838 - 1.09810e-05 * t - 1.6078e-06 * t2
        d = 0.000191075
        e = -9.9348e-07 + 2.0816e-08 * t + 9.16970e-10 * t2

        aw = 3.239908 + 0.00143713 * t + 0.000116092 * t2 - 5.77905e-07 * t3
        bw = 8.50935e-05 - 6.12293e-06 * t + 5.2787e-08 * t2

        kw = 19652.21 + 148.4206 * t - 2.327105 * t2 + 0.01360477 * t3 - 5.155288e-05 * t4

        new_a = aw + c * s + d * s15
        new_b = bw + e * s

        k0 = kw + a * s + b * s15

        k = k0 + new_a * p + new_b * p ** 2
        return k

    @staticmethod
    def _compute_water_density_based_on_temperature_and_salinity(t, t2, t3, t4, t5, s, s15, s2):
        """Computes the water Bulk Modulus based on temperature, salinity, and pressure.

        Args:
            t (float): temperature
            t2 (float): temperature^2
            t3 (float): temperature^3
            t4 (float): temperature^4
            t5 (float): temperature^5
            s (float): salinity
            s15 (float): salinity^1.5
            s2 (float): salinity^2
        """
        # Method pulled from https://www.translatorscafe.com/unit-converter/en-US/CalcData/salt-water-density/
        # Frank J.Millero, Alain Poisson. International one-atmosphere equation of state of seawater.
        #   Deep Sea Research Part A. Oceanographic Research Papers. Vol 28, Issue 6, June 1981, Pages 625–629
        p0 = 999.84259 + 6.793952e-02 * t - 9.095290e-03 * t2 + 1.001685e-04 * t3 - 1.120083e-6 * t4 + 6.536332e-9 * t5
        asp = 0.824493 - 4.0899e-03 * t + 7.6438e-5 * t2 - 8.2467e-7 * t3 + 5.3875e-9 * t4
        bsp = -5.72466e-3 + 1.0227e-4 * t - 1.6546e-6 * t2
        csp = 4.8314e-4
        rho_with_t_s_no_p = p0 + asp * s + bsp * s15 + csp * s2
        return rho_with_t_s_no_p

    @staticmethod
    def _compute_water_viscosity_based_on_temperature_and_salinity(t, t2, s, s2):
        """Compute the viscosity of water based on temperature and salinity.

        Args:
            t (float): temperature
            t2 (float): temperature^2
            s (float): salinity
            s2 (float): salinity^2
        """
        # Desalination and water treatment
        # by Mostafa H. Sharqawy, John H. Lienhard V, Sed M. Zubair, Nov 2009, December 2009
        # Sharqawy 2010
        # Equation 22 on pdf page 11
        water_visc = 4.2844e-05 + (0.157 * (t + 64.9933) ** 2 - 91.296) ** -1
        # I found the below equation to spectacularly fail with salinity compared to the report's own graph
        # new_a = 1.541 + 1.998e-02 * t - 9.52e-05 * t2
        # new_b = 7.974 - 7.561e-02 * t + 4.724e-04 * t2
        # new_visc = water_visc * (1 + new_a * s + new_b * s2)

        # So instead, I'm creating my function that matches his data
        # Can't use interpolation code because it uses constants
        adjusted_coef = 1.0
        if s > 150:
            s = 150
        if s > 0.0:
            s_value = [0, 40, 80, 120, 150]
            s_coef = [1.0, 1.075, 1.18, 1.3, 1.424]
            interp_coef = 0.0
            for idx, _ in enumerate(s_value):
                if s_value[idx] >= s:
                    interp_coef = (s - s_value[idx - 1]) * (s_coef[idx] - s_coef[idx - 1]) / \
                                  (s_value[idx] - s_value[idx - 1]) + s_coef[idx - 1]
                    break
            if interp_coef > 1.0:
                adjusted_coef = interp_coef + 1e-03 * t - 3e-6 * t2

        new_visc = water_visc * adjusted_coef

        valid = False
        if 0 <= t < 180 and 0 <= s < 150:
            valid = True

        return new_visc, valid

    def compute_constants_from_constant_data(self):
        """Computes the constants from constant data."""
        temperature = self.input['Temperature'].get_val()
        temperature = self.input['Temperature'].get_val()
        salinity = self.input['Water salinity'].get_val()
        pressure = self.input['Ambient pressure'].get_val()
        new_unit_w_water, new_dynamic_viscosity = \
            self.setup_variables_and_conversions_for_constant_computations_based_on_t_s_and_p(temperature,
                                                                                              salinity,
                                                                                              pressure)

        g = self.input['Gravity'].get_val()
        new_unit_w_water, new_dynamic_viscosity = \
            self.setup_variables_and_conversions_for_constant_computations_based_on_t_s_and_p(temperature,
                                                                                              salinity,
                                                                                              pressure)

        g = self.input['Gravity'].get_val()
        # Convert units and set values
        self.input['Temperature'].set_val(temperature)
        self.input['Water density'].set_val(new_unit_w_water / g)
        self.input['Unit weight of water'].set_val(new_unit_w_water)
        self.input['Water density'].set_val(new_unit_w_water / g)
        self.input['Unit weight of water'].set_val(new_unit_w_water)

        # viscosity
        new_kinematic_viscosity = new_dynamic_viscosity / self.input['Water density'].get_val()
        self.input['Dynamic viscosity of water'].set_val(new_dynamic_viscosity)
        new_kinematic_viscosity = new_dynamic_viscosity / self.input['Water density'].get_val()
        self.input['Dynamic viscosity of water'].set_val(new_dynamic_viscosity)
        self.input['Kinematic viscosity of water'].set_val(new_kinematic_viscosity)

    def setup_variables_and_conversions_for_constant_computations_based_on_t_s_and_p(self, temperature, salinity,
                                                                                     pressure):
        """Computes the constants based on variables for temperature, salinity, and pressure.

        Args:
            temperature (float): temperature
            salinity (float): salinity
            pressure (float): pressure
        """
        # Atlantic Ocean = 35.4 g/kg
        # Pacific Ocean = 34 to 32
        # Indian Ocean = 35
        # Gulf of Mexico 34 to 36
        # Mediterranean Sea 38
        # Dead Sea 337
        # Great Salt Lake 317 (North Arm), 142 (South Arm)
        # See wikipedia list of bodies of water by salinity

        # Convert units to metric
        # Can't use conversion code because of circular dependencies
        t = (temperature - 32) * 0.5556  # F to C
        s = salinity
        p = pressure * 1.013  # 'atm' to 'bars'

        # compute t and s values
        t2 = t ** 2
        t3 = t ** 3
        t4 = t ** 4
        t5 = t ** 5
        s15 = s ** 1.5
        s2 = s ** 2

        new_unit_w_water, new_dynamic_viscosity = self._compute_constants_based_on_temperature_salinity_and_pressure(
            t, t2, t3, t4, t5, s, s15, s2, p)

        # Convert units
        new_unit_w_water_english = new_unit_w_water * 0.0624279606  # 'kg/m^3', 'lb/ft^3'
        new_unit_w_water_english = new_unit_w_water * 0.0624279606  # 'kg/m^3', 'lb/ft^3'

        # viscosity
        new_dynamic_viscosity_english = new_dynamic_viscosity * 0.0208854342  # 'N*s/m^2', 'lbf*s/ft^2'

        return new_unit_w_water_english, new_dynamic_viscosity_english

    def _compute_constants_based_on_temperature_salinity_and_pressure(self, t, t2, t3, t4, t5, s, s15, s2, p):
        """Computes the constants based on variables for temperature, salinity, and pressure.

        Args:
            t (float): temperature
            t2 (float): temperature^2
            t3 (float): temperature^3
            t4 (float): temperature^4
            t5 (float): temperature^5
            s (float): salinity
            s15 (float): salinity^1.5
            s2 (float): salinity^2
            p (float): pressure
        """
        # Atlantic Ocean = 35.4
        # Pacific Ocean = 34 to 32
        # Indian Ocean = 35
        # Gulf of Mexico 34 to 36
        # Mediterranean Sea
        # Dead Sea 337
        # Great Salt Lake 317 (North Arm), 142
        # See wikipedia list of bodies of water by salinity

        # This method comes from:
        # https://www.translatorscafe.com/unit-converter/en-US/CalcData/salt-water-density/?s=35&su=ppt&t=5&tu=C&p=1000&pu=bar # noqa: E501
        # April 27, 2022

        rho_with_t_s_no_p = self._compute_water_density_based_on_temperature_and_salinity(t, t2, t3, t4, t5, s, s15, s2)
        k_with_t_s_p = self._compute_water_bulk_modulus_based_on_temperature_salinity_and_pressure(t, t2, t3, t4, s,
                                                                                                   s15, p)
        new_unit_w_water = rho_with_t_s_no_p / (1 - (p / k_with_t_s_p))
        new_unit_w_water = rho_with_t_s_no_p / (1 - (p / k_with_t_s_p))

        # viscosity
        new_dynamic_viscosity, valid = self._compute_water_viscosity_based_on_temperature_and_salinity(t, t2, s, s2)

        return new_unit_w_water, new_dynamic_viscosity
        return new_unit_w_water, new_dynamic_viscosity

    def set_constants_to_freshwater(self):
        """Set constants for freshwater."""
        self.input['Temperature'].set_val(60.0)
        self.input['Water salinity'].set_val(0.0)
        self.input['Water density'].set_val(1.94)
        self.input['Unit weight of water'].set_val(62.4)
        self.input['Water salinity'].set_val(0.0)
        self.input['Water density'].set_val(1.94)
        self.input['Unit weight of water'].set_val(62.4)
        self.input['Kinematic viscosity of water'].set_val(0.00001208)
        self.input['Dynamic viscosity of water'].set_val(1.1212)

    def set_constants_to_seawater(self):
        """Set constants for seawater."""
        self.input['Temperature'].set_val(60.0)
        self.input['Water salinity'].set_val(35.4)
        self.input['Water density'].set_val(1.99)
        self.input['Unit weight of water'].set_val(64.0)
        # self.input['Kinematic viscosity of water'].set_val(0.00001208)
        # self.input['Dynamic viscosity of water'].set_val(1.1212)
        self.input['Water salinity'].set_val(35.4)
        self.input['Water density'].set_val(1.99)
        self.input['Unit weight of water'].set_val(64.0)
        # self.input['Kinematic viscosity of water'].set_val(0.00001208)
        # self.input['Dynamic viscosity of water'].set_val(1.1212)

    def set_g_to_earth_at_sea_level(self):
        """Set gravity constant for Earth's sea-level."""
        self.input['Gravity'].set_val(32.17405)

    def compute_g_for_elevation(self, elevation):
        """Compute the gravity constant for an elevation on Earth.

        Args:
            elevation (float): elevation
        """
        gravity = 32.17405
        # Equation from Wikipedia https://en.wikipedia.org/wiki/Gravity_of_Earth
        # July 2, 2020
        earths_mean_radius = 2.0856e+7
        g = gravity * (earths_mean_radius / (earths_mean_radius + elevation)) ** 2.0
        self.input['Gravity'].set_val(g)
        return g

    def compute_pressure_for_elevation(self, elevation, pressure_at_sea_level=1, temp_at_sea_level=58.727861):
        """Compute the ambient pressure for an elevation on Earth.

        Args:
            elevation (float): elevation
            pressure_at_sea_level (float): pressure at sea level
            temp_at_sea_level (float): temperature at sea level
        """
        # Equation from https://www.mide.com/air-pressure-at-altitude-CalcData
        # April 27, 2022

        # Using ISA standards, the defaults for pressure and temperature at sea level are 101,325 Pa and 288 K
        # Can't use conversion code because of circular dependencies

        # static pressure (pressure at sea level) [Pa]
        pb = pressure_at_sea_level * 101325  # 'atm' to 'bar'
        # standard temperature (temperature at sea level) [K]
        tb = (temp_at_sea_level - 32) * .5556 + 273.15  # 'F' to 'K'
        # standard temperature lapse rate [K/m]
        lb = -0.0065
        # height above sea level [m]
        h = elevation / 3.281  # 'ft' to 'm'
        # height at the bottom of atmospheric layer [m]
        hb = 0.0
        # Universal gas constant [Nm/mol*K]
        r = 8.31432
        # gravitational acceleration constant = 9.380665
        g = 9.80665  # Needs to be in metric and reference sea level, so just use this value m/s^2.
        # molar mass of Earth's air = 0.0289644 [kg/mol]
        m = 0.0289644

        pressure = pb * (1 + lb / tb * (h - hb)) ** (-g * m / (r * lb))
        pressure_atm = pressure / 101325  # 'pa', 'atm'
        return pressure_atm
