"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


short_si_prefixes = {
    'Y': 1e24,
    'Z': 1e21,
    'E': 1e18,
    'P': 1e15,
    'T': 1e12,
    'G': 1e9,
    'M': 1e6,
    'k': 1e3,
    'h': 1e2,
    'da': 1e1,
    'd': 1e-1,
    'c': 1e-2,
    'm': 1e-3,
    'μ': 1e-6, 'u': 1e-6,  # 'u' as ASCII micro
    'n': 1e-9,
    'p': 1e-12,
    'f': 1e-15,
    'a': 1e-18,
    'z': 1e-21,
    'y': 1e-24,
}
si_prefixes = {
    'yotta': 1e24,
    'zetta': 1e21,
    'exa': 1e18,
    'peta': 1e15,
    'tera': 1e12,
    'giga': 1e9,
    'mega': 1e6,
    'kilo': 1e3,
    'hecto': 1e2,
    'deca': 1e1, 'deka': 1e1,
    'deci': 1e-1,
    'centi': 1e-2,
    'milli': 1e-3,
    'micro': 1e-6,
    'nano': 1e-9,
    'pico': 1e-12,
    'femto': 1e-15,
    'atto': 1e-18,
    'zepto': 1e-21,
    'yocto': 1e-24,
}

# Computer science (binary) prefixes for bytes and bits
short_cs_iec_prefixes = {
    'Yi': 2**80,   # yobibyte/yobibit
    'Zi': 2**70,   # zebibyte/zebibit
    'Ei': 2**60,   # exbibyte/exbibit
    'Pi': 2**50,   # pebibyte/pebibit
    'Ti': 2**40,   # tebibyte/tebibit
    'Gi': 2**30,   # gibibyte/gibibit
    'Mi': 2**20,   # mebibyte/mebibit
    'Ki': 2**10,   # kibibyte/kibibit
    # Also map common ambiguous forms to  SI standard)
    'Y': 1e24, 'Z': 1e21, 'E': 1e18, 'P': 1e15,
    'T': 1e12, 'G': 1e9, 'M': 1e6, 'K': 1e3, 'k': 1e3,
}
cs_iec_prefixes = {
    'yobi': 2**80,   # yobibyte/yobibit
    'zebi': 2**70,   # zebibyte/zebibit
    'exbi': 2**60,   # exbibyte/exbibit
    'pebi': 2**50,   # pebibyte/pebibit
    'tebi': 2**40,   # tebibyte/tebibit
    'gibi': 2**30,   # gibibyte/gibibit
    'mebi': 2**20,   # mebibyte/mebibit
    'kibi': 2**10,   # kibibyte/kibibit
    # Also map common ambiguous forms to  SI standard)
    'yotta': 1e24, 'zetta': 1e21, 'exa': 1e18, 'peta': 1e15, 'tera': 1e12, 'giga': 1e9, 'mega': 1e6,
    'kilo': 1e3, 'hecto': 1e2, 'deca': 1e1, 'deka': 1e1, 'deci': 1e-1, 'centi': 1e-2, 'milli': 1e-3,
    'micro': 1e-6, 'nano': 1e-9, 'pico': 1e-12, 'femto': 1e-15, 'atto': 1e-18, 'zepto': 1e-21, 'yocto': 1e-24,
}

short_cs_prefixes = {
    'Yi': 2**80, 'Y': 2**80,   # yobibyte/yobibit
    'Zi': 2**70, 'Z': 2**70,   # zebibyte/zebibit
    'Ei': 2**60, 'E': 2**60,   # exbibyte/exbibit
    'Pi': 2**50, 'P': 2**50,   # pebibyte/pebibit
    'Ti': 2**40, 'T': 2**40,   # tebibyte/tebibit
    'Gi': 2**30, 'G': 2**30,   # gibibyte/gibibit
    'Mi': 2**20, 'M': 2**20,   # mebibyte/mebibit
    'Ki': 2**10, 'K': 2**10, 'ki': 2**10, 'k': 2**10,   # kibibyte/kibibit

}

cs_prefixes = {
    'yobi': 2**80, 'yotta': 2**80,   # yobibyte/yobibit
    'zebi': 2**70, 'zetta': 2**70,   # zebibyte/zebibit
    'exbi': 2**60, 'exa': 2**60,     # exbibyte/exbibit
    'pebi': 2**50, 'peta': 2**50,    # pebibyte/pebibit
    'tebi': 2**40, 'tera': 2**40,    # tebibyte/tebibit
    'gibi': 2**30, 'giga': 2**30,    # gibibyte/gibibit
    'mebi': 2**20, 'mega': 2**20,    # mebibyte/mebibit
    'kibi': 2**10, 'kilo': 2**10,    # kibibyte/kibibit
}

accepted_si_base_units = [
    'm', 'm^2', 'm^3', 'g', 's', 'A', 'C', 'K', 'mol', 'cd', 'B', 'b', 'N',
    'meter', 'gram', 'second', 'ampere', 'kelvin', 'mole', 'candela', 'byte', 'bit', 'newton',
    'meters', 'grams', 'seconds', 'amperes', 'kelvins', 'moles', 'candelas', 'bytes', 'bits', 'newtons',
    'METERS', 'GRAMS', 'SECONDS', 'AMPERES', 'KELVINS', 'MOLES', 'CANDELAS', 'BYTES', 'BITS', 'NEWTONS',
    'METER', 'GRAM', 'SECOND', 'AMPERE', 'KELVIN', 'MOLE', 'CANDELA', 'BYTE', 'BIT', 'NEWTON']


def normalize_si_unit(unit: str, use_bytes: bool = False, use_strict_iec_definition: bool = False) -> tuple:
    """Normalize an SI-prefixed unit to its base unit and multiplier.

    E.g., 'km' -> ('m', 1e3), 'mg' -> ('g', 1e-3), 'megagram' -> ('gram', 1e6)

    Args:
        unit (str): The unit string, e.g., 'km', 'mm', 'megagram', 'μs'

    Returns:
        tuple: (base_unit, multiplier)
    """
    short_prefixes = short_si_prefixes
    prefixes = si_prefixes
    if use_bytes and use_strict_iec_definition:
        short_prefixes = short_cs_iec_prefixes
        prefixes = cs_iec_prefixes
    elif use_bytes:  # and not use_strict_iec_definition:
        prefixes = cs_prefixes
        short_prefixes = short_cs_prefixes

    if len(unit) < 2:  # Can have no prefix
        return False, unit, 1.0

    # Try name prefixes first (longest first, e.g., 'deca' before 'da' before 'd')
    for prefix in sorted(prefixes, key=len, reverse=True):
        # Also check for full-word prefixes (e.g., 'mega', 'micro'), and lower the case for typos
        if len(unit) >= 2 and unit.lower().startswith(prefix):
            base_unit = unit[len(prefix):]
            # if it has nothing left, it isn't a prefix and check for good base unit (don't take f out ft)
            if len(base_unit) > 0 and base_unit in accepted_si_base_units:
                return True, base_unit, prefixes[prefix]
    for prefix in sorted(short_prefixes, key=len, reverse=True):
        if unit.startswith(prefix):
            base_unit = unit[len(prefix):]
            # if it has nothing left, it isn't a prefix and check for good base unit (don't take f out ft)
            if len(base_unit) > 0 and base_unit in accepted_si_base_units:
                return True, base_unit, short_prefixes[prefix]
    # No prefix found, return as is
    return False, unit, 1.0


class ConversionByteCalc():
    """Perform byte unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Weir coefficient
    # ----------------------------------------------------------------------
    byte = ['byte', 'Byte', 'BYTE', 'B']
    bit = ['bit', 'Bit', 'BIT', 'b']

    def __init__(self, use_strict_iec_definition: bool = False):
        """Initialize the ConversionCalc Class.
        """
        self.use_strict_iec_definition = use_strict_iec_definition

    def get_si_complementary_unit(self, from_unit, use_strict_iec_definition: bool = False):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        _, from_unit_si, _ = normalize_si_unit(
            from_unit, use_bytes=True, use_strict_iec_definition=self.use_strict_iec_definition)
        _, to_unit_si, _ = normalize_si_unit(
            from_unit, use_bytes=True, use_strict_iec_definition=self.use_strict_iec_definition)
        if from_unit_si in self.byte or to_unit_si in self.bit:
            return True, from_unit
        return False, ''

    # ----------------------------------------------------------------------
    # Bytes/Bits
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert byte/bit units.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        _, from_unit_si, from_ratio = normalize_si_unit(
            from_unit, use_bytes=True, use_strict_iec_definition=self.use_strict_iec_definition)
        _, to_unit_si, to_ratio = normalize_si_unit(
            to_unit, use_bytes=True, use_strict_iec_definition=self.use_strict_iec_definition)

        if from_unit_si in self.byte:
            result, new_value = self.convert_from_byte(to_unit_si, value * from_ratio)
            return result, new_value / to_ratio
        elif from_unit_si in self.bit:
            result, new_value = self.convert_from_bit(to_unit_si, value * from_ratio)
            return result, new_value / to_ratio

        return False, value

    def convert_from_byte(self, to_unit, value):
        """Convert from the byte unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.byte:
            return True, value
        elif to_unit in self.bit:
            return True, value * 8

        else:
            return False, value

    def convert_from_bit(self, to_unit, value):
        """Convert from the bit unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.byte:
            return True, value / 8
        elif to_unit in self.bit:
            return True, value

        else:
            return False, value
