"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionTemperatureCalc:
    """perform temperature unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Temperature
    # ----------------------------------------------------------------------
    f = ['° f', '° F', '° fahrenheit', '° Fahrenheit', '°f', '°F',
         '°fahrenheit', '°Fahrenheit', 'f', 'F', 'fahrenheit', 'Fahrenheit']
    c = ['° c', '° C', '° celsius', '° Celsius', '°c', '°C', '°celsius', '°Celsius', 'c', 'C', 'celsius', 'Celsius']
    k = ['k', 'K', 'kelvin', 'Kelvin']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.f:
            return True, '° C'
        elif from_unit in self.k:
            return True, 'K'
        return False, ''

    # ----------------------------------------------------------------------------------------------------------------------
    # Temperature
    # ----------------------------------------------------------------------
    def convert_units(self, from_unit, to_unit, value):
        """Convert temperature.

        Args:
            from_unit (string): unit the value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if from_unit in self.f:
            return self.convert_from_fahrenheit(to_unit, value)
        elif from_unit in self.c:
            return self.convert_from_celsius(to_unit, value)
        elif from_unit in self.k:
            return self.convert_from_kelvin(to_unit, value)

        else:
            return False, value

    def convert_from_fahrenheit(self, to_unit, value):
        """Convert from the fahrenheit unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.f:
            return True, value
        elif to_unit in self.c:
            return True, (value - 32.0) * 5.0 / 9.0
        elif to_unit in self.k:
            return True, (value - 32.0) * 5.0 / 9.0 + 273.15

        else:
            return False, value

    def convert_from_celsius(self, to_unit, value):
        """Convert from the Celsius unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.f:
            return True, (value * 9.0 / 5.0) + 32.0
        elif to_unit in self.c:
            return True, value
        elif to_unit in self.k:
            return True, value + 273.15

        else:
            return False, value

    def convert_from_kelvin(self, to_unit, value):
        """Convert from the Kelvin unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.f:
            return True, (value - 273.15) * 9.0 / 5.0 + 32.0
        elif to_unit in self.c:
            return True, value - 273.15
        elif to_unit in self.k:
            return True, value

        else:
            return False, value
