"""Module to perform xms.api Query operations for the ADCIRC mapped boundary conditions component."""

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file
from xms.tides.data import tidal_data as td

# 4. Local modules
from xms.adcirc.data.mapped_flow_data import MappedFlowData
from xms.adcirc.dmi.bc_component_queries import query_for_constituent_datasets
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.model_checks.model_checker import grid_files_match


class MappedBcComponentQueries:
    """Performs xms.api calls for the BC coverage component."""
    def __init__(self, bc_comp, query):
        """Initialize the API helper.

        Args:
            bc_comp (:obj:`MappedBcComponent`): The component instance to perform API call for
            query (:obj:`xms.api.dmi.ActionRequest`): Object for communicating with XMS
        """
        self.bc_comp = bc_comp
        self.query = query

    def get_linked_do_ugrid(self, comp_item):
        """Query SMS for the mesh linked to the simulation.

        Args:
            comp_item (:obj:`TreeNode`): The mapped BC component tree item

        Returns:
            (:obj:`data_objects.parameters.UGrid`): The linked mesh data_object
        """
        mesh_item = tree_util.descendants_of_type(
            comp_item.parent, xms_types=['TI_MESH2D_PTR'], allow_pointers=True, only_first=True, recurse=False
        )
        if not mesh_item:  # No linked mesh
            raise RuntimeError('Error getting mesh geometry.')
        # Get the geometry dump
        do_ugrid = self.query.item_with_uuid(mesh_item.uuid)
        if not do_ugrid:
            raise RuntimeError('Error getting mesh geometry.')
        return do_ugrid

    def query_for_tidal_mapping_data(self, tidal_data, xms_data):
        """Get the dataset and geometry dumps of the selected user constituent amplitudes and phases.

        Args:
            tidal_data (:obj:`TidalData`): Data of the source tidal component
            xms_data (:obj:`dict`): Dictionary containing the data retrieved from XMS prior to mapping. Will add:
            ::
                {
                    'amp_dsets': [DatasetReader]
                    'phase_dsets': [DatasetReader]
                    'geoms': {str: InterpLinear} - key is geom UUID
                    'mesh': CoGrid
                }

        Returns:
            (:obj:`bool`): False on error
        """
        amp_dsets = []
        phase_dsets = []
        geoms = {}
        try:
            if tidal_data.info.attrs['source'].item() == td.USER_DEFINED_INDEX:
                num_amps = tidal_data.user.Amplitude.data.size
                num_phases = tidal_data.user.Phase.data.size
                if num_amps != num_phases:
                    raise RuntimeError('Number of amplitude datasets does not match the number of phase datasets.')
                elif num_amps == 0:
                    raise RuntimeError('No amplitude or phase datasets selected.')
                amp_dsets, phase_dsets, geoms = query_for_constituent_datasets(
                    self.query, tidal_data.user.Amplitude, tidal_data.user.Phase
                )
        except Exception:
            XmLog().instance.error(
                'Could not retrieve user defined constituent amplitude and phase datasets for mapping tidal '
                'constituents.'
            )
            return False
        xms_data['amp_dsets'] = amp_dsets
        xms_data['phase_dsets'] = phase_dsets
        xms_data['geoms'] = geoms

        # Get the domain mesh
        try:
            comp_item = tree_util.find_tree_node_by_uuid(self.query.project_tree, self.bc_comp.uuid)
            do_ugrid = self.get_linked_do_ugrid(comp_item)
            xms_data['mesh'] = read_grid_from_file(do_ugrid.cogrid_file)
        except Exception:
            XmLog().instance.exception('Could not retrieve domain mesh.')
            return False

        # Check if the currently linked mesh is out of date with the one the BCs were mapped with.
        if not grid_files_match(do_ugrid.cogrid_file, self.bc_comp.data.info.attrs['grid_crc']):
            XmLog().instance.warning(
                'The mesh currently linked to the simulation has been edited since the applied boundary conditions '
                'were created. The applied tidal constituents created by this operation may be invalid. See '
                'https://www.xmswiki.com/wiki/SMS:ADCIRC for more information.'
            )
        return True

    def add_xms_data_for_tidal_mapping(self, xms_data, mapped_tidal_comp):
        """Send the new mapped Tidal component to XMS, unlink the source tidal constituents, and ActionRequests.

        Context must be at the simulation level

        Args:
            xms_data (:obj:`dict`): Dictionary containing the data retrieved from XMS prior to mapping
            mapped_tidal_comp (:obj:`xms.data_objects.parameters.Component`): The new mapped BC component
        """
        # Add the mapped component to the Context and send back to XMS.
        self.query.add_component(mapped_tidal_comp)
        self.query.link_item(xms_data['sim_uuid'], mapped_tidal_comp.uuid)

        # Unlink the source tidal constituents
        self.query.unlink_item(xms_data['sim_uuid'], xms_data['tidal_uuid'])

    def get_coverage_convert_data(self):
        """Get all the XMS data needed to convert a mapped BC component to a source BC coverage.

        Returns:
            :obj:`dict`: The xms data required to perform the conversion.
                'temp_dir': :obj:`str`,  # Path to the XMS temp component directory

                'comp_name': :obj:`str`,  # Tree item name of the mapped BC component being converted

                'mesh':

                'flow_data': :obj:`MappedFlowData`,  # Mapped flow component data. None if not applicable.
        """
        xms_data = {
            'temp_dir': '',
            'comp_name': 'Boundary Conditions',
            'mesh': None,
            'flow_data': None,
        }
        try:
            # Get the mapped BC component's tree item name and the XMS temp component directory.
            comp_item = tree_util.find_tree_node_by_uuid(self.query.project_tree, self.bc_comp.uuid)
            xms_data['comp_name'] = comp_item.name
            xms_data['temp_dir'] = os.path.join(self.query.xms_temp_directory, 'Components')
            os.makedirs(xms_data['temp_dir'], exist_ok=True)

            # Get the mapped flow component data, if there is any.
            mapped_flows = tree_util.descendants_of_type(
                comp_item.parent, unique_name='Mapped_Flow_Component', recurse=False
            )
            if mapped_flows:  # Should be at most one
                xms_data['flow_data'] = MappedFlowData(mapped_flows[0].main_file)
        except Exception:
            XmLog().instance.exception(
                'Error retrieving data from SMS for converting applied boundary conditions to a coverage.'
            )
        return xms_data

    def add_xms_data_for_coverage_convert(self, bc_cov, bc_comp, py_comp, flow_forcing_mesh, flow_forcing_dsets):
        """Send the newly converted source BC coverage and its hidden component to XMS.

        Parent of the current Context node should be the mapped BC component's parent simulation.

        Args:
            bc_cov (:obj:`xms.data_objects.parameters.Coverage`): The BC coverage geometry
            bc_comp (:obj:`xms.data_objects.parameters.Component`): The BC coverage's hidden component.
            py_comp (:obj:`xmsadcirc.components.bc_component.BcComponent`): The BC component's Python object.
            flow_forcing_mesh (:obj:`xms.data_objects.parameters.UGrid`): Geometry containing the locations of flow
                boundary nodes. None if no flow boundaries or using non-periodic flow forcing (fort.20).
            flow_forcing_dsets (:obj:`list`): Periodic flow forcing amplitude and phase datasets. Empty list if no flow
                boundaries or using non-periodic flow forcing (fort.20).
        """
        comp_data = [
            {
                'component_coverage_ids': [py_comp.uuid, py_comp.update_ids],
                'display_options': py_comp.get_display_options()
            }
        ]
        self.query.add_coverage(
            bc_cov,
            model_name='ADCIRC',
            coverage_type='Boundary Conditions',
            components=[bc_comp],
            component_keywords=comp_data
        )

        # Add the periodic flow forcing geometry, if periodic flow forcing was defined.
        if flow_forcing_mesh:
            self.query.add_ugrid(flow_forcing_mesh)
            # Add the periodic flow amplitude and phase datasets (1 each per defined periodic flow forcing constituent).
            for dset in flow_forcing_dsets:
                self.query.add_dataset(dset)
