"""Worker thread for the fort.15 exporter."""

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.adcirc.feedback.xmlog import XmLog
from xms.adcirc.file_io.fort15_writer import Fort15Writer


class Fort15ExportWorkerThread(ProcessFeedbackThread):
    """Worker thread for exporting ADCIRC fort.15 files."""
    def __init__(self, filename, query, xms_data, parent):
        """Construct the worker.

        Args:
            filename (:obj:`str`): Path to the export location.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            xms_data (:obj:`dict`): Dict of all the XMS data required to export the fort.15 and any other simulation
                files if this is a full simulation export.
            parent (:obj:`QWidget`): Parent of the QThread
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._filename = filename
        self._query = query
        self._xms_data = xms_data

    def _do_work(self):
        """Thread runner that exports the fort.15."""
        try:
            writer = Fort15Writer(filename=self._filename, xms_data=self._xms_data, query=self._query)
            writer.write()
        except Exception:
            XmLog().instance.exception('Error! Could not export ADCIRC simulation.')


def export_fort15_with_feedback(filename, query, xms_data, note, parent):
    """Export the fort.15 with a feedback dialog.

    Args:
        filename (:obj:`str`): Path to the fort.15 to be exported
        query (:obj:`Optional[Query]`): For communicating with XMS. Will not be used if `xms_data` is provided.
            `xms_data` and `query` should not be provided when this script is called from XMS as a simulation-level
            export for a model run. Tests should provide `xms_data` and not `query`. Other entry points such as
            component dialogs should provide `query` at the ADCIRC simulation component level but not `xms_data`.
        xms_data (:obj:`Optional[dict]`): All the data required to export. If not provided, will Query XMS for it.
        ::
            {
                'projection': :obj:`data_objects.parameters.Projection`,

                'bc_data': :obj:`MappedBcData`,

                'sim_data': :obj:`SimData`,

                'tidal_data': :obj:`[MappedTidalData]`,

                'domain_name': :obj:`str`,

                'wind_grid': :obj:`str` (CoGrid file for wind grid),

                'station_comp': :obj:`StationComponent`,

                'station_pts': :obj:`{id: data_objects.parameters.Point}`

                'flow_data': :obj:`MappedFlowData`
            }

        note (:obj:`str`): Note to add to the dialog
        parent (:obj:`QWidget`): The Qt parent
    """
    ensure_qapplication_exists()  # Won't be a QApp if simulation export, will be if in partial export command dialog

    # Parse parent HWNDs and icon path from commandline arguments.
    main_hwnd = None
    if parent is None:
        parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
        win_cont = xms_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = win_gui.create_and_connect_raise_timer(main_hwnd, win_cont)  # Keep the timer in scope
        parent = win_cont

    testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    # autoload = 'Automatically close this dialog on successful export' if not query else ''
    worker = Fort15ExportWorkerThread(filename, query, xms_data, parent)
    display_text = {
        'title': 'Exporting fort.15',
        'working_prompt': f'Exporting fort.15 to \"{filename}\" Please wait...',
        'error_prompt': 'Error(s) encountered exporting fort.15. Review log output for more details.',
        'warning_prompt': 'Warning(s) encountered exporting fort.15. Review log output for more details.',
        'success_prompt': f'Successfully exported fort.15 to  \"{filename}\"',
        'note': note,
        'auto_load': 'Automatically close this dialog on successful export',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms', worker, parent)
    feedback_dlg.testing = testing
    if feedback_dlg.exec() and not testing and not query:  # User did not cancel, we are not testing, and exporting sim
        if LogEchoQSignalStream.logged_error:  # If we have an error on simulation export, log the error for XMS
            XmEnv.report_export_error()
    elif not query:  # Simulation export, but the user cancelled
        XmEnv.report_export_aborted()
    if main_hwnd is not None:
        win_gui.raise_main_xms_window(main_hwnd)  # Bring top-level Python window to foreground
