"""Writer for ADCIRC fort.22 wind input files."""

# 1. Standard Python modules
import copy
import datetime
import math
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.constraint import read_grid_from_file
from xms.core.filesystem import filesystem as io_util
from xms.data_objects.parameters import datetime_to_julian, FilterLocation, julian_to_datetime
from xms.guipy.settings import SettingsManager
from xms.guipy.time_format import ISO_DATETIME_FORMAT

# 4. Local modules
from xms.adcirc.data.sim_data import REG_KEY_SUPPRESS_WIND_FILE
from xms.adcirc.dmi.fort22_data_getter import read_fort22_data_json
from xms.adcirc.feedback.xmlog import report_error, XmLog
from xms.adcirc.file_io.dataset_interpolator import DatasetInterpolator


def normalize_angle(a_angle):
    """Ensures that an angle is between 0-360 degrees.

    Args:
        a_angle (:obj:`float`): The angle (in degrees) to normalize to the 0-360 degree range

    Returns:
        (:obj:`float`): a_angle between 0-360 degrees
    """
    while a_angle < 0.0:
        a_angle += 360.0
    while a_angle >= 360.0:
        a_angle -= 360.0
    return a_angle


def cart_to_meteor(a_angle):
    """Converts an angle from Cartesian convention to Meteorological convention.

    Args:
        a_angle (:obj:`float`): The angle to convert (in Cartesian convention degrees)

    Returns:
        (:obj:`float`): a_angle in Meteorlogical convention degrees between 0-360
    """
    met_angle = 270.0 - a_angle
    return normalize_angle(met_angle)


def cart_to_ocean(a_angle):
    """Converts an angle from Cartesian convention to Oceanographic convention.

    Args:
        a_angle (:obj:`float`): The angle to convert (in Cartesian convention degrees)

    Returns:
        (:obj:`float`): a_angle in Oceanographic convention degrees between 0-360
    """
    ocean_angle = 90.0 - a_angle
    return normalize_angle(ocean_angle)


def transform_1d_list_for_grid(vals, isize, jsize):
    """Convert a 1-D list of values to a 2-D list where rows are i-direction values and columns are j-direction values.

    This is for datasets on wind grids (NWS=3,6).

    Args:
        vals (:obj:`list`): The values to unflatten.
        isize (:obj:`int`): Size of the grid in the i direction
        jsize (:obj:`int`): Size of the grid in the j direction

    Returns:
        (:obj:`list`): See description
    """
    return [[vals[(j * isize) + i] for i in range(isize)] for j in range(jsize)]


def get_time_info(sim_data, radstress):
    """Get the target timestep increment size, run length in days, and simulation reference datetime.

    Args:
        sim_data (:obj:`SimData`): The simulation component data.
        radstress (:obj:`bool`): True if exporting radiation stress dataset (fort.23), False if exporting a fort.22

    Returns:
        (:obj:`tuple`): Tuple where first element is the simulation reference date as a Julian double, the second
        element is the target timestep increment size in seconds, and the third element is the length of the ADCIRC
        run in days.
    """
    # Convert the interpolation reference time to a Julian double.
    julian_ref = datetime_to_julian(datetime.datetime.strptime(sim_data.timing.attrs['ref_date'], ISO_DATETIME_FORMAT))
    # Get the wind time increment in seconds. Use DTDP if NWS=1, RSTIMINC if exporting fort.23, otherwise use WTIMINC.
    if radstress:
        timeinc = sim_data.wind.attrs['RSTIMINC']
    else:
        timeinc = sim_data.timing.attrs['DTDP'] if sim_data.wind.attrs['NWS'] == 1 else sim_data.wind.attrs['WTIMINC']
    # Get the length of the run in days.
    rundays = sim_data.timing.attrs['RUNDAY']
    return julian_ref, timeinc, rundays


class Fort22ExportError(RuntimeError):
    """Exception for known errors, will just print pretty message for user without traceback."""
    pass


class Fort22Writer:
    """fort.22 exporter for the SMS DMI ADCIRC interface."""
    def __init__(self, filename, sim_export, xms_data):
        """Construct the writer.

        Args:
            filename (:obj:`str`): Filename of the file to write.
            sim_export (:obj:`bool`): True if called as part of a simulation export.
                If True, no error thrown when NWS=0.
            xms_data (:obj:`dict`): Dictionary containing all the data needed from XMS.
        """
        self._writer = None
        self._filename = filename
        self._sim_export = sim_export
        self._suppress_missing_file_errors = False
        self._xms_data = xms_data

        # Check registry to see if we are suppressing missing wind file errors.
        self._read_registry_settings()

        # if we are using an existing file, copy it.
        if self._xms_data['sim_data'].wind.attrs['use_existing']:
            self._copy_existing_fort22()
        # creating a new fort.22 - set up the writer for the specific NWS type.
        else:
            self._set_nws_writer()

    def _read_registry_settings(self):
        """Read settings from registry for suppressing error messages."""
        settings = SettingsManager()
        self._suppress_missing_file_errors = settings.get_setting('xmsadcirc', REG_KEY_SUPPRESS_WIND_FILE, 0) == 1

    def _copy_existing_fort22(self):
        """Copy an existing fort.22 to the export location if option enabled.

        Returns:
            (:obj:`bool`): True if existing fort.22 copied to the export location. No more work for writer to do.
        """
        # Copy selected files to the export location.
        nws_type = self._xms_data['sim_data'].wind.attrs['NWS']
        # Check if NWS type has a fort.22 file (12, 15, and 16 do but we will always write it).
        if nws_type not in [0, 10, 11, 12, 15, 16]:
            fort22 = io_util.resolve_relative_path(
                self._xms_data['sim_data'].info.attrs['proj_dir'],
                self._xms_data['sim_data'].wind.attrs['existing_file']
            )
            if os.path.isfile(fort22):
                io_util.copyfile(fort22, self._filename)
                return True
            elif not self._suppress_missing_file_errors:
                report_error('Unable to copy existing fort.22 to export folder.', self._sim_export)
        return False

    def _set_nws_writer(self):
        """Initialize the nested writer for this specific NWS type."""
        if not self._xms_data['sim_data']:
            return  # Need to have this by now.

        # Determine which writer to use based on the simplified NWS type number.
        nws_simple = self._xms_data['sim_data'].wind.attrs['NWS']
        if nws_simple == 0:  # Wind is not enabled or format has no fort.22, don't write it
            # Raise an error if wind is disabled and this is not an entire simulation export (came from GUI command).
            # Full simulation export, doesn't have to have wind defined so fail silently.
            if not self._sim_export:
                report_error(
                    "Wind is not enabled for this ADCIRC simulation. Setup wind parameters in the simulation's "
                    '"Model Control" dialog and try again.', False
                )
            return
        elif nws_simple in [1, 2, 5]:
            self._writer = NWSMeshDatasetWriter(self._xms_data)
        elif nws_simple == 3:
            self._writer = NWS3Writer(self._xms_data)
        elif nws_simple == 4:
            self._writer = NWS4Writer(self._xms_data)
        elif nws_simple in [6, 7]:
            self._writer = NWS6Writer(self._xms_data)
        elif nws_simple in [10, 11]:
            self._writer = NWSMetFileCopier(self._xms_data)
        elif nws_simple == 12:
            self._writer = NWS12Writer(self._xms_data)
        elif nws_simple in [15, 16]:
            self._writer = NWSMetFileWriter(self._xms_data)
        elif nws_simple in [8, 19, 20]:
            self._writer = NWSWindTrackWriter(self._xms_data)
            self._writer.load_data()  # Read wind coverage dump if storm track type and no query
        else:
            raise Fort22ExportError(f'Unsupported NWS type: {nws_simple}. fort22 file was not exported.')
        self._writer.suppress_missing_file_errors = self._suppress_missing_file_errors
        self._writer.sim_export = self._sim_export

    def _write_radstress(self):
        """Write the fort.23, if applicable."""
        if not self._xms_data.get('radstress'):
            return  # No radiation stress defined
        writer = NWS4Writer(xms_data=self._xms_data, radstress=True)
        writer.write(os.path.join(os.path.dirname(self._filename), 'fort.23'))

    def write(self):
        """Export the fort.22."""
        if self._writer:
            self._writer.write(self._filename)
            self._write_radstress()


# Specialized classes for writing specific NWS types.
class NWSWriter:
    """Base class for specialized fort.22 writers."""
    def __init__(self, xms_data):
        """Construct the writer.

        Args:
            xms_data (:obj:`dict`): Dictionary of data retrieved from XMS needed to export the fort.22. Should at least
                have the simulation component data at this point.
        """
        self._xms_data = xms_data
        self.suppress_missing_file_errors = False
        self.sim_export = False

    def write(self, filename):
        """Write the fort.22 in format required for specific NWS type.

        Args:
            filename (:obj:`str`): Location of the file to write
        """
        pass


class NWSMeshDatasetWriter(NWSWriter):
    """Writer for NWS=1,2,5 types (wind on specified as domain mesh datasets)."""
    def __init__(self, xms_data):
        """Construct the writer.

        Args:
            xms_data (:obj:`dict`): Dictionary of data retrieved from XMS needed to export the fort.22. Should at least
                have the simulation component data at this point.
                ::
                    {
                        sim_data: SimData,
                        global_time: datetime.datetime,
                        pressure: xms.data_objects.parameters.Dataset,
                        vector: xms.data_objects.parameters.Dataset,
                    }
        """
        super().__init__(xms_data)

    def write(self, filename):
        """Write the domain wind datasets to fort.22.

        Args:
            filename (:obj:`str`): Location of the file to write
        """
        sim_data = self._xms_data['sim_data']
        julian_ref, timeinc, rundays = get_time_info(sim_data, False)

        # Get the wind stress/velocity and pressure values. Interpolate to target timesteps.
        vector_interper = DatasetInterpolator(
            self._xms_data['vector'], julian_ref, timeinc, rundays, self._xms_data['global_time']
        )
        pressure_interper = DatasetInterpolator(
            self._xms_data['pressure'], julian_ref, timeinc, rundays, self._xms_data['global_time']
        )

        num_ts = vector_interper.get_num_timesteps()
        with open(filename, 'w', 100000000) as f:  # Buffer the in-memory data because these can get big.
            for ts_idx in range(num_ts):  # Loop through the timesteps
                XmLog().instance.info(f'Interpolating timestep {ts_idx + 1} of {num_ts}...')
                wind_ts = vector_interper.interpolate_timestep(ts_idx)
                press_ts = pressure_interper.interpolate_timestep(ts_idx)
                for node, (wind_val, press_val) in enumerate(zip(wind_ts, press_ts)):
                    # node_number  wind_speed  wind_direction  pressure
                    f.write(f"{node + 1:<8} {wind_val[0]:g} {wind_val[1]:g} {press_val:g}\n")


class NWS3Writer(NWSWriter):
    """Writer for NWS=3 types (wind velocity specified on a Cartesian grid)."""
    def __init__(self, xms_data):
        """Construct the writer.

        Args:
            xms_data (:obj:`dict`): Dictionary of data retrieved from XMS needed to export the fort.22. Should at least
                have the simulation component data at this point.
                    ::
                        {
                            sim_data: SimData,
                            global_time: datetime.datetime,
                            wind_grid: xms.data_objects.parameters.RectilinearGrid,
                            velocity: xms.data_objects.parameters.Dataset,
                        }
        """
        super().__init__(xms_data)

    def write(self, filename):
        """Write the wind grid velocity dataset to fort.22.

        Args:
            filename (:obj:`str`): Location of the file to write
        """
        co_grid = read_grid_from_file(self._xms_data['wind_grid'])
        size = (len(co_grid.locations_x) - 1, len(co_grid.locations_y) - 1)
        sim_data = self._xms_data['sim_data']
        julian_ref, timeinc, rundays = get_time_info(sim_data, False)

        # Interpolate the velocity dataset through time to the target timesteps.
        velocity_interper = DatasetInterpolator(
            self._xms_data['velocity'], julian_ref, timeinc, rundays, self._xms_data['global_time']
        )

        num_ts = velocity_interper.get_num_timesteps()
        with open(filename, 'w', 100000000) as f:  # Buffer the in-memory data because these can get big.
            for ts_idx in range(num_ts):  # Loop through the timesteps
                XmLog().instance.info(f'Interpolating timestep {ts_idx + 1} of {num_ts}...')
                ts_julian = velocity_interper.get_target_time(ts_idx)
                ts_dt = julian_to_datetime(ts_julian)
                # IWTIME = (if NWS =3, 103) time of the wind field in the following integer format:
                #   YEAR*1000000 + MONTH*10000 + DAY*100 + HR
                # ^^^This is not true. That's what the docs say but should be: YYMMDDHH
                time_integer = int(ts_dt.year * 1000000 + ts_dt.month * 10000 + ts_dt.day * 100 + ts_dt.hour)
                date_str = str(time_integer)[2:]  # Strip off first two numbers from the year.
                f.write(date_str + '\n')

                # Convert Vx, Vy to magnitude and direction
                velocity_ts = velocity_interper.interpolate_timestep(ts_idx)
                for cell in velocity_ts:
                    vx = cell[0]
                    vy = cell[1]
                    cell[0] = math.hypot(vx, vy)  # compute the magnitude
                    cell[1] = math.degrees(math.atan2(vy, vx))  # compute the cartesian direction

                # Convert from a 1-D list to a 2-D list (rows=i values, cols=j values)
                velocity_ts = transform_1d_list_for_grid(velocity_ts, size[0], size[1])

                # By default:
                #   SMS orders grid cells in i-j order start at bottom left corner
                #     inner loop increments col (to the right), outer loop increments the row (up).
                #   We think ADCIRC is expecting cells in i-j order starting at the top right corner
                #     inner loop increments col (to the right), outer loop decrements the row (down).
                #     This is the order specified for NWS=6. For NWS=3 it is not explicitly defined in ADCIRC docs.
                for i in range(size[1] - 1, -1, -1):  # Start at the top
                    row = velocity_ts[i]
                    for j in range(0, size[0]):  # Speed
                        f.write(f'{row[j][0]:.3f} ')
                    f.write('\n')
                for i in range(size[1] - 1, -1, -1):  # Start at the top
                    row = velocity_ts[i]
                    for j in range(0, size[0]):  # Direction
                        f.write(f'{cart_to_meteor(row[j][1]):.3f} ')
                    f.write('\n')


class NWS4Writer(NWSMeshDatasetWriter):
    """Writer for NWS=4 types (wind on specified as domain mesh datasets in PBL/JAG format)."""
    def __init__(self, xms_data, radstress=False):
        """Construct the writer.

        Args:
            xms_data (:obj:`dict`): Dictionary of data retrieved from XMS needed to export the fort.22. Should at least
                have the simulation component data at this point.
                    ::
                        {
                            sim_data: SimData,
                            global_time: datetime.datetime,
                            scalar: xms.data_objects.parameters.Dataset,
                            vector: xms.data_objects.parameters.Dataset,
                        }

            radstress (:obj:`bool`): True if exporting a radstress dataset (fort.23), False if exporting a fort.22 for
                a mesh dataset type.
        """
        super().__init__(xms_data)
        self._radstress = radstress

    def write(self, filename):
        """Write the domain mesh datasets to fort.22 in PBL/JAG format.

        Args:
            filename (:obj:`str`): Location of the file to write
        """
        sim_data = self._xms_data['sim_data']
        julian_ref, timeinc, rundays = get_time_info(sim_data, self._radstress)

        if self._radstress:
            # If exporting a radiation stress dataset (fort.23), there is no scalar column in the file.
            vector_interper = DatasetInterpolator(
                self._xms_data['radstress'], julian_ref, timeinc, rundays, self._xms_data['global_time']
            )
            pressure_interper = None
        else:
            # Get the wind stress/velocity values. Interpolate to target timesteps.
            vector_interper = DatasetInterpolator(
                self._xms_data['vector'], julian_ref, timeinc, rundays, self._xms_data['global_time']
            )
            pressure_interper = DatasetInterpolator(
                self._xms_data['pressure'], julian_ref, timeinc, rundays, self._xms_data['global_time']
            )

        num_ts = vector_interper.get_num_timesteps()
        with open(filename, 'w', 100000000) as f:  # Buffer the in-memory data because these can get big.
            f.write('\n')  # Needs to be a blank line at the beginning of the file. See Mantis issue 12004
            for ts_idx in range(num_ts):  # Loop through the timesteps
                XmLog().instance.info(f'Interpolating timestep {ts_idx + 1} of {num_ts}...')
                vec_ts = vector_interper.interpolate_timestep(ts_idx)
                press_ts = None
                if pressure_interper:
                    press_ts = pressure_interper.interpolate_timestep(ts_idx)
                    f.write(f' # -- Wind Record Number {ts_idx + 1}\n')
                else:
                    f.write(f' # -- Radiation Stress Record Number {ts_idx + 1}\n')
                for node in range(len(vec_ts)):
                    vec_val = vec_ts[node]
                    press_val = press_ts[node] if press_ts is not None else 0.0
                    if vec_val[0] != 0.0 or vec_val[1] != 0.0:  # Only write values if non-zero
                        if self._radstress:
                            # node_number  radiation stress x  radiation stress y
                            f.write(f'{node + 1:>8} {vec_val[0]:>12E} {vec_val[1]:>12E}\n')
                        else:
                            # node_number  wind_speed  wind_direction  pressure
                            f.write(f'{node + 1:>8} {vec_val[0]:>12E} {vec_val[1]:>12E} {press_val:>12E}\n')


class NWS6Writer(NWS3Writer):
    """Writer for NWS=6 types (wind velocity and pressure specified on a Cartesian grid)."""
    def __init__(self, xms_data):
        """Construct the writer.

        Args:
            xms_data (:obj:`dict`): Dictionary of data retrieved from XMS needed to export the fort.22. Should at least
                have the simulation component data at this point.
                    ::
                        {
                            sim_data: SimData,
                            global_time: datetime.datetime,
                            'wind_grid': str (CoGrid file for wind grid),
                            velocity: xms.data_objects.parameters.Dataset,
                            pressure: xms.data_objects.parameters.Dataset,
                        }
        """
        super().__init__(xms_data)

    def write(self, filename):
        """Write the wind grid velocity and pressure datasets to fort.22 for NWS = 6 (rectangular grid).

        Args:
            filename (:obj:`str`): Location of the file to write
        """
        # grid can be in degrees or meters, to match the mesh (so mostly degrees)
        co_grid = read_grid_from_file(self._xms_data['wind_grid'])
        size = (len(co_grid.locations_x) - 1, len(co_grid.locations_y) - 1)
        sim_data = self._xms_data['sim_data']
        julian_ref, timeinc, rundays = get_time_info(sim_data, False)

        # Interpolate the velocity and pressure datasets through time to the target timesteps.
        velocity_interper = DatasetInterpolator(
            self._xms_data['velocity'], julian_ref, timeinc, rundays, self._xms_data['global_time']
        )
        pressure_interper = DatasetInterpolator(
            self._xms_data['pressure'], julian_ref, timeinc, rundays, self._xms_data['global_time']
        )

        num_ts = velocity_interper.get_num_timesteps()
        with open(filename, 'w', 100000000) as f:  # Buffer the in-memory data because these can get big.
            for ts_idx in range(num_ts):  # Loop through the timesteps
                XmLog().instance.info(f'Interpolating timestep {ts_idx + 1} of {num_ts}...')
                velocity_ts = velocity_interper.interpolate_timestep(ts_idx)
                press_ts = pressure_interper.interpolate_timestep(ts_idx)
                velocity_ts = transform_1d_list_for_grid(velocity_ts, size[0], size[1])
                press_ts = transform_1d_list_for_grid(press_ts, size[0], size[1])

                # By default:
                #   SMS orders grid cells in i-j order start at bottom left corner
                #     inner loop increments col (to the right), outer loop increments the row (up).
                #   We think ADCIRC is expecting cells in i-j order starting at the top right corner
                #     inner loop increments col (to the right), outer loop decrements the row (down).
                write_ts_idx = True
                for i in range(size[1] - 1, -1, -1):  # Start at the top
                    wind_row = velocity_ts[i]
                    press_row = press_ts[i]
                    for j in range(0, size[0]):
                        # There was code here previously to convert:
                        #   wind components from Vx,Vy to Mag,Dir
                        #   wind direction from cartesian to oceanographic.
                        #   I think this was because NWS 3 uses mag/dir
                        # vx  vy  pressure
                        if write_ts_idx:
                            # AKZ added a ts index to make the file readable.  I'm not sure ADCIRC will like it
                            f.write(f"{wind_row[j][0]:.3f} {wind_row[j][1]:.3f} {press_row[j]:.3f}   // ts{ts_idx+1}\n")
                            write_ts_idx = False
                        else:
                            f.write(f"{wind_row[j][0]:.3f} {wind_row[j][1]:.3f} {press_row[j]:.3f}\n")


class NWSWindTrackWriter(NWSWriter):
    """Export best track format fort.22 for NWS=8,19,20 wind track coverage types.

    https://www.nrlmry.navy.mil/atcf_web/docs/database/new/abrdeck.html
    """
    BASINS = ['WP', 'IO', 'SH', 'CP', 'EP', 'AL', 'SL']
    TY = ['DB', 'TD', 'TS', 'TY', 'ST', 'TC', 'HU', 'SD', 'SS', 'EX', 'IN', 'DS', 'LO', 'WV', 'ET', 'XX']
    WINDCODES = [
        'AAA', 'NNS', 'NES', 'EES', 'SES', 'SSS', 'SWS', 'WWS', 'NWS', 'NNQ', 'NEQ', 'EEQ', 'SEQ', 'SSQ', 'SWQ', 'WWQ',
        'NWQ'
    ]  # The documentation doesn't mention semicircles...
    RAD = ['34', '50', '64', '100']
    SUBREGIONS = ['A', 'B', 'C', 'E', 'L', 'P', 'Q', 'S', 'W']
    DEPTHS = ['D', 'M', 'S', 'X']

    def __init__(self, xms_data):
        """Construct the writer.

        Args:
            xms_data (:obj:`dict`): Dictionary of data retrieved from XMS needed to export the fort.22. Should at least
                have the simulation component data at this point.
                    ::
                        {
                            sim_data: SimData,
                            wind_cov: xms.coverage._windCoverage.WindCoverage  # Wind coverage dump
                        }
        """
        super().__init__(xms_data)
        self.coords = []
        self.timesteps = []

    @staticmethod
    def _get_lat_lon_coords(x, y):
        """Get lat,lon coordinate strings in ATCF format given an x,y point location.

        Flips x,y to be lat,lon and stringfies with N,S,E,W character

        Args:
            x (:obj:`float`): X coordinate of the point
            y (:obj:`float`): Y coordinate of the point

        Returns:
            (:obj:`tuple of str`): Tuple of str where first element is latitude and second element is longitude.
        """
        xstr = round(x * 10)
        if x < 0:
            xstr = str(abs(xstr)) + 'W'
        else:
            xstr = str(xstr) + 'E'

        ystr = round(y * 10)
        if y < 0:
            ystr = str(abs(ystr)) + 'S'
        else:
            ystr = str(ystr) + 'N'
        return ystr, xstr

    def _load_nodal_atts(self):
        """Load the attributes of each storm track node from the coverage dump file."""
        intensity_col = 11
        rad1_col = 13
        rad2_col = 14
        rad3_col = 15
        rad4_col = 16

        # Get storm track nodal values for the file
        basin = self.BASINS[self._xms_data['wind_cov'].m_basin]
        sub_reg = self._xms_data['wind_cov'].m_subregion
        cy_num = self._xms_data['wind_cov'].m_cycloneNumber
        nws = self._xms_data['sim_data'].wind.attrs['NWS']
        start_time = None
        for i in range(len(self._xms_data['wind_cov'].m_nodeWind)):  # Get values for each node (point)
            ncoords = self.coords[i]
            n = self._xms_data['wind_cov'].m_nodeWind[i]
            timestep = list()
            # BASIN
            timestep.append(f"{basin:>s}")
            # CY
            timestep.append(f"{cy_num:>2d}")
            # YYYYMMDDHH
            node_dt = julian_to_datetime(n.m_date)
            timestep.append(f'{node_dt.year:>04d}{node_dt.month:>02d}{node_dt.day:>02d}{node_dt.hour:>02d}')
            # TECHNUM / MIN
            timestep.append(f"{n.m_iTechNumOrMinutes:>02d}")
            # TECH
            timestep.append(f"{n.m_sTech:>4.4s}")
            # TAU - time offset in hours from storm start (ignored if NWS=8 and all "BEST" type).
            if start_time is None:  # First node must be 0
                start_time = node_dt
                timestep.append(f"{0:>3d}")
            else:
                offset = (node_dt - start_time).total_seconds()
                offset = int(round(offset / 3600.0))  # Convert to hours from storm start and round to nearest integer
                timestep.append(f"{offset:>3d}")
            # LatN / S
            timestep.append(f"{ncoords[0]:>4s}")
            # LonE / W
            timestep.append(f"{ncoords[1]:>5s}")
            # VMAX
            timestep.append(f"{n.m_iVMax:>3d}")
            # MSLP
            timestep.append(f"{n.m_iMinSeaLevelPressure:>4d}")
            # TY
            timestep.append(f"{self.TY[n.m_eLevelOfDevelopment]:>2s}")
            # RAD
            radindex = None
            for w in n.m_radii:
                if radindex is not None:
                    break
                for q in n.m_radii[w].m_data:
                    if q.m_radius != -1:
                        radindex = w
                        break
            if radindex is not None:
                timestep.append(f"{self.RAD[radindex]:>3s}")
            else:
                timestep.append("{:>3s}".format(""))
            # WINDCODE
            timestep.append(f"{self.WINDCODES[n.m_eWindRadiusCode]:>3s}")
            radii = [0, 0, 0, 0]  # Getting the values for the radii
            if radindex is not None:
                radii = [radius.m_radius for radius in n.m_radii[radindex].m_data]
            # RAD1
            timestep.append(f"{radii[0]:>4d}")
            # RAD2
            timestep.append(f"{radii[1]:>4d}")
            # RAD3
            timestep.append(f"{radii[2]:>4d}")
            # RAD4
            timestep.append(f"{radii[3]:>4d}")
            # RADP
            timestep.append(f"{n.m_iPressure:>4d}")
            # RRP
            timestep.append(f"{n.m_iPressureRadius:>4d}")
            # MRD
            timestep.append(f"{n.m_iMaxWindRadius:>3d}")
            # GUSTS
            timestep.append(f"{n.m_iGusts:>3d}")
            # EYE
            timestep.append(f"{n.m_iEyeDiameter:>3d}")
            # SUBREGION
            timestep.append(f"{self.SUBREGIONS[sub_reg]:>3s}")
            # MAXSEAS
            timestep.append(f"{n.m_iMaxSeas:>3d}")
            # INITIALS
            timestep.append(f"{n.m_sInitials:>3.3s}")
            # DIR
            timestep.append(f"{n.m_iStormDirection:>3d}")
            # SPEED
            timestep.append(f"{n.m_iStormSpeed:>3d}")
            # STORMNAME
            # This column needs to be 12 characters wide for ADCIRC (joined with ", ")
            timestep.append(f"{n.m_sStormName:>11s}")
            if nws == 8:
                # DEPTH
                # SEAS
                timestep.append(f"{self.DEPTHS[n.m_eDepth]:>s}")
                # SEASCODE
                timestep.append(f"{n.m_iWaveHeight:>2d}")
                # SEAS1
                timestep.append(f"{self.WINDCODES[n.m_eSeasRadiusCode]:>s}")
                # SEAS2
                timestep.append(f"{n.m_iSeas1:>3d}")
                # SEAS3
                timestep.append(f"{n.m_iSeas2:>3d}")
                # SEAS4
                timestep.append(f"{n.m_iSeas3:>3d}")
                timestep.append(f"{n.m_iSeas4:>3d}")
                self.timesteps.append(timestep)
            else:
                isos = {}
                for radii in n.m_radii:
                    if not n.m_radii[radii].m_defined:
                        continue  # Only write lines for defined isotach (may have no radii enabled though).

                    quads = [0 for _ in range(4)]
                    quad_maxs = [0.0 for _ in range(4)]
                    intensity_radii = [0 for _ in range(4)]
                    if nws == 20:
                        hollands = [0.0 for _ in range(4)]  # only for NWS=20
                        vmaxs = [0.0 for _ in range(4)]  # only for NWS=20
                    for j in range(4):
                        if n.m_radii[radii].m_data[j].m_on:
                            quads[j] = 1
                        intensity_radii[j] = n.m_radii[radii].m_data[j].m_radius
                        # Write max value (and Holland B if NWS=20) even if disabled. Will be ignored by ADCIRC but
                        # preserves any data defined in SMS. This allows us to read it on model-native import.
                        quad_maxs[j] = n.m_radii[radii].m_data[j].m_rMax
                        if nws == 20:  # add quadrant varying HollandB and VMax values to the line
                            hollands[j] = n.m_radii[radii].m_data[j].m_hollandB
                            vmaxs[j] = n.m_radii[radii].m_data[j].m_vMax
                        if nws == 20:
                            isos[radii] = quads, quad_maxs, self.RAD[radii], intensity_radii, hollands, vmaxs
                        else:
                            isos[radii] = quads, quad_maxs, self.RAD[radii], intensity_radii
                for iso in isos:  # Output a line for each isotach defined for this node
                    new_line = copy.deepcopy(timestep)
                    # Replace radii and wind intensity for this line. Was initialized with values for the first
                    # enabled isotach and radius found, which may not be this one.
                    new_line[intensity_col] = f'{isos[iso][2]:>3s}'
                    new_line[rad1_col] = f'{isos[iso][3][0]:>4d}'
                    new_line[rad2_col] = f'{isos[iso][3][1]:>4d}'
                    new_line[rad3_col] = f'{isos[iso][3][2]:>4d}'
                    new_line[rad4_col] = f'{isos[iso][3][3]:>4d}'
                    # For some reason the following section is fixed-width formatting for ADCIRC.
                    # BEGIN FIXED-WIDTH
                    new_line.append(f'{n.m_id:>3d}')
                    new_line.append(f'   {len(isos)}')
                    for j in range(4):  # Add the quadrant enabled/disabled flags
                        new_line.append(f"{isos[iso][0][j]}")
                    # END FIXED-WIDTH
                    for j in range(4):  # Add the quadrant maxs
                        new_line.append("%.6f" % isos[iso][1][j])
                    new_line.append("%.6f" % n.m_hollandB)
                    if nws == 20:  # add the NWS=20 specific values
                        for j in range(4):  # Add the quadrant HollandB values
                            new_line.append("%.6f" % isos[iso][4][j])
                        for j in range(4):  # Add the quadrant Vmaxs
                            new_line.append("%.6f" % isos[iso][5][j])
                    self.timesteps.append(new_line)

    def load_data(self):
        """Build up lines of the ATCF file from the wind coverage dump."""
        # Get storm track node locations as lat,lon coordinates
        XmLog().instance.info('Loading wind track coverage node locations and attributes...')
        points = self._xms_data['wind_cov'].m_cov.get_points(FilterLocation.PT_LOC_ALL)
        self.coords = [self._get_lat_lon_coords(pt.x, pt.y) for pt in points]
        self._load_nodal_atts()

    def write(self, filename):
        """Write the storm track coverage to fort.22 as an ATCF formatted file.

        Args:
            filename (:obj:`str`): Location of the file to write
        """
        XmLog().instance.info('Writing wind track coverage node locations and attributes to fort.22...')
        with open(filename, 'w', 100000000) as f:  # Buffer the in-memory data because these can get big.
            for ts_node in self.timesteps:
                line = ', '.join(ts_node)
                f.write(f'{line}\n')


class NWS12Writer(NWSWriter):
    """Writes an ADCIRC NWS=12 fort.22 file. Mostly file references and a couple parameters."""
    def __init__(self, xms_data):
        """Construct the writer.

        Args:
            xms_data (:obj:`dict`): Dictionary of data retrieved from XMS needed to export the fort.22. Should at least
                have the simulation component data at this point.
                    ::
                        {
                            sim_data: SimData,
                        }
        """
        super().__init__(xms_data)

    def write(self, filename):
        """Writes an ADCIRC NWS=12 fort.22 file. Mostly file references and a couple parameters.

        Args:
            filename (:obj:`str`): Location of the file to write
        """
        # Find out how many sets of .pre/.win files we have. At least one is required, regional is optional.
        sim_data = self._xms_data['sim_data']
        if 'dim_0' in sim_data.nws12_files.sizes:
            num_file_pairs = sim_data.nws12_files.sizes['dim_0']
        else:
            num_file_pairs = sim_data.nws12_files.sizes['index']
        if num_file_pairs < 1:
            if not self.suppress_missing_file_errors:
                report_error(
                    'At least one set of OWI pressure and wind files must be specified when NWS=12. fort22 file was '
                    'not exported.', self.sim_export
                )
            return

        # Copy required basin files to the export directory
        dest_dir = os.path.dirname(filename)
        proj_dir = sim_data.info.attrs['proj_dir']

        pressure_files = sim_data.nws12_files['Pressure File'].data.tolist()
        wind_files = sim_data.nws12_files['Wind File'].data.tolist()

        nwbs = sim_data.wind.attrs['NWBS']
        dwm = sim_data.wind.attrs['DWM']
        XmLog().instance.info('Writing fort.22 file...')
        with open(filename, 'w') as f:
            # Always use the non-hardcoded ADCIRC filename format by specifying the number of sets as a
            # negative number.
            f.write(f"{len(pressure_files) * -1:<40} ! Number of sets of .pre and .win files - NWSET\n")
            f.write(f"{int(nwbs):<40} ! Number of snaps at the beginning where default value is used - NWBS\n")
            f.write(f"{dwm:<40} ! Wind Velocity Multiplier\n")
            for pressure_file, wind_file in zip(pressure_files, wind_files):
                # Reconstruct the filepath. We store them as relative from the project directory once the project
                # has been saved.
                pressure_file = io_util.resolve_relative_path(proj_dir, pressure_file)
                wind_file = io_util.resolve_relative_path(proj_dir, wind_file)
                if not os.path.isfile(pressure_file):
                    if not self.suppress_missing_file_errors:
                        pressure_file = pressure_file if XmEnv.xms_environ_running_tests() != 'TRUE' else \
                            os.path.basename(pressure_file)
                        report_error(f'Non-existent OWI pressure file specified: {pressure_file}', self.sim_export)
                    continue
                if not os.path.isfile(wind_file):
                    if not self.suppress_missing_file_errors:
                        wind_file = wind_file if XmEnv.xms_environ_running_tests() != 'TRUE' else \
                            os.path.basename(wind_file)
                        report_error(f'Non-existent OWI wind file specified: {wind_file}', self.sim_export)
                    continue

                # ADCIRC is dumb. The file references in the fort.22 for NWS=12 (but not NWS=15,16) must be
                # relative to the run location. If we can't reference the file with a relative path, we have
                # to copy it to the export location. This happens if the file is on a separate drive. Not ideal
                # because these are huge files.
                pressure_basename = os.path.basename(pressure_file)
                wind_basename = os.path.basename(wind_file)
                try:
                    pressure_path = io_util.compute_relative_path(dest_dir, pressure_file)
                except Exception:
                    XmLog().instance.info(f'Unable to reference {pressure_file} in fort.22. Copying to {dest_dir}...')
                    io_util.copyfile(pressure_file, os.path.join(dest_dir, pressure_basename))
                    pressure_path = pressure_basename

                try:
                    wind_path = io_util.compute_relative_path(dest_dir, wind_file)
                except Exception:
                    XmLog().instance.info(f'Unable to reference {wind_file} in fort.22. Copying to {dest_dir}...')
                    io_util.copyfile(wind_file, os.path.join(dest_dir, wind_basename))
                    wind_path = wind_basename
                f.write(f'"{pressure_path}", "{wind_path}"\n')


class NWSMetFileWriter(NWSWriter):
    """Writes an ADCIRC NWS=15,16 fort.22 file. Mostly file references and a couple parameters."""
    def __init__(self, xms_data):
        """Construct the writer.

        Args:
            xms_data (:obj:`dict`): Dictionary of data retrieved from XMS needed to export the fort.22. Should at least
                have the simulation component data at this point.
                    ::
                        {
                            sim_data: SimData,
                        }
        """
        super().__init__(xms_data)

    def write(self, filename):
        """Writes an ADCIRC NWS=15,16 fort.22 file.

        Args:
            filename (:obj:`str`): Location of the file to write
        """
        sim_data = self._xms_data['sim_data']
        nws = sim_data.wind.attrs['NWS']
        type_str = "#GFDL" if nws == 16 else "!HWND"
        proj_dir = sim_data.info.attrs['proj_dir']

        with open(filename, 'w') as f:
            f.write(f"{type_str}\n")
            f.write(f"{sim_data.wind.attrs['DWM']:g}\n")  # Wind multiplier
            use_press = False
            if nws == 15:
                wind_relationship = sim_data.wind.attrs['wind_pressure']
                f.write(f"{wind_relationship}\n")  # Wind pressure relationship
                use_press = wind_relationship == "specifiedPc"
                dset = sim_data.nws15_files
            else:
                f.write(f"{sim_data.wind.attrs['max_extrap']:g}\n")  # Maximum extrapolation distance
                dset = sim_data.nws16_files

            num_rows = dset.Hours.data.size
            if nws == 15:
                for i in range(num_rows):
                    XmLog(
                    ).instance.info(f'Writing HWND parameters and file reference for file {i + 1} of {num_rows}...')
                    pressure = dset['Central Pressure'].data[i] if use_press else -1
                    hwnd_filename = io_util.resolve_relative_path(proj_dir, dset['HWND File'].data[i])
                    f.write(
                        f"{dset['Hours'].data[i]} {pressure} {dset['Ramp Multiplier'].data[i]} "
                        f"\"{hwnd_filename}\"\n"
                    )
                    if not os.path.isfile(hwnd_filename) and not self.suppress_missing_file_errors:
                        hwnd_filename = hwnd_filename if XmEnv.xms_environ_running_tests() != 'TRUE' else \
                            os.path.basename(hwnd_filename)
                        report_error(f'Non-existent HWND file specified: {hwnd_filename}', self.sim_export)
            else:  # NWS=16
                for i in range(num_rows):
                    XmLog(
                    ).instance.info(f'Writing GFDL parameters and file reference for file {i + 1} of {num_rows}...')
                    gfdl_filename = io_util.resolve_relative_path(proj_dir, dset['GFDL File'].data[i])
                    f.write(f"{dset['Hours'].data[i]} {dset['Ramp Multiplier'].data[i]} "
                            f"\"{gfdl_filename}\"\n")
                    if not os.path.isfile(gfdl_filename) and not self.suppress_missing_file_errors:
                        gfdl_filename = gfdl_filename if XmEnv.xms_environ_running_tests() != 'TRUE' else \
                            os.path.basename(gfdl_filename)
                        report_error(f'Non-existent GFDL file specified: {gfdl_filename}', self.sim_export)
            if num_rows == 0 and not self.suppress_missing_file_errors:
                file_type = 'HWND' if nws == 15 else 'GFDL'
                report_error(f'No {file_type} wind files specified.', self.sim_export)


class NWSMetFileCopier(NWSWriter):
    """Copies all fort.xxx files in the Model Control for NWS=10,11."""
    def __init__(self, xms_data):
        """Construct the writer.

        Args:
            xms_data (:obj:`dict`): Dictionary of data retrieved from XMS needed to export the fort.22. Should at least
                have the simulation component data at this point.
                    ::
                        {
                            sim_data: SimData,
                        }
        """
        super().__init__(xms_data)

    def write(self, filename):
        """Copies all met files to export location for NWS=10,11.

        Args:
            filename (:obj:`str`): Location of the file to write
        """
        dest_dir = os.path.dirname(filename)
        sim_data = self._xms_data['sim_data']
        nws = sim_data.wind.attrs['NWS']
        met_files = sim_data.nws10_files['AVN File'].data.tolist() if nws == 10 else \
            sim_data.nws11_files['ETA File'].data.tolist()
        file_type = 'AVN' if nws == 10 else 'ETA'
        proj_dir = sim_data.info.attrs['proj_dir']
        for met_file in met_files:
            orig_file = io_util.resolve_relative_path(proj_dir, met_file)
            if os.path.isfile(orig_file):
                io_util.copyfile(orig_file, os.path.join(dest_dir, os.path.basename(orig_file)))
            elif not self.suppress_missing_file_errors:
                report_error(f'Non-existent {file_type} wind file specified: {met_file}', self.sim_export)
        if len(met_files) == 0 and not self.suppress_missing_file_errors:
            report_error(f'No {file_type} wind files specified.', self.sim_export)


def export_wind_for_sim(filename):
    """Entry point for fort.22 exporter when exporting the entire ADCIRC simulation for a run.

    Args:
        filename (:obj:`str`): Path to the fort.22 to write
    """
    writer = Fort22Writer(filename, True, read_fort22_data_json())
    writer.write()
