"""Dialog for viewing the attributes of applied boundary conditions."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, QSortFilterProxyModel, Qt
from PySide2.QtGui import QBrush, QPixmap
from PySide2.QtWidgets import QAbstractItemView, QMessageBox, QPushButton, QStyle, QStyledItemDelegate, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.delegates.check_box_no_text import CheckBoxNoTextDelegate
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.widgets.qx_table_view import QxTableView

# 4. Local modules
from xms.adcirc.data import bc_data as bcd
from xms.adcirc.gui import gui_util
from xms.adcirc.gui.mapped_bc_dlg_ui import Ui_MappedBcDlg
import xms.adcirc.gui.mapped_bc_table_model as bc_model
from xms.adcirc.gui.mapped_bc_viewer_dlg import MappedBCViewerDlg
from xms.adcirc.gui.view_bc_dlg import ViewLeveeDlg, ViewRiverDlg
from xms.adcirc.gui.view_node_ids_dlg import ViewNodeIdsDlg

HIDE_TANGENTIAL_TYPES = [
    bc_model.BC_TYPES[bcd.UNASSIGNED_INDEX],
    bc_model.BC_TYPES[bcd.OCEAN_INDEX],
    bc_model.BC_TYPES[bcd.LEVEE_INDEX],
    bc_model.BC_TYPES[bcd.RADIATION_INDEX],
    bc_model.BC_TYPES[bcd.ZERO_NORMAL_INDEX],
    bc_model.BC_TYPES[bcd.FLOW_AND_RADIATION_INDEX],
]

LEVEE_TYPES = [bc_model.BC_TYPES[bcd.LEVEE_INDEX], bc_model.BC_TYPES[bcd.LEVEE_OUTFLOW_INDEX]]


class ViewBCButtonDelegate(QStyledItemDelegate):
    """Delegate for the view levee button column."""
    def __init__(self, bc_data, mapped_data, ugrid, xd, parent=None):
        """Initializes the class.

        Args:
            bc_data (:obj:`BcData`): BcData object containing the source BC data
            mapped_data (:obj:`MappedBcData`): Data object containing the mapped BC data
            ugrid (UGrid): Ugrid used for mapping
            xd (XmsData): Object for communicating with XMS
            parent (:obj:`QObject`): The parent object.
        """
        super().__init__(parent)
        self.bc_data = bc_data
        self._ugrid = ugrid
        self._xd = xd
        self.mapped_data = mapped_data

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (:obj:`QWidget`): The editor widget.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (:obj:`QPainter`): The painter.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            bc_type = index.model().data(index.model().index(index.row(), bc_model.COL_BC_TYPE))
            if bc_type in LEVEE_TYPES:
                btn.setText('Levee...')
            elif bc_type == bc_model.BC_TYPES[bcd.RIVER_INDEX]:
                btn.setText('River Flow...')
            else:
                btn.setText('N/A')
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())

            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (:obj:`QEvent`): The editor event that was triggered.
            model (:obj:`QAbstractItemModel`): The data model.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                row = index.row()
                bc_type = index.model().data(index.model().index(row, bc_model.COL_BC_TYPE))
                comp_id = self.mapped_data.nodestrings.comp_id[row].data.item()
                if bc_type in LEVEE_TYPES:
                    is_levee_outflow = bc_type == bc_model.BC_TYPES[bcd.LEVEE_OUTFLOW_INDEX]
                    partner_id = self.mapped_data.nodestrings.partner_id[row].data.item()
                    try:
                        df = self.mapped_data.levees.sel(comp_id=comp_id).to_dataframe()
                    except Exception:
                        df = None
                    if (partner_id < 0 and not is_levee_outflow) or df is None:
                        msg = QMessageBox(
                            QMessageBox.Critical, '', 'Error finding levee pair. Levee boundary condition'
                            ' is invalid.', QMessageBox.Ok, self.parent()
                        )
                        msg.exec()
                    else:
                        gui_util.fix_df_column_names_for_gui(df)
                        # Convert nodestring ids to 1-based for GUI.
                        partner_id = partner_id if partner_id < 0 else partner_id + 1  # Leave negative if no partner
                        dlg = ViewLeveeDlg(df, row + 1, partner_id, self._ugrid, self._xd, self.parent())
                        if dlg.exec():
                            # Save levee subset Dataset values to original Dataset on 'Ok'.
                            gui_util.fix_df_column_names_for_io(df)
                            data = df.to_xarray()
                            self.mapped_data.update_levee(comp_id, data)
                elif bc_type == bc_model.BC_TYPES[bcd.RIVER_INDEX]:
                    try:
                        ds = self.mapped_data.river_flows
                        ds_filtered = ds.where(ds['comp_id'] == comp_id, drop=True)
                        node_ids = ds_filtered.coords['Node Id'].values.tolist()
                        node_ids = [node_id + 1 for node_id in node_ids]
                        df = ds_filtered['Flow'].to_dataframe().reset_index()
                        df = df.pivot(index='TS', columns='Node Id', values='Flow')
                        df.columns = [f'Node {id + 1}' for id in df.columns]
                    except Exception:
                        df = None
                    if df is not None:
                        dlg = ViewRiverDlg(df, row + 1, node_ids, self._xd)
                        dlg.exec()
                return True
        return super().editorEvent(event, model, option, index)


class NodeIdButtonDelegate(QStyledItemDelegate):
    """Delegate for the view node ids button column."""
    def __init__(self, data, parent=None):
        """Initializes the class.

        Args:
            data (:obj:`MappedBcData`): MappedBcData object containing the mapped node ids dataset
            parent (:obj:`QObject`): The parent object.
        """
        super().__init__(parent)
        self.data = data

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (:obj:`QWidget`): The editor widget.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (:obj:`QPainter`): The painter.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            btn.setText('Node IDs...')
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())

            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (:obj:`QEvent`): The editor event that was triggered.
            model (:obj:`QAbstractItemModel`): The data model.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                start_idx = self.data.nodestrings.nodes_start_idx[index.row()].data.item()
                node_count = self.data.nodestrings.node_count[index.row()].data.item()
                dlg = ViewNodeIdsDlg(self.data.nodes.id[start_idx:start_idx + node_count], self.parent())
                dlg.exec()
                return True
        return super().editorEvent(event, model, option, index)


class MappedBcTableDisableEditModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states."""
    def __init__(self, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(parent)

    def flags(self, index):
        """Set flags for an item in the model.

        Args:
            index (:obj:`QModelIndex`): The item's model index

        Returns:
            (:obj:`Qt.ItemFlags`): Updated flags for the item
        """
        ret_flags = super().flags(index)
        row = index.row()
        col = index.column()
        # Disable tangential slip toggle if not
        if col == bc_model.COL_TANGENTIAL:
            bc_type = index.model().data(index.model().index(row, bc_model.COL_BC_TYPE))
            bc_option = index.model().data(index.model().index(row, bc_model.COL_BC_OPTION))
            if bc_option == 'Natural' or bc_type in HIDE_TANGENTIAL_TYPES:
                ret_flags = ret_flags & (~Qt.ItemIsEnabled)
        elif col == bc_model.COL_GALERKIN:
            bc_type = index.model().data(index.model().index(row, bc_model.COL_BC_TYPE))
            if bc_type != bc_model.BC_TYPES[bcd.ZERO_NORMAL_INDEX]:
                ret_flags = ret_flags & (~Qt.ItemIsEnabled)
        elif col in [bc_model.COL_NODES, bc_model.COL_LEVEE]:
            ret_flags = ret_flags & (~Qt.ItemIsEditable)
            if col == bc_model.COL_LEVEE:
                bc_type = index.model().data(index.model().index(row, bc_model.COL_BC_TYPE))
                if bc_type not in LEVEE_TYPES:
                    # ret_flags = ret_flags & (~Qt.ItemIsEnabled)
                    pass
        return ret_flags


class MappedBcDlg(XmsDlg):
    """A dialog showing the mapped boundary conditions attributes."""
    def __init__(self, win_cont, bc_data, mapped_data, ugrid, xd):
        """Initializes the dialog.

        Args:
            win_cont (:obj:`QWidget`): Parent dialog
            bc_data (:obj:`BcData`): Dataset containing the boundary conditions parameters
            mapped_data (:obj:`MappedBcData`): Dataset containing the boundary conditions parameters
            ugrid (UGrid): Ugrid used for mapping
            xd (XmsData): Object for communicating with XMS
        """
        super().__init__(win_cont, 'xmsadcirc.gui.mapped_bc_dlg')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:Display_Options'
        self.model = bc_model.MappedBcTableModel(bc_data, mapped_data, self)
        self.enable_model = MappedBcTableDisableEditModel(self)
        self.cbx_view = None
        self._ugrid = ugrid
        self._xd = xd
        self._viewer = MappedBCViewerDlg(parent=self, bc_data=bc_data, mapped_data=mapped_data, ugrid=ugrid,
                                         xd=xd)
        self._viewer.show()
        self.ui = Ui_MappedBcDlg()
        self.ui.setupUi(self)

        # Setup the dialog
        self._setup_ui()

    def _setup_ui(self):
        """Setup widgets in the dialog."""
        # Add the read-only BC table widget
        self.table_view = QxTableView(self)
        self.table_view.setModel(self.model)
        opt_cbx_delegate = QxCbxDelegate(self)
        opt_cbx_delegate.set_strings(bc_model.BC_OPTIONS)
        self.table_view.setItemDelegateForColumn(bc_model.COL_BC_OPTION, opt_cbx_delegate)
        tog_delegate = CheckBoxNoTextDelegate(self)
        self.table_view.setItemDelegateForColumn(bc_model.COL_TANGENTIAL, tog_delegate)
        self.table_view.setItemDelegateForColumn(bc_model.COL_GALERKIN, tog_delegate)
        view_nodes_delegate = NodeIdButtonDelegate(self.model.mapped_data, self)
        self.table_view.setItemDelegateForColumn(bc_model.COL_NODES, view_nodes_delegate)
        view_levee_delegate = ViewBCButtonDelegate(self.model.bc_data, self.model.mapped_data, self._ugrid,
                                                   self._xd, self)
        self.table_view.setItemDelegateForColumn(bc_model.COL_LEVEE, view_levee_delegate)
        self.table_view.filter_model = self.enable_model
        self.table_view.filter_model.setSourceModel(self.model)
        self.table_view.setModel(self.table_view.filter_model)
        self.table_view.setEditTriggers(QAbstractItemView.AllEditTriggers)
        # Resizing to contents is extremely slow
        # self.table_view.horizontalHeader().setSectionResizeMode(bc_model.COL_TANGENTIAL, QHeaderView.ResizeToContents)
        # self.table_view.horizontalHeader().setSectionResizeMode(bc_model.COL_GALERKIN, QHeaderView.ResizeToContents)
        self.table_view.resize_height_to_contents()
        layout = QVBoxLayout()
        layout.addWidget(self.table_view)
        self.ui.grp_view_bc.setLayout(layout)

        self.table_view.selectionModel().selectionChanged.connect(self._changed_sel_bc)
        self._changed_sel_bc(None)

    def _changed_sel_bc(self, s):
        """Change the selected bc when the user changes the selected row.

        Args:
            s (:obj:`list`): The newly selected index. Should only ever be size 0 or 1
        """
        return self._viewer.update_viewer(s[0].indexes()[0].row() if s and s[0].isValid() else None)
