"""LeveeCrestElevationsTool class."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
from distutils.dir_util import copy_tree
import math
import os
import uuid

# 2. Third party modules
from geopandas import GeoDataFrame
import numpy as np
import rtree
from shapely.geometry import LineString, MultiPoint, Point
from shapely.ops import nearest_points
import xarray as xr

# 3. Aquaveo modules
from xms.components.display.display_options_io import write_display_option_ids
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.core.filesystem import filesystem
from xms.data_objects.parameters import Component, Coverage
from xms.guipy.data.target_type import TargetType
from xms.tool_core import ALLOW_ONLY_COVERAGE_TYPE, ALLOW_ONLY_MODEL_NAME, IoDirection
from xms.tool_core.coverage_writer import CoverageWriter

# 4. Local modules
from xms.adcirc.components.bc_component import BcComponent
from xms.adcirc.components.bc_component_display import BC_JSON, BC_POINT_ID_FILE, BC_POINT_JSON, BcComponentDisplay
from xms.adcirc.data import bc_data as bcd
from xms.adcirc.data.bc_data_manager import BcDataManager
from xms.adcirc.mapping.mapping_util import buffer_segment, levee_snap_to_segments
from xms.adcirc.tools import levee_check_tool_consts as const
from xms.adcirc.tools.levee_check_tool_base import LeveeCheckToolBase

msg_partial_intersect = 'Levee node pair segments defined between the following mesh nodes does not intersect ' \
    'with the check geometry. \nThe crest elevations for these node pairs will not be checked/adjusted ' \
    'in the output coverage.'


class LeveeCrestElevationsTool(LeveeCheckToolBase):
    """Tool to check and adjust levee pair crest elevations against a check geometry."""
    ARG_INPUT_CHECK_COVERAGE = 2
    ARG_INPUT_TAUZ = 3
    ARG_INPUT_USE_NEAREST = 4
    ARG_OUTPUT_COVERAGE = 5
    UNITS_SANITY_CHECK_MULTIPLIER = 3.0  # If check_elevation > 3 * crest_elevation, warn about potential unit mismatch
    # If check line extends beyond levee 1.5x distance to from edge to ajacent node pair segment, warn about partially
    # unused check line
    PARTIALLY_UNUSED_CHECK_LINE_MULIPLIER = 1.5

    def __init__(self):
        """Initializes the class."""
        super().__init__(const.TOOL_TYPE_CREST_ELEVATION)
        self._rtree = rtree.index.Index()
        self._check_lines = []  # [LineString] - Shapely LineStrings representing the arcs of the check coverage
        # numpy arrays that are parallel with self._check_lines
        self._unused_check_lines = None  # np.ndarray([bool]) - Mask that is False if we used the check line
        self._check_line_feature_ids = None  # np.ndarray([int]) - Feature arc ids of the check lines
        # Extents of the check geometry for trivial rejection
        self._minx = None
        self._miny = None
        self._maxx = None
        self._maxy = None
        self._use_nearest = True
        self._units_error_reported = False

        # xarray data_vars and coords we will fill with levees we update
        self._updated_vars = {
            'Parametric __new_line__ Length': ('comp_id', []),
            'Parametric __new_line__ Length 2': ('comp_id', []),
            'Zcrest (m)': ('comp_id', []),
            'Subcritical __new_line__ Flow Coef': ('comp_id', []),
            'Supercritical __new_line__ Flow Coef': ('comp_id', []),
        }
        self._updated_coords = {'comp_id': []}
        self._drop_comp_ids = []  # comp_ids of levees we updated and need to drop old rows for
        self._results.update({  # Plot levee crest elevation vs. check elevation in results
            const.RESULTS_CHECK_ELEVATION_COL_NAME: [],  # Levee node pair segment check intersection elevations
            'Crest Elevation Adjusted': [],  # Adjusted crest elevation defined on the levee
            'Crest Elevation Original': [],  # Original crest elevation defined on the levee
        })
        self._reset_candidate_levee()  # Initialize the temp storage

        self.new_comp = None  # Just store this so I can check in tests

    def _get_inputs(self, arguments):
        """Get the inputs to the tool from XMS.

        Args:
            arguments (:obj:`list`): The tool arguments.

        Returns:
            (:obj:`tuple(xarray.Dataset, numpy.array)`): The levee dataset and the levee comp_ids
        """
        levee_data, levee_comp_ids = super()._get_inputs(arguments)
        if levee_comp_ids:  # If no levees to check, don't bother with constructing the rtree
            self._threshold = arguments[self.ARG_INPUT_TAUZ].value
            check_cov = self.get_input_coverage(arguments[self.ARG_INPUT_CHECK_COVERAGE].value)
            self._build_rtree(check_cov)
            self._use_nearest = arguments[self.ARG_INPUT_USE_NEAREST].value
        return levee_data, levee_comp_ids

    def _build_rtree(self, cov: GeoDataFrame):
        """Build the rtree for intersection operations.

        Args:
            cov (GeoDataFrame): The GeoDataFrame check coverage
        """
        self.logger.info('Building rtree for intersection operations...')
        feature_ids = []
        arcs = cov[cov['geometry_types'] == 'Arc']
        for idx, arc in enumerate(arcs.itertuples()):
            self._check_lines.append(arc.geometry)
            feature_ids.append(arc.id)
            self._rtree.insert(idx, self._check_lines[-1].bounds)
        self._minx = self._rtree.bounds[0]
        self._miny = self._rtree.bounds[1]
        self._maxx = self._rtree.bounds[2]
        self._maxy = self._rtree.bounds[3]
        self._unused_check_lines = np.full(len(self._check_lines), True)
        self._check_line_feature_ids = np.array(feature_ids)

    def _same_elevations(self, intersections):
        """Check if all the points have the same elevation.

        Args:
            intersections (:obj:`list[Point]`): The shapely points to check.

        Returns:
            (:obj:`bool`): True if all the points have the same elevation within the user specified threshold.
        """
        for i in range(1, len(intersections)):
            if not math.isclose(intersections[i].z, intersections[i - 1].z, abs_tol=self._threshold):
                return False
        return True

    def _check_intersection_validity(
        self, levee_comp_id, intersections, intersection_index, node1_id, node2_id, all_intersections
    ):
        """Check that there is one and only one intersection between check geometry and a segment between node pairs.

        Args:
            levee_comp_id (:obj:`int`): Component id of the levee
            intersections (:obj:`list`): The shapely points of the intersections of the segment between the node
                pairs and the check geometry
            intersection_index (:obj:`int`): Index of the shapely LineString (in self._check_lines) where the
                intersection occurred. If zero or more than one intersection, will report an error.
            node1_id (:obj:`int`): 1-base mesh id of the first node in the levee pair
            node2_id (:obj:`int`): 1-base mesh id of the second node in the levee pair
            all_intersections (:obj:`list`): Out variable that the point of intersection will be appended to if valid
        """
        if not intersections:  # Segment between this node pair does not intersect check geometry
            if msg_partial_intersect not in self._error_dict.keys():
                self._error_dict[msg_partial_intersect] = []
                self._error_levels[msg_partial_intersect] = 'Error'
            self._error_dict[msg_partial_intersect].append((node1_id, node2_id))
            intersections[:] = [None]
            all_intersections.append(None)  # Append None to indicate no valid intersection at this node pair
            self._add_levee_issue(levee_comp_id, const.LEVEE_ISSUE_PARTIAL_INTERSECT)
        elif len(intersections) > 1:  # More than one intersection with this node pair's segment and check geometry
            if self._same_elevations(intersections):  # Suppress error if all intersections have the same elevation
                msg = 'Levee node pair segments defined between the following mesh nodes intersects the ' \
                    'check geometry multiple times \nbut all intersection elevations match within the error ' \
                    'threshold. Crest elevation for these node pairs will be checked against \nthe first found ' \
                    'intersection.'
                if msg not in self._error_dict.keys():
                    self._error_dict[msg] = []
                    self._error_levels[msg] = 'Warning'
                self._error_dict[msg].append((node1_id, node2_id))
                intersections[:] = [intersections[0]]  # Pretend we only found the one intersection
            else:  # Multiple intersections that do not have the same elevations
                msg = 'Levee node pair segments defined between the following mesh nodes intersects the ' \
                    'check geometry multiple times \nat different elevations. The crest elevations for these ' \
                    'node pairs will not be checked or adjusted in the output coverage.'
                if msg not in self._error_dict.keys():
                    self._error_dict[msg] = []
                    self._error_levels[msg] = 'Error'
                self._error_dict[msg].append((node1_id, node2_id))
                intersections[:] = [None]
                all_intersections.append(None)  # Append None to indicate no valid intersection at this node pair
                self._add_levee_issue(levee_comp_id, const.LEVEE_ISSUE_MULTI_INTERSECT)

        # If 1 valid intersection found between node pair's segment and check geometry, it's a candidate for adjustment.
        if self._valid_intersection(intersections):
            # The shapely intersection operations are in 2D space, which is what we want but it ends up averaging the
            # elevation of the mesh ground node with the check intersection, which is not what we want.
            line = self._check_lines[intersection_index]
            if len(list(set(line.coords))) != len(list(line.coords)):
                # do stuff
                seen = set()
                new_coords = []
                for pt in line.coords:
                    if pt not in seen:
                        seen.add(pt)
                        new_coords.append(pt)
                line = LineString(new_coords)

            if self._use_nearest:
                # Convert LineString to a MultiPoint collection to find the nearest vertex to the intersection.
                multi_point = MultiPoint(line.coords)
                nearest_point = nearest_points(multi_point, intersections[0])[0]
                # Even though nearest_point is an existing point on the check line, the nearest_point() operation
                # drops the Z coordinate, so we have to reproject nearest_point along the line to get its elevation.
                # debugging
                nearest_point_z = line.interpolate(line.project(nearest_point))
                all_intersections.append(Point(intersections[0].x, intersections[0].y, nearest_point_z.z))
            else:
                all_intersections.append(line.interpolate(line.project(intersections[0])))
            self._unused_check_lines[intersection_index] = False  # Flag this check line as used

    def _valid_intersection(self, intersections):
        """Check if rtree intersection is valid for a levee node pair segment.

        Args:
            intersections (:obj:`list[Point]`): List of the shapely Point intersections between the segment and
                check geometry

        Returns:
            (:obj:`bool`): True if there is exactly one intersection and it is not None
        """
        return len(intersections) == 1 and intersections[0] is not None

    def _intersect_node_pairs(self, levee_comp_id, snap_data):
        """Intersect segments created between each levee's node pairs with the check geometry.

        Args:
            levee_comp_id (:obj:`int`): Component id of the levee pair
            snap_data (:obj:`dict`): The stored snap data for the levee from the first pass

        Returns:
            (:obj:`list[Point]`): The intersection points. None for all segments without a valid intersection.

            Returns empty list if levee was trivially rejected and should be excluded from the results.
        """
        all_intersections = []
        first_intersection_index = None  # rtree intersection of the first levee node pair segment
        last_intersection_index = None  # rtree intersection of the last levee node pair segment
        snap1 = snap_data['snap1']
        snap2 = snap_data['snap2']
        bounds, segments = levee_snap_to_segments(snap1, snap2)  # Create segments between each of the levee node pairs
        if not self._selected_arcs:
            # If no arcs are selected, trivially reject and ignore levee's outside the check geometry bounds.
            # bounds = [min_x, min_y, max_x, max_y]
            if bounds[2] < self._minx or bounds[3] < self._miny or bounds[0] > self._maxx or bounds[1] > self._maxy:
                return []

        last_segment_idx = len(segments) - 1
        for idx, segment in enumerate(segments):  # Loop through each node pair on the levee
            # Intersect the segment formed between each of the levee node pairs with the check geometry.
            node1_id = snap1['id'][idx] + 1
            node2_id = snap2['id'][idx] + 1
            intersection_index, intersections = self._intersect_segment(
                segment, False, node1_id, node2_id, levee_comp_id
            )

            # If the first segment intersects at all do not extrapolate the second segment's intersection elevation. If
            # the first segment has an invalid multi-intersection we want to report an error instead of extrapolating.
            if idx == 0 and intersections:
                first_intersection_index = intersection_index
            # If check coverage does not intersect the final node pair segment at all but it does intersect the second
            # to last, extrapolate the intersection elevation to the last node pair. If the last segment has an invalid
            # multi-intersection we want to report an error instead of extrapolating.
            if idx == last_segment_idx and not intersections and all_intersections[-1] is not None:
                all_intersections.append(all_intersections[-1])
                continue

            # Ensure there is one and only one intersection between the check geometry and the segment formed by this
            # node pair.
            self._check_intersection_validity(
                levee_comp_id, intersections, intersection_index, node1_id, node2_id, all_intersections
            )
            if idx == 1:
                # If the check coverage does not intersect the first node pair segment of the levee but it does
                # intersect with the second, extrapolate the intersection elevation to the first node pair.
                if first_intersection_index is None and self._valid_intersection(intersections):
                    # Clear warning reported for first node pair since we have an elevation now
                    self._remove_levee_issue(levee_comp_id, const.LEVEE_ISSUE_PARTIAL_INTERSECT)
                    self._error_dict[msg_partial_intersect].pop()
                    all_intersections[0] = all_intersections[1]
                # If we had an invalid multi-intersection with the first segment and only set first_intersection_index
                # so we would not extrapolate elevation from the second intersection, clear it now to skip later check.
                elif all_intersections[0] is None:
                    first_intersection_index = None
            elif idx == last_segment_idx and self._valid_intersection(intersections):
                # If the last segment has a valid intersection, update rtree index for later check.
                last_intersection_index = intersection_index

        # Check for check lines that intersect with either the first or last levee node pair segment but extend for
        # a distance greater than 1.5 the distance to the first/last node pairs adjacent node pair.
        self._warn_about_partially_unused_check_lines(
            first_intersection_index, all_intersections[0], all_intersections[1], snap1['id'][0], snap2['id'][0],
            segments[1], levee_comp_id
        )
        self._warn_about_partially_unused_check_lines(
            last_intersection_index, all_intersections[-1], all_intersections[-2], snap1['id'][-1], snap2['id'][-1],
            segments[-2], levee_comp_id
        )
        return all_intersections

    def _warn_about_partially_unused_check_lines(
        self, intersection_index, intersection, adjacent_intersection, node1_idx, node2_idx, adjacent_segment,
        levee_comp_id
    ):
        """Report a warning if the checkline that intersects the first or last segment but extends beyond the levee.

        Notes:
            If the distance from the first/last intersection to the end of the checkline is greater than 1.5 times the
            distance between the first/last intersection and the adjacent intersection, we log a warning.

        Args:
            intersection_index (:obj:`int`): rtree index of the edge intersection
            intersection (:obj:`Point`): The shapely point of the levee edge intersection
            adjacent_intersection (:obj:`Point`): The shapely point of the intersection adjacent to the levee edge
            node1_idx (:obj:`int`): 0-base node id of the first node in the levee edge node pair
            node2_idx (:obj:`int`): 0-base node id of the second node in the levee edge node pair
            adjacent_segment (:obj:`LineString`): The between the levee node pair adjacent to the edge pair. Need this
                so we can compute the midpoint if the adjacent levee pair did not have a valid intersection.
            levee_comp_id (:obj:`int`): Component id of the levee pair
        """
        if intersection_index is not None:
            line = self._check_lines[intersection_index]
            # Get the distance from the start of the line to the first/last levee intersection
            start_to_intersect = line.project(intersection)
            # If there wasn't a valid intersection at the adjacent node pair, just use its segment midpoint.
            if adjacent_intersection is None:
                mid_x = (adjacent_segment.coords[0][0] + adjacent_segment.coords[-1][0]) / 2
                mid_y = (adjacent_segment.coords[0][1] + adjacent_segment.coords[-1][1]) / 2
                adjacent_intersection = Point(mid_x, mid_y, 0.0)
            # Get the distance from the start of the line to the adjacent levee intersection
            start_to_adjacent_intersect = line.project(adjacent_intersection)
            # Check if we are moving in the direction of the line's end point at the levee intersection.
            if start_to_intersect > start_to_adjacent_intersect:
                start_to_intersect = line.length - start_to_intersect  # Compute distance from line end to intersection
            # Compute distance between first/last intersection and its adjacent intersection. Can't take the difference
            # of start_to_intersect and start_to_adjacent_intersect because the adjacent intersection might have
            # actually been on a different line
            dist_to_adjacent = math.sqrt(
                (adjacent_intersection.x - intersection.x)**2 + (adjacent_intersection.y - intersection.y)**2
            )
            if start_to_intersect > (dist_to_adjacent * self.PARTIALLY_UNUSED_CHECK_LINE_MULIPLIER):
                check_arc_id = self._check_line_feature_ids[intersection_index]
                msg = f'Partially unused checkline. The check line with feature arc id {check_arc_id} that ' \
                      f'intersects the levee edge \npair with node ids {node1_idx + 1} and {node2_idx + 1} extends ' \
                      'beyond the levee for a distance greater than the distance between the edge \nintersection ' \
                      "and its adjacent levee node pair."
                self._error_stack.append(('Warning', msg))
                self._add_levee_issue(levee_comp_id, const.LEVEE_ISSUE_PARTIAL_USED_CHECK_LINE)

    def _intersect_segment(self, segment, buffered, node1_id, node2_id, levee_comp_id):
        """Intersect a segment with the check geometry, retrying with a buffer if segment does not intersect.

        Args:
            segment (:obj:`LineString`): Shapely LineString of the levee node pair segment
            buffered (:obj:`bool`): True on the second pass to buffer. Should be called with False and will call with
                True internally if needed.
            node1_id (:obj:`int`): Mesh node id of the first node in the levee pair
            node2_id (:obj:`int`): Mesh node id of the second node in the levee pair
            levee_comp_id (:obj:`int`): Component id of the levee pair

        Returns:
            (:obj:`tuple[int,list]`): Index of the intersecting line in self._check_lines, list of the segment's
            intersections with the check geometry. If len != 1, an error unless len > 1 and the Z value of all the
            intersection points are equal within the elevation error tolerance.
        """
        if buffered:  # If this is a retry, buffer the segment on either side by the levee width
            segment = buffer_segment(segment)
        intersections = []
        intersection_index = None  # If zero or more than one intersection, error
        candidate_indices = self._rtree.intersection(segment.bounds)
        for candidate_index in candidate_indices:  # Look for an intersection between segment and candidates
            intersection = self._check_lines[candidate_index].intersection(segment)
            if intersection.is_empty:
                continue  # Segment does not intersect this candidate
            if type(intersection) is Point:  # Intersects this line once, all good if no other intersections
                intersections.append(intersection)
                intersection_index = candidate_index  # Should only ever be one, else error will be reported later
            elif type(intersection) is MultiPoint:  # Multiple intersections with same line, report error later
                intersections.extend([intersection.geoms[i] for i in range(len(intersection.geoms))])
                intersection_index = candidate_index

        if not intersections and not buffered:  # Try again, but this time buffer the segment
            return self._intersect_segment(segment, True, node1_id, node2_id, levee_comp_id)
        else:
            # Report a warning if we found a valid intersection on the second, buffered attempt.
            if buffered and self._valid_intersection(intersections):
                msg = 'Intersection for a levee node pair segment defined between the following mesh nodes was only ' \
                    'found after buffering \nthe segment by the levee width.'
                if msg not in self._error_dict.keys():
                    self._error_dict[msg] = []
                    self._error_levels[msg] = 'Warning'
                self._error_dict[msg].append((node1_id, node2_id))
                self._add_levee_issue(levee_comp_id, const.LEVEE_ISSUE_BUFFER_INTERSECT)
            return intersection_index, intersections

    def _check_levee(self, levee_comp_id, intersections):
        """Map levee coverage data attributes to snapped node locations, updating crest elevation if needed.

        Args:
            levee_comp_id (:obj:`int`): Component id of the levee pair
            intersections (:obj:`list`): The shapely Point intersections of the segments between each node pair and the
                check geometry

        Returns:
            (:obj:`tuple[float,float,list[str]]`): Minimum check elevation at intersections with the levee,
            maximum check elevation at intersections with the levee, list of the adjusted levee crest elevations for
            reporting to the log window,
        """
        # Store mapped levee attributes to update if we adjust
        min_z = float('inf')  # Range of levee crest elevations for report summary
        max_z = float('-inf')
        adjustments = []  # Reports of the adjusted levee pair crest elevations
        new_zcrests = []
        # Unpack the applied data we stored in the first pass of checks
        snap_data = self._snap_data[levee_comp_id]
        t_lens = snap_data['t_length']
        zcrests = snap_data['zcrest']
        node1_idxs = snap_data['snap1']['id']
        node2_idxs = snap_data['snap2']['id']
        check_units = True  # Only warn about possible units mismatch once per levee
        # Map the levee's attribute curves to each node pair
        for t_len, node1_idx, node2_idx, original_elevation, intersection in zip(
            t_lens, node1_idxs, node2_idxs, zcrests, intersections
        ):
            crest_elevation = original_elevation
            # If an intersection was not found for a levee node pair, set the check elevation equal to the original.
            if intersection is None:
                check_elevation = crest_elevation
                plot_check_elevation = np.nan  # Plot as nan (gap in the check elevation curve)
            else:
                check_elevation = intersection.z
                plot_check_elevation = check_elevation
            min_z = min(min_z, check_elevation)  # Update min/max check elevations for summary report
            max_z = max(max_z, check_elevation)
            if check_units:
                check_units = self._units_sanity_check(
                    node1_idx, node2_idx, check_elevation, crest_elevation, levee_comp_id
                )
                if not check_units and not self._units_error_reported:
                    msg = 'Potential units mismatch (ft <-> m) detected. Extreme delta Z values may indicate ' \
                        'inconsistent projections between the check data and ADCIRC inputs.'
                    self.logger.warning(msg)
                    self._units_error_reported = True
            # If delta Z between the intersection and the original crest elevation is outside threshold, adjust.
            if abs(check_elevation - original_elevation) > self._threshold:
                # Node1ID Node2Id CrestElevation CheckElevation
                adjustments.append(
                    f'{node1_idx + 1:<12}{node2_idx + 1:<12}{original_elevation:<29.6f}'
                    f'{check_elevation:<.6f}'
                )
                crest_elevation = check_elevation
            # Store results report data and adjusted crest elevation
            new_zcrests.append(crest_elevation)
            self._store_node_pair_row(t_len, plot_check_elevation, crest_elevation, original_elevation)
        if adjustments:  # If we adjusted the crest elevation, store updated attributes for output coverage data
            self._add_updated_levee_atts(
                levee_comp_id, t_lens, new_zcrests, snap_data['sub_coef'], snap_data['super_coef']
            )
        return min_z, max_z, adjustments

    def _units_sanity_check(self, node1_id, node2_id, check_elevation, crest_elevation, levee_comp_id):
        """Perform a sanity check and warn if check elevation is more than 3x the levee crest elevation.

        Notes:
            We think this might catch a common issue with projection unit mismatches in the check geometry, which most
            likely was originally a shapefile.

        Args:
            node1_id (:obj:`int`): Mesh node id of the first node in the levee pair
            node2_id (:obj:`int`): Mesh node id of the second node in the levee pair
            check_elevation (:obj:`float`): Check geometry's elevation at its intersection with levee pair segment
            crest_elevation (:obj:`float`): Levee pair's crest elevation at its intersection with the check geometry
            levee_comp_id (:obj:`int`): Component ID of the levee to check

        Returns:
            (:obj:`bool`): True if the delta Z was reasonable or we did not check (one or both elevations at sea level),
            False if we warned about a possible units mismatch
        """
        if check_elevation == 0.0 or crest_elevation == 0.0:
            return True  # Only perform this check if both elevations are positive, non-zero values
        check_elevation = abs(check_elevation)
        crest_elevation = abs(crest_elevation)
        # Perform a sanity check to warn the user if delta Z between check and crest is significantly off.
        check_too_high = check_elevation > crest_elevation * self.UNITS_SANITY_CHECK_MULTIPLIER
        check_too_low = crest_elevation > check_elevation * self.UNITS_SANITY_CHECK_MULTIPLIER
        if check_too_high or check_too_low:
            self._add_levee_issue(levee_comp_id, const.LEVEE_ISSUE_UNITS_MISMATCH)
            return False
        return True

    def _add_updated_levee_atts(self, levee_comp_id, lengths, crest_elevations, sub_coefs, super_coefs):
        """Add an updated levee's data rows to variables we will use to build the new coverage component data.

        Args:
            levee_comp_id (:obj:`int`): Component id of the levee pair
            lengths (:obj:`list`): The parametric lengths column for the levee
            crest_elevations (:obj:`list`): The crest elevations column for the levee
            sub_coefs (:obj:`list`): The subcritical coefficient column for the levee
            super_coefs (:obj:`list`): The supercritical coefficient column for the levee
        """
        num_lengths = len(lengths)
        self._updated_vars['Parametric __new_line__ Length'][1].extend(lengths)
        self._updated_vars['Parametric __new_line__ Length 2'][1].extend(lengths)
        self._updated_vars['Zcrest (m)'][1].extend(crest_elevations)
        self._updated_vars['Subcritical __new_line__ Flow Coef'][1].extend(sub_coefs)
        self._updated_vars['Supercritical __new_line__ Flow Coef'][1].extend(super_coefs)
        self._updated_coords['comp_id'].extend([levee_comp_id] * num_lengths)
        self._drop_comp_ids.append(levee_comp_id)

    def _write_component_id_files(self, new_comp, new_comp_folder):
        """Write the component id files for each BC arc type as well as pipe points.

        Args:
            new_comp (:obj:`BcComponent`): The new component
            new_comp_folder (:obj:`str`): Absolute path to the new component's folder

        Returns:
            (:obj:`tuple`): The used arc component ids, the used pipe point component ids
        """
        arc_comp_ids = []
        point_comp_ids = []
        # Create id files for the BC arcs.
        for bc_type in range(bcd.FLOW_AND_RADIATION_INDEX + 1):
            arcs = new_comp.data.arcs.where(new_comp.data.arcs.type == bc_type, drop=True)
            if arcs.sizes['comp_id'] > 0:  # The coverage has arcs of this type
                loc_file = BcComponentDisplay.get_display_id_file(bc_type, new_comp_folder)
                comp_ids = arcs.comp_id.data.astype(int).tolist()  # Needs to be a Python list of int
                write_display_option_ids(loc_file, comp_ids)
                arc_comp_ids.extend(comp_ids)
        # Create the pipe point id file.
        if new_comp.data.pipes.sizes['comp_id'] > 0:
            loc_file = os.path.join(new_comp_folder, BC_POINT_ID_FILE)
            point_comp_ids = new_comp.data.pipes.comp_id.data.tolist()
            write_display_option_ids(loc_file, point_comp_ids)
        return arc_comp_ids, point_comp_ids

    def _initialize_coverage_comp_ids(self, new_comp, new_comp_folder, cov_uuid, arc_comp_ids, point_comp_ids):
        """Initialize the id-based coverage component id maps of the new source BC coverage.

        Args:
            new_comp (:obj:`BcComponent`): The new component
            new_comp_folder (:obj:`str`): Absolute path to the new component's folder
            cov_uuid (:obj:`str`): UUID of the new coverage geometry
            arc_comp_ids (:obj:`list`): The used arc component ids
            point_comp_ids (:obj:`list`): The used point component ids
        """
        # Set the pipe point component ids.
        pt_map = self._bc_comp.comp_to_xms.get(self._bc_comp.cov_uuid, {}).get(TargetType.point, {})
        for point_comp_id in point_comp_ids:
            att_ids = pt_map.get(point_comp_id, [])
            for att_id in att_ids:
                new_comp.update_component_id(TargetType.point, att_id, point_comp_id)
        # Set the BC arc component ids.
        arc_map = self._bc_comp.comp_to_xms.get(self._bc_comp.cov_uuid, {}).get(TargetType.arc, {})
        for arc_comp_id in arc_comp_ids:
            att_ids = arc_map.get(arc_comp_id, [])
            for att_id in att_ids:
                new_comp.update_component_id(TargetType.arc, att_id, arc_comp_id)
        # Add some display lists for SMS to read for the new BC coverage.
        arc_json = os.path.join(new_comp_folder, BC_JSON)
        pt_json = os.path.join(new_comp_folder, BC_POINT_JSON)
        new_comp.display_option_list = [
            XmsDisplayMessage(file=arc_json, edit_uuid=cov_uuid),  # arcs
            XmsDisplayMessage(file=pt_json, edit_uuid=cov_uuid),  # points
        ]

    def _build_bc_component(self, cov_uuid):
        """Build the component for the new BC coverage with updated levee crest elevations.

        Args:
            cov_uuid (:obj:`str`): UUID of the new coverage geometry

        Returns:
            (:obj:`tuple`): The data_objects Component
        """
        self.logger.info('Assigning updated levee attributes to feature coverage...')
        comp_uuid = str(uuid.uuid4())
        new_comp_folder = os.path.join(os.path.dirname(os.path.dirname(self._bc_comp.main_file)), comp_uuid)
        # Copy the original component's folder
        os.makedirs(new_comp_folder)
        copy_tree(os.path.dirname(self._bc_comp.main_file), new_comp_folder)
        # Update the levees
        new_main_file = os.path.join(new_comp_folder, os.path.basename(self._bc_comp.main_file))
        self.new_comp = BcComponent(new_main_file)
        self.new_comp.cov_uuid = cov_uuid
        # Drop old rows of levees who were updated
        mask = np.logical_not(self.new_comp.data.levees.comp_id.isin(self._drop_comp_ids))
        self.new_comp.data.levees = self.new_comp.data.levees.where(mask, drop=True)
        new_levee_atts = xr.Dataset(data_vars=self._updated_vars, coords=self._updated_coords)
        self.new_comp.data.add_levee_atts(new_levee_atts)
        # Set all flags to edited.
        mask = np.logical_not(self.new_comp.data.levee_flags.comp_id.isin(self._drop_comp_ids))
        self.new_comp.data.levee_flags = self.new_comp.data.levee_flags.where(mask, drop=True)
        arr = np.unique(new_levee_atts.coords['comp_id'])
        new_levee_flags = bcd.default_levee_flags(fill_num=len(arr), coords={'comp_id': arr})
        self.new_comp.data.add_levee_flags(new_levee_flags)
        # Update UUIDs, display lists, etc.
        self.new_comp.data.info.attrs['cov_uuid'] = cov_uuid
        self.new_comp.data.arcs['use_elevs'].loc[dict(comp_id=self._drop_comp_ids)] = False
        arc_comp_ids, point_comp_ids = self._write_component_id_files(self.new_comp, new_comp_folder)
        self._initialize_coverage_comp_ids(self.new_comp, new_comp_folder, cov_uuid, arc_comp_ids, point_comp_ids)
        data_manager = BcDataManager(self._bc_comp)
        data_manager.update_display_options_file(new_main_file, new_comp_folder, self.new_comp.data)
        self.new_comp.data.commit()
        # Build the data_objects Component
        do_comp = Component(
            comp_uuid=comp_uuid, main_file=new_main_file, model_name='ADCIRC', unique_name='Bc_Component'
        )
        return do_comp

    def _create_new_coverage(self, cov_name):
        """Create the new ADCIRC BC coverage to send back to SMS.

        Args:
            cov_name (:obj:`str`): Name of the new coverage
        """
        if not self._num_adjusted and not self._num_adjusted_issue:
            self.logger.info(
                'No checked levees violated the crest elevation threshold. No new coverage will be '
                'created.'
            )
            return  # No updated output to send to SMS
        self.logger.info('Creating new Boundary Conditions coverage...')
        # Copy the coverage geometry, except update the elevations. Also change the uuid and name
        cov_uuid = str(uuid.uuid4())
        self._bc_coverage.attrs['uuid'] = cov_uuid
        self._bc_coverage.attrs['name'] = cov_name
        # edit the elevations
        self._set_levee_arc_elevs()

        filename = filesystem.temp_filename(suffix='.h5')
        writer = CoverageWriter(filename)
        writer.write(self._bc_coverage)
        new_cov = Coverage(filename)
        do_comp = self._build_bc_component(cov_uuid)

        comp_data = [
            {
                'component_coverage_ids': [self.new_comp.uuid, self.new_comp.update_ids],
                'display_options': self.new_comp.get_display_options()
            }
        ]

        self.query.add_coverage(
            new_cov,
            model_name='ADCIRC',
            coverage_type='Boundary Conditions',
            components=[do_comp],
            component_keywords=comp_data
        )

    def _set_levee_arc_elevs(self):
        """Create the elevations for the coverage to match the adjusted values.

        Args:
            new_cov (:obj:`Coverage`): New coverage
        """
        self.logger.info('Interpolating adjusted values to arc nodes and vertices...')
        arc_map = self._bc_comp.comp_to_xms.get(self._bc_comp.cov_uuid, {}).get(TargetType.arc, {})

        elevs_dict = dict()
        for arc_comp_id in self._drop_comp_ids:
            att_ids = arc_map.get(arc_comp_id, [])
            p_len, zcrest = self._get_crest_and_length_for_comp(arc_comp_id)

            for att_id in att_ids:
                elevs_dict[att_id] = (zcrest, p_len)

        mod_arcs = list(elevs_dict.keys())
        if len(mod_arcs) == 0:
            return

        arcs = self._bc_coverage[self._bc_coverage['geometry_types'] == 'Arc']
        nodes = self._bc_coverage[self._bc_coverage['geometry_types'] == 'Node']
        for id in mod_arcs:
            # get the info from the coverage
            arc = next(arcs[arcs['id'] == id].itertuples())
            old_ls = arc.geometry
            old_coords = list(old_ls.coords)
            par_dists = [old_ls.project(Point(coord[0], coord[1]), True) for coord in old_coords]

            zcrest, p_len = elevs_dict[id]

            # interpolate the other points
            interp_z = np.interp(par_dists, p_len, zcrest)

            # new linestring creation
            new_coords = [(old_coords[i][0], old_coords[i][1], interp_z[i]) for i in range(len(old_coords))]
            updated_ls = LineString(new_coords)

            # update the geometry
            arcs.loc[arcs["id"] == id, "geometry"] = updated_ls

            # also need to update the nodes
            node = next(nodes[nodes['id'] == arc.start_node].itertuples())
            s_node = node.geometry
            nodes.loc[nodes["id"] == arc.start_node, "geometry"] = Point(s_node.x, s_node.y, interp_z[0])

            node = next(nodes[nodes['id'] == arc.end_node].itertuples())
            e_node = node.geometry
            nodes.loc[nodes["id"] == arc.end_node, "geometry"] = Point(e_node.x, e_node.y, interp_z[-1])

        mask = self._bc_coverage['geometry_types'] == 'Arc'
        self._bc_coverage.loc[mask] = arcs.values
        mask = self._bc_coverage['geometry_types'] == 'Node'
        self._bc_coverage.loc[mask] = nodes.values

    def _get_crest_and_length_for_comp(self, comp_id):
        # Access aligned data arrays
        comp_ids = self._updated_coords['comp_id']
        lengths = self._updated_vars['Parametric __new_line__ Length'][1]
        crests = self._updated_vars['Zcrest (m)'][1]

        # Find indices matching the given comp_id
        indices = [i for i, cid in enumerate(comp_ids) if cid == comp_id]

        # Extract the corresponding values
        lengths_for_comp = [lengths[i] for i in indices]
        crests_for_comp = [crests[i] for i in indices]

        return lengths_for_comp, crests_for_comp

    def _run_second_pass(self):
        """Run the second pass of checks if there were no fatal errors on the first pass."""
        self.logger.info('Intersecting levee node pair segments with check geometry and comparing elevation...')
        for comp_id, snap_data in self._snap_data.items():
            adjustments = []  # Reports of the adjusted levee pair crest elevations
            num_pairs = 0
            min_z = float('inf')  # Range of levee crest elevations for report summary
            max_z = float('-inf')
            arc_ids = snap_data['arc_id']
            if snap_data['snap1'] is not None:  # Don't bother intersecting if error during snap
                # Intersect segments between each node pair with the check geometry
                intersections = self._intersect_node_pairs(comp_id, snap_data)
                num_pairs = len(snap_data['snap1']['id'])
                # First check for empty list, which means we trivially rejected and should exclude the levee from the
                # results. This can only happen when we are checking all levees (no levees were selected when).
                num_real_intersections = len([intx for intx in intersections if intx is not None])
                if num_real_intersections == 0:
                    msg = 'The levees defined by the following feature arcs are outside the bounds of ' \
                          'the check geometry and will be excluded from the results.'
                    if msg not in self._global_error_dict.keys():
                        self._global_error_dict[msg] = []
                        self._global_error_levels[msg] = 'Warning'
                    self._global_error_dict[msg].append(arc_ids)
                    arc_ids = []
                elif num_real_intersections == 1:
                    msg = 'The levees defined by the following feature arcs do not sufficiently ' \
                          'intersect the check geometry and will be excluded from the results.'
                    if msg not in self._global_error_dict.keys():
                        self._global_error_dict[msg] = []
                        self._global_error_levels[msg] = 'Warning'
                    self._global_error_dict[msg].append(arc_ids)
                    arc_ids = []
                elif num_real_intersections == 2 and intersections[0] is not None and intersections[-1] is not None:
                    msg = 'The levees defined by the following feature arcs do not sufficiently ' \
                          'intersect the check geometry and will be excluded from the results.'
                    if msg not in self._global_error_dict.keys():
                        self._global_error_dict[msg] = []
                        self._global_error_levels[msg] = 'Warning'
                    self._global_error_dict[msg].append(arc_ids)
                    arc_ids = []
                else:  # Successfully intersected levee node pair segments with the check geometry. Keep checking.
                    # Get the attribute curve defined for this levee pair and map them to the mesh. Compare the crest
                    # elevation mapped to each node pair with the elevation of the intersection between the check
                    # geometry and the segment between the node pair.
                    min_z, max_z, adjustments = self._check_levee(comp_id, intersections)
            if arc_ids:  # Don't report non-existent or skipped levees
                self._report_levee_check(comp_id, arc_ids[0], arc_ids[1], adjustments, num_pairs, min_z, max_z)
            else:
                self._clear_error_stack(comp_id)
                self._reset_candidate_levee()

    def _warn_about_unused_check_lines(self):
        """Log global warnings for check lines that didn't intersect any levee."""
        unused_ids = self._check_line_feature_ids[self._unused_check_lines].tolist()
        if len(unused_ids) == 0:
            return
        msg = 'Check lines with the following feature arc ids did not intersect with any levee node pairs ' \
              'and did not contribute to the checks/adjustments.\n'
        first = True
        for unused_id in unused_ids:
            if not first:
                msg += ', '
            msg += f'{unused_id}'
            first = False
        self._global_error_stack.append(('Warning', msg))  # Store so we can display in results dialog

    def initial_arguments(self):
        """Get initial arguments for tool - must override.

        Returns:
            (:obj:`list`): A list of the initial tool arguments.
        """
        bc_filter = {ALLOW_ONLY_MODEL_NAME: 'ADCIRC', ALLOW_ONLY_COVERAGE_TYPE: 'Boundary Conditions'}
        arguments = [
            self.coverage_argument(
                name='input_bc_coverage', description='Input ADCIRC Boundary Conditions coverage', filters=bc_filter
            ),
            self.grid_argument(name='domain_grid', description='Domain grid'),
            self.coverage_argument(name='input_check_coverage', description='Input check geometry coverage'),
            self.float_argument(
                name='tauz', description='TauZ (allowable levee elevation error threshold)', value=0.0, min_value=0.0
            ),
            self.bool_argument(name='use_nearest', description='Use nearest check line point', value=True),
            self.coverage_argument(
                name='output_coverage',
                description='Output coverage',
                io_direction=IoDirection.OUTPUT,
                value='Adjusted Levees'
            )
        ]
        return arguments

    def run(self, arguments):
        """Run the tool - must override.

        Args:
            arguments (:obj:`list`): The tool arguments.
        """
        if not self._map_all_levees_for_check(arguments):  # This applies levees to mesh and runs first pass of checks
            return  # Fatal error encountered and logged on the first pass, abort checks
        elif not self._check_lines:
            self.logger.error(
                'No arcs found in the check coverage geometry. Select a check coverage with arcs at the '
                "elevations of the input levees' center lines to run this tool."
            )
            return  # No check lines to intersect with levee pairs, abort checks
        self._run_second_pass()  # Run levee crest elevation checks
        self._warn_about_unused_check_lines()
        self._create_new_coverage(arguments[self.ARG_OUTPUT_COVERAGE].value)
        self._report_results()  # Print summary to log window and write results data for the results dialog
