"""A component for simulation data and commands."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import pandas as pd
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg

# 4. Local modules
from xms.adh.components.adh_component import AdHComponent
from xms.adh.components.coverage_mapper_runner import CoverageMapperRunner
from xms.adh.data.model_control import ModelControl
from xms.adh.data.xms_data import XmsData
from xms.adh.gui.model_control_dialog import ModelControlDialog


class SimComponent(AdHComponent):
    """A hidden Dynamic Model Interface (DMI) component for the AdH model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.data = ModelControl(self.main_file)
        self.tree_commands = [
            ('Model Control...', 'open_model_control'), ('Generate Snap Preview', 'create_snap_preview')
        ]  # [(menu_text, menu_method)...]
        if not os.path.exists(main_file):
            self.data.commit()

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        action = ActionRequest(
            modality='NO_DIALOG',
            method_name='initialize_on_create',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file
        )
        messages = []
        action_requests = [action]
        return messages, action_requests

    def initialize_on_create(self, query, _params):
        """Initialize data when the component is created.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            _params (:obj:`dict'): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        from xms.adh.data.xms_query_data import XmsQueryData
        xms_data = XmsQueryData(query)
        horizontal_units = xms_data.horizontal_units
        if 'FEET' in horizontal_units.upper():
            model_control = ModelControl(self.main_file)
            model_control.update_constants_to_feet()
            model_control.commit()
        return [], []

    def link_event(self, link_dict, lock_state):
        """This will be called when one or more coverages, ugrids, or other components are linked to this component.

        Args:
            link_dict (dict): A dictionary with keys being UUIDs as strings representing the objects being linked into
                this component. The values of this dictionary are a list of strings of the parameter names of the
                "takes" from the XML that this is a part of.
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        messages = []
        actions = []
        for link_uuid, link_xml_params in link_dict.items():
            for xml_param in link_xml_params:
                if xml_param == 'theMesh':
                    self.data.domain_uuid = link_uuid
        self.data.commit()
        return messages, actions

    def unlink_event(self, unlinks, lock_state):
        """This will be called when a coverage, or a ugrid, or another component is unlinked from this component.

        Args:
            unlinks (:obj:`list` of str): A list of UUIDs as strings representing the objects being unlinked.
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        if self.data.domain_uuid in unlinks:
            self.data.domain_uuid = ''
            # Empty hot start dataframe when we unlink the grid
            self.data.hot_starts = pd.DataFrame(columns=self.data.hot_starts.columns)
            self.data.commit()
        return [], []

    def open_model_control(self, query, _params, win_cont):
        """Opens the Model Control dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            _params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        from xms.adh.data.xms_query_data import XmsQueryData
        xms_data = XmsQueryData(query, at_sim=False)
        xms_data.sim_component = self
        xms_data.adh_data.model_control = self.data
        self.validate_hot_starts(xms_data)
        dlg = ModelControlDialog(win_cont, 'Model Control', xms_data)
        if dlg.exec():
            self.data = dlg.data
            self.data.commit()
            self._check_sediment_units(xms_data, win_cont)
        return [], []

    def validate_hot_starts(self, xms_data: XmsData):
        """Validates that the selected hot starts still exist in the project explorer."""
        hot_start_df = self.data.hot_starts
        if hot_start_df.empty:
            return
        # Check if uuids match datasets
        is_valid_mask = hot_start_df['uuid'].apply(lambda uuid: xms_data.get_dataset_from_uuid(uuid) is not None)
        if not is_valid_mask.all():
            # Get names of invalid datasets
            removed_entries_df = hot_start_df[~is_valid_mask]
            removed_names = removed_entries_df['name'].tolist()
            # Warning popup
            message = (
                "The following hot start dataset(s) could not be found and have been removed:"
                + "\n".join(f"- {name}" for name in removed_names)
            )
            message_with_ok(None, message, os.environ.get('XMS_PYTHON_APP_NAME'))
            self.data.hot_starts = hot_start_df[is_valid_mask].copy()

    def create_snap_preview(self, query, _params, win_cont):
        """Creates mapped components to display AdH data on a mesh.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            _params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        note = ''
        worker = CoverageMapperRunner(query)
        error_str = 'Error(s) encountered applying coverages to simulation. Review log output for more details.'
        warning_str = 'Warning(s) encountered applying coverages to simulation. Review log output for more details.'
        display_text = {
            'title': 'AdH Snap Preview',
            'working_prompt': 'Applying coverages to geometry. Please wait...',
            'error_prompt': error_str,
            'warning_prompt': warning_str,
            'success_prompt': 'Successfully created snap preview',
            'note': note,
            'auto_load': 'Close this dialog automatically when exporting is finished.'
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.adh', worker, win_cont)
        feedback_dlg.exec()
        return [], []

    def _check_sediment_units(self, xms_data, win_cont: QWidget) -> None:
        """Check units are correct if using sediment.

        Args:
            xms_data (:obj:`xms.adh.data.xms_data.XmsData`): Object for communicating with SMS.
            win_cont (:obj:`QWidget`): The parent widget.
        """
        if xms_data.sediment_material_coverage is not None:
            # should use MKS units if using sediment
            gravity = self.data.param_control.model_constants.gravity
            if gravity < 9.81 - 0.75 or gravity > 9.81 + 0.75:
                message = ('Must use MKS units when using sediment transport."')
                message_with_ok(win_cont, message, os.environ.get('XMS_PYTHON_APP_NAME'))
