"""A component for weir, flap gate, and sluice gate data and commands."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import pandas as pd
from PySide2.QtGui import QIcon

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, MenuItem
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon

# 4. Local modules
from xms.adh.components.adh_component import AdHComponent
from xms.adh.data.structures_io import StructuresIO
from xms.adh.gui.flap_dialog import FlapGateDialog
from xms.adh.gui.sluice_dialog import SluiceGateDialog
from xms.adh.gui.view_all_dialog import ViewAllDialog
from xms.adh.gui.weir_dialog import WeirDialog


class StructureConceptualComponent(AdHComponent):
    """A hidden Dynamic Model Interface (DMI) component for the AdH model structures coverage.

    Structures, which in this case refer to only weirs, flap gates, and sluice gates (NOT any FR cards), are to be
    kept separate from the boundary conditions coverage.
    """
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        # [(menu_text, menu_method)...]
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('Display Options...', 'open_display_options'),
            ('View All Structures...', 'open_view_all'),
        ]
        self.data = StructuresIO(self.main_file)
        if os.path.exists(self.main_file):
            self.cov_uuid = self.data.info.attrs['cov_uuid']
        # Copy default display options if needed
        comp_dir = os.path.dirname(self.main_file)
        self.arc_disp_opts_file = os.path.join(comp_dir, 'struct_arc_display_options.json')
        self.point_disp_opts_file = os.path.join(comp_dir, 'struct_point_display_options.json')
        if not os.path.exists(self.arc_disp_opts_file):
            # Read the default arc display options, and save ourselves a copy with a randomized UUID.
            arc_categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            arc_default_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                'struct_arc_default_display_options.json'
            )
            json_dict = read_display_options_from_json(arc_default_file)
            arc_categories.from_dict(json_dict)
            write_display_options_to_json(self.arc_disp_opts_file, arc_categories)
            # Save our display list UUID to the main file
            self.data.info.attrs['arc_display_uuid'] = arc_categories.uuid

            # Read the default point display options, and save ourselves a copy with a randomized UUID.
            point_categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            point_default_file = os.path.join(
                os.path.dirname(os.path.dirname(__file__)), 'gui', 'resources', 'default_data',
                'struct_point_default_display_options.json'
            )
            point_json_dict = read_display_options_from_json(point_default_file)
            point_categories.from_dict(point_json_dict)
            write_display_options_to_json(self.point_disp_opts_file, point_categories)
            # Save our display list UUID to the main file
            self.data.info.attrs['point_display_uuid'] = point_categories.uuid
            self.data.commit()

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        action = ActionRequest(
            modality='NO_DIALOG',
            method_name='initialize_display',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file
        )

        messages = []
        action_requests = [action]
        return messages, action_requests

    def initialize_display(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        if not self.cov_uuid:
            self.cov_uuid = query.parent_item_uuid()
            self.data.info.attrs['cov_uuid'] = self.cov_uuid
            self.data.commit()

        # Send the default display options to XMS.
        self.display_option_list.append(XmsDisplayMessage(file=self.arc_disp_opts_file, edit_uuid=self.cov_uuid))
        self.display_option_list.append(XmsDisplayMessage(file=self.point_disp_opts_file, edit_uuid=self.cov_uuid))
        return [], []

    @staticmethod
    def _get_id_string(ids):
        """Gets the ids as strings. Anything that is not just 1 id will have special text.

        Args:
            ids (:obj:`list`): List of int ids.

        Returns:
            (:obj:`tuple`): tuple containing:
                - text (:obj:`str`): Id string.
                - valid (:obj:`bool`): True if only one id.

        """
        valid = False
        if len(ids) == 0:
            text = '<deleted>'
        elif len(ids) == 1:
            text = str(ids[0])
            valid = True
        else:
            text = '<' + ','.join(map(str, ids)) + '>'
        return text, valid

    def add_weir_attributes(self, query, params, win_cont):
        """Opens the Edit weir dialog and saves component data state with new weir on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all

        arc_att_ids, arc_comp_ids = self._get_arc_att_and_comp_ids(params)
        point_att_ids, point_comp_ids = self._get_point_att_and_comp_ids(params)

        dlg = WeirDialog(win_cont, QIcon(get_xms_icon()), 'Weir Attributes')
        up_point = point_att_ids[0]
        down_point = point_att_ids[1]
        up_arc = arc_att_ids[0]
        down_arc = arc_att_ids[1]
        dlg.set_assignment((up_arc, True), (down_arc, True), (up_point, True), (down_point, True))
        dlg.set_options(0.0, 0.0, 0.0)
        dlg.set_name('Weir')
        if dlg.exec():
            down_arc_comp, down_point_comp, up_arc_comp, up_point_comp = self._get_accepted_comp_ids(
                arc_comp_ids, dlg, point_comp_ids
            )
            length, crest, height = dlg.get_options()
            all_weir_ids = self.data.weir_names['ID'].to_list()
            if all_weir_ids:
                new_weir_id = max(all_weir_ids) + 1
            else:
                new_weir_id = 1
            weir_name = dlg.get_name()
            new_row = pd.DataFrame([[new_weir_id, weir_name]], columns=['ID', 'NAME'])
            self.data.weir_names = pd.concat([self.data.weir_names, new_row])
            values = [new_weir_id, up_point_comp, down_point_comp, up_arc_comp, down_arc_comp, length, crest, height]
            df2 = pd.DataFrame(
                [values],
                columns=[
                    'WRS_NUMBER', 'S_UPSTREAM', 'S_DOWNSTREAM', 'WS_UPSTREAM', 'WS_DOWNSTREAM', 'LENGTH', 'CREST_ELEV',
                    'HEIGHT'
                ]
            )
            self.data.bc.weirs = pd.concat([self.data.bc.weirs, df2])
            self._process_new_comp_ids(arc_att_ids, arc_comp_ids, point_att_ids, point_comp_ids)
        return [], []

    def edit_weir(self, query, params, win_cont):
        """Opens the Edit weir dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all
        files_dict = query.load_component_ids(self, arcs=True, points=True, delete_files=False)

        arc_att_ids, arc_comp_ids = self._get_att_and_comp_ids(TargetType.arc)
        point_att_ids, point_comp_ids = self._get_att_and_comp_ids(TargetType.point)
        edit_id = params['edit_id']

        name_result = self.data.weir_names.loc[self.data.weir_names['ID'] == edit_id]
        name = name_result['NAME'].iloc[0]
        weir_result = self.data.bc.weirs.loc[self.data.bc.weirs['WRS_NUMBER'] == edit_id]
        up_point_comp = int(weir_result['S_UPSTREAM'].iloc[0])
        down_point_comp = int(weir_result['S_DOWNSTREAM'].iloc[0])
        up_arc_comp = int(weir_result['WS_UPSTREAM'].iloc[0])
        down_arc_comp = int(weir_result['WS_DOWNSTREAM'].iloc[0])
        length = int(weir_result['LENGTH'].iloc[0])
        crest_elev = int(weir_result['CREST_ELEV'].iloc[0])
        height = int(weir_result['HEIGHT'].iloc[0])

        name_row_index = self.data.weir_names.index[self.data.weir_names['ID'] == edit_id]
        weir_row_index = self.data.bc.weirs.index[self.data.bc.weirs['WRS_NUMBER'] == edit_id]

        down_arc, down_point, up_arc, up_point = self._set_attribute_ids_from_comp_ids(
            arc_att_ids, arc_comp_ids, down_arc_comp, down_point_comp, point_att_ids, point_comp_ids, up_arc_comp,
            up_point_comp
        )

        dlg = WeirDialog(win_cont, QIcon(get_xms_icon()), 'Weir Attributes')
        dlg.set_assignment(
            self._get_id_string(up_arc), self._get_id_string(down_arc), self._get_id_string(up_point),
            self._get_id_string(down_point)
        )
        dlg.set_options(length, crest_elev, height)
        dlg.set_name(name)
        if dlg.exec():
            length, crest_elev, height = dlg.get_options()
            weir_name = dlg.get_name()
            self.data.weir_names.at[name_row_index, 'NAME'] = weir_name
            self.data.bc.weirs.at[weir_row_index, 'LENGTH'] = length
            self.data.bc.weirs.at[weir_row_index, 'CREST_ELEV'] = crest_elev
            self.data.bc.weirs.at[weir_row_index, 'HEIGHT'] = height
            if dlg.get_point_assignment_changed():
                self.data.bc.weirs.at[weir_row_index, 'S_UPSTREAM'] = down_point_comp
                self.data.bc.weirs.at[weir_row_index, 'S_DOWNSTREAM'] = up_point_comp
            if dlg.get_arc_assignment_changed():
                self.data.bc.weirs.at[weir_row_index, 'WS_UPSTREAM'] = down_arc_comp
                self.data.bc.weirs.at[weir_row_index, 'WS_DOWNSTREAM'] = up_arc_comp
            self._write_all_id_files()
            self.data.commit()
        self.remove_id_files(files_dict)
        return [], []

    def add_flap_gate_attributes(self, query, params, win_cont):
        """Opens the Edit flap gate dialog and saves component data state with new flap gate on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all

        arc_att_ids, arc_comp_ids = self._get_arc_att_and_comp_ids(params)
        point_att_ids, point_comp_ids = self._get_point_att_and_comp_ids(params)

        dlg = FlapGateDialog(win_cont, QIcon(get_xms_icon()), 'Flap Gate Attributes')
        up_point = point_att_ids[0]
        down_point = point_att_ids[1]
        up_arc = arc_att_ids[0]
        down_arc = arc_att_ids[1]
        dlg.set_assignment((up_arc, True), (down_arc, True), (up_point, True), (down_point, True))
        dlg.set_options(0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
        dlg.set_name('Flap Gate')
        if dlg.exec():
            down_arc_comp, down_point_comp, up_arc_comp, up_point_comp = self._get_accepted_comp_ids(
                arc_comp_ids, dlg, point_comp_ids
            )
            coeff_a, coeff_b, coeff_c, coeff_d, coeff_e, coeff_f, length = dlg.get_options()
            all_flap_ids = self.data.flap_names['ID'].to_list()
            if all_flap_ids:
                new_flap_id = max(all_flap_ids) + 1
            else:
                new_flap_id = 1
            flap_name = dlg.get_name()
            new_row = pd.DataFrame([[new_flap_id, flap_name]], columns=['ID', 'NAME'])
            self.data.flap_names = pd.concat([self.data.flap_names, new_row])
            values = [
                new_flap_id, 1, up_point_comp, down_point_comp, up_arc_comp, down_arc_comp, coeff_a, coeff_b, coeff_c,
                coeff_d, coeff_e, coeff_f, length
            ]
            df2 = pd.DataFrame(
                [values],
                columns=[
                    'FGT_NUMBER', 'USER', 'S_UPSTREAM', 'S_DOWNSTREAM', 'FS_UPSTREAM', 'FS_DOWNSTREAM', 'COEF_A',
                    'COEF_B', 'COEF_C', 'COEF_D', 'COEF_E', 'COEF_F', 'LENGTH'
                ]
            )
            self.data.bc.flap_gates = pd.concat([self.data.bc.flap_gates, df2])
            self._process_new_comp_ids(arc_att_ids, arc_comp_ids, point_att_ids, point_comp_ids)
        return [], []

    def edit_flap_gate(self, query, params, win_cont):
        """Opens the Edit flap gate dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all
        files_dict = query.load_component_ids(self, arcs=True, points=True)

        arc_att_ids, arc_comp_ids = self._get_att_and_comp_ids(TargetType.arc)
        point_att_ids, point_comp_ids = self._get_att_and_comp_ids(TargetType.point)
        edit_id = params['edit_id']

        name_result = self.data.flap_names.loc[self.data.flap_names['ID'] == edit_id]
        name = name_result['NAME'].iloc[0]
        flap_result = self.data.bc.flap_gates.loc[self.data.bc.flap_gates['FGT_NUMBER'] == edit_id]
        up_point_comp = int(flap_result['S_UPSTREAM'].iloc[0])
        down_point_comp = int(flap_result['S_DOWNSTREAM'].iloc[0])
        up_arc_comp = int(flap_result['FS_UPSTREAM'].iloc[0])
        down_arc_comp = int(flap_result['FS_DOWNSTREAM'].iloc[0])
        coeff_a = int(flap_result['COEF_A'].iloc[0])
        coeff_b = int(flap_result['COEF_B'].iloc[0])
        coeff_c = int(flap_result['COEF_C'].iloc[0])
        coeff_d = int(flap_result['COEF_D'].iloc[0])
        coeff_e = int(flap_result['COEF_E'].iloc[0])
        coeff_f = int(flap_result['COEF_F'].iloc[0])
        length = int(flap_result['LENGTH'].iloc[0])

        name_row_index = self.data.flap_names.index[self.data.flap_names['ID'] == edit_id]
        flap_row_index = self.data.bc.flap_gates.index[self.data.bc.flap_gates['FGT_NUMBER'] == edit_id]

        down_arc, down_point, up_arc, up_point = self._set_attribute_ids_from_comp_ids(
            arc_att_ids, arc_comp_ids, down_arc_comp, down_point_comp, point_att_ids, point_comp_ids, up_arc_comp,
            up_point_comp
        )

        dlg = FlapGateDialog(win_cont, QIcon(get_xms_icon()), 'Flap Gate Attributes')
        dlg.set_assignment(
            self._get_id_string(up_arc), self._get_id_string(down_arc), self._get_id_string(up_point),
            self._get_id_string(down_point)
        )
        dlg.set_options(coeff_a, coeff_b, coeff_c, coeff_d, coeff_e, coeff_f, length)
        dlg.set_name(name)
        if dlg.exec():
            coeff_a, coeff_b, coeff_c, coeff_d, coeff_e, coeff_f, length = dlg.get_options()
            flap_name = dlg.get_name()
            self.data.flap_names.at[name_row_index, 'NAME'] = flap_name
            self.data.bc.flap_gates.at[flap_row_index, 'LENGTH'] = length
            self.data.bc.flap_gates.at[flap_row_index, 'COEF_A'] = coeff_a
            self.data.bc.flap_gates.at[flap_row_index, 'COEF_B'] = coeff_b
            self.data.bc.flap_gates.at[flap_row_index, 'COEF_C'] = coeff_c
            self.data.bc.flap_gates.at[flap_row_index, 'COEF_D'] = coeff_d
            self.data.bc.flap_gates.at[flap_row_index, 'COEF_E'] = coeff_e
            self.data.bc.flap_gates.at[flap_row_index, 'COEF_F'] = coeff_f
            if dlg.get_point_assignment_changed():
                self.data.bc.flap_gates.at[flap_row_index, 'S_UPSTREAM'] = down_point_comp
                self.data.bc.flap_gates.at[flap_row_index, 'S_DOWNSTREAM'] = up_point_comp
            if dlg.get_arc_assignment_changed():
                self.data.bc.flap_gates.at[flap_row_index, 'FS_UPSTREAM'] = down_arc_comp
                self.data.bc.flap_gates.at[flap_row_index, 'FS_DOWNSTREAM'] = up_arc_comp
            self.data.commit()
            self._write_all_id_files()
        self.remove_id_files(files_dict)
        return [], []

    def add_sluice_gate_attributes(self, query, params, win_cont):
        """Opens the Edit sluice gate dialog and saves component data state with new sluice gate on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all

        arc_att_ids, arc_comp_ids = self._get_arc_att_and_comp_ids(params)
        point_att_ids, point_comp_ids = self._get_point_att_and_comp_ids(params)

        dlg = SluiceGateDialog(win_cont, QIcon(get_xms_icon()), 'Sluice Gate Attributes')
        up_point = point_att_ids[0]
        down_point = point_att_ids[1]
        up_arc = arc_att_ids[0]
        down_arc = arc_att_ids[1]
        dlg.set_assignment((up_arc, True), (down_arc, True), (up_point, True), (down_point, True))
        dlg.set_length(0.0)
        dlg.set_name('Sluice Gate')
        if dlg.exec():
            down_arc_comp, down_point_comp, up_arc_comp, up_point_comp = self._get_accepted_comp_ids(
                arc_comp_ids, dlg, point_comp_ids
            )
            length = dlg.get_length()
            all_sluice_ids = self.data.sluice_names['ID'].to_list()
            if all_sluice_ids:
                new_sluice_id = max(all_sluice_ids) + 1
            else:
                new_sluice_id = 1
            sluice_name = dlg.get_name()
            new_row = pd.DataFrame([[new_sluice_id, sluice_name]], columns=['ID', 'NAME'])
            self.data.sluice_names = pd.concat([self.data.sluice_names, new_row])
            if self.data.bc.time_series.keys():
                new_key = max(self.data.bc.time_series.keys()) + 1
            else:
                new_key = 1
            self.data.bc.time_series[new_key] = dlg.get_opening()
            values = [new_sluice_id, up_point_comp, down_point_comp, up_arc_comp, down_arc_comp, length, new_key]
            df2 = pd.DataFrame(
                [values],
                columns=[
                    'SLS_NUMBER', 'S_UPSTREAM', 'S_DOWNSTREAM', 'SS_UPSTREAM', 'SS_DOWNSTREAM', 'LENGTH', 'TS_OPENING'
                ]
            )
            self.data.bc.sluice_gates = pd.concat([self.data.bc.sluice_gates, df2])
            self._process_new_comp_ids(arc_att_ids, arc_comp_ids, point_att_ids, point_comp_ids)
        return [], []

    def edit_sluice_gate(self, query, params, win_cont):
        """Opens the Edit sluice gate dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        params = params[0]  # for some reason it put it in a list, which we do not want at all
        files_dict = query.load_component_ids(self, arcs=True, points=True)

        arc_att_ids, arc_comp_ids = self._get_att_and_comp_ids(TargetType.arc)
        point_att_ids, point_comp_ids = self._get_att_and_comp_ids(TargetType.point)
        edit_id = params['edit_id']

        name_result = self.data.sluice_names.loc[self.data.sluice_names['ID'] == edit_id]
        name = name_result['NAME'].iloc[0]
        sluice_result = self.data.bc.sluice_gates.loc[self.data.bc.sluice_gates['SLS_NUMBER'] == edit_id]
        up_point_comp = int(sluice_result['S_UPSTREAM'].iloc[0])
        down_point_comp = int(sluice_result['S_DOWNSTREAM'].iloc[0])
        up_arc_comp = int(sluice_result['SS_UPSTREAM'].iloc[0])
        down_arc_comp = int(sluice_result['SS_DOWNSTREAM'].iloc[0])
        length = int(sluice_result['LENGTH'].iloc[0])
        series_id = int(sluice_result['TS_OPENING'].iloc[0])

        name_row_index = self.data.sluice_names.index[self.data.sluice_names['ID'] == edit_id]
        sluice_row_index = self.data.bc.sluice_gates.index[self.data.bc.sluice_gates['SLS_NUMBER'] == edit_id]

        down_arc, down_point, up_arc, up_point = self._set_attribute_ids_from_comp_ids(
            arc_att_ids, arc_comp_ids, down_arc_comp, down_point_comp, point_att_ids, point_comp_ids, up_arc_comp,
            up_point_comp
        )

        opening = None
        if series_id and series_id in self.data.bc.time_series:
            opening = self.data.bc.time_series[series_id]

        dlg = SluiceGateDialog(win_cont, QIcon(get_xms_icon()), 'Sluice Gate Attributes')
        dlg.set_assignment(
            self._get_id_string(up_arc), self._get_id_string(down_arc), self._get_id_string(up_point),
            self._get_id_string(down_point)
        )
        dlg.set_length(length)
        dlg.set_opening(opening)
        dlg.set_name(name)
        if dlg.exec():
            length = dlg.get_length()
            self.data.bc.time_series[series_id] = dlg.get_opening()
            sluice_name = dlg.get_name()
            self.data.sluice_names.at[name_row_index, 'NAME'] = sluice_name
            self.data.bc.sluice_gates.at[sluice_row_index, 'LENGTH'] = length
            self.data.bc.sluice_gates.at[sluice_row_index, 'TS_OPENING'] = series_id
            if dlg.get_point_assignment_changed():
                self.data.bc.sluice_gates.at[sluice_row_index, 'S_UPSTREAM'] = down_point_comp
                self.data.bc.sluice_gates.at[sluice_row_index, 'S_DOWNSTREAM'] = up_point_comp
            if dlg.get_arc_assignment_changed():
                self.data.bc.sluice_gates.at[sluice_row_index, 'SS_UPSTREAM'] = down_arc_comp
                self.data.bc.sluice_gates.at[sluice_row_index, 'SS_DOWNSTREAM'] = up_arc_comp
            self._write_all_id_files()
            self.data.commit()
        self.remove_id_files(files_dict)
        return [], []

    def open_display_options(self, query, params, win_cont):
        """Opens the display options dialog for boundary conditions."""
        # Read the arc display options
        arc_categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
        json_dict = read_display_options_from_json(self.arc_disp_opts_file)
        arc_categories.from_dict(json_dict)

        # Read the default point display options, and save ourselves a copy with a randomized UUID.
        point_categories = CategoryDisplayOptionList()
        point_json_dict = read_display_options_from_json(self.point_disp_opts_file)
        point_categories.from_dict(point_json_dict)

        dlg = CategoryDisplayOptionsDialog([arc_categories, point_categories], win_cont)
        dlg.setWindowIcon(QIcon(get_xms_icon()))
        if dlg.exec():
            # write files
            category_lists = dlg.get_category_lists()
            for category_list in category_lists:
                disp_file = self.arc_disp_opts_file
                if category_list.target_type == TargetType.point:
                    disp_file = self.point_disp_opts_file
                write_display_options_to_json(disp_file, category_list)
                self.display_option_list.append(
                    XmsDisplayMessage(file=disp_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
                )
        return [], []

    def open_view_all(self, query, params, win_cont):
        """Opens the "view all structures" dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused for this.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        files_dict = query.load_component_ids(self, arcs=True, points=True)
        id_map = {}
        if self.cov_uuid in self.comp_to_xms:
            id_map = self.comp_to_xms[self.cov_uuid]
        dlg = ViewAllDialog(win_cont, QIcon(get_xms_icon()), 'Structures', self.data, id_map)
        if dlg.exec():
            self._write_all_id_files()
            self.data.commit()
        self.remove_id_files(files_dict)
        return [], []

    def add_menu_items_for_selection(self, menu_list, selection, unpacked_id_files):
        """Adds menu items based on the selected feature objects.

        Args:
            menu_list (:obj:`list` of :obj:`xms.api.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                    that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None"
                    may be added to the list to indicate a separator.
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                    The value of the dictionary is a list of IntegerLiteral ids of the selected feature objects.
            unpacked_id_files (dict): A dictionary with the key being an entity type, and the value being a tuple.
                    The first value of the tuple is a temporary file of XMS ids. The second value of the tuple is
                    a temporary file of component ids.

        Returns:
            menu_list (:obj:`list` of :obj:`xms.api.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
                added to the list to indicate a separator.
        """
        menu_list = super().add_menu_items_for_selection(menu_list, selection, unpacked_id_files)
        point_sel_count = 0
        arc_sel_count = 0
        arc_id_files = None
        point_id_files = None
        arc_comp_ids = []
        point_comp_ids = []
        if 'POINT' in selection:
            point_sel_count = len(selection['POINT'])
            point_id_files = unpacked_id_files['POINT'] if 'POINT' in unpacked_id_files else None
        if 'ARC' in selection:
            arc_sel_count = len(selection['ARC'])
            arc_id_files = unpacked_id_files['ARC'] if 'ARC' in unpacked_id_files else None

        if arc_id_files:
            arc_comp_ids = read_display_option_ids(arc_id_files[1])
        if point_id_files:
            point_comp_ids = read_display_option_ids(point_id_files[1])

        assign = {'point_id_files': point_id_files, 'arc_id_files': arc_id_files, 'selection': selection}

        weir_id_result = self.data.bc.weirs.loc[self.data.bc.weirs['WS_UPSTREAM'].isin(arc_comp_ids) |  # noqa: W504
                                                self.data.bc.weirs['WS_DOWNSTREAM'].isin(arc_comp_ids) |  # noqa: W504
                                                self.data.bc.weirs['S_UPSTREAM'].isin(point_comp_ids) |  # noqa: W504
                                                self.data.bc.weirs['S_DOWNSTREAM'].isin(point_comp_ids)]
        flap_id_result = \
            self.data.bc.flap_gates.loc[self.data.bc.flap_gates['FS_UPSTREAM'].isin(arc_comp_ids) |  # noqa: W504
                                        self.data.bc.flap_gates['FS_DOWNSTREAM'].isin(arc_comp_ids) |  # noqa: W504
                                        self.data.bc.flap_gates['S_UPSTREAM'].isin(point_comp_ids) |  # noqa: W504
                                        self.data.bc.flap_gates['S_DOWNSTREAM'].isin(point_comp_ids), ['FGT_NUMBER']]
        sluice_id_result = \
            self.data.bc.sluice_gates.loc[self.data.bc.sluice_gates['SS_UPSTREAM'].isin(arc_comp_ids) |  # noqa: W504
                                          self.data.bc.sluice_gates['SS_DOWNSTREAM'].isin(arc_comp_ids) |  # noqa: W504
                                          self.data.bc.sluice_gates['S_UPSTREAM'].isin(point_comp_ids) |  # noqa: W504
                                          self.data.bc.sluice_gates['S_DOWNSTREAM'].isin(point_comp_ids),
                                          ['SLS_NUMBER']]

        has_weir = weir_id_result.size != 0
        has_flap = flap_id_result.size != 0
        has_sluice = sluice_id_result.size != 0
        # can create a weir, flap gate, or sluice gate
        is_point_arc_two = point_sel_count == 2 and arc_sel_count == 2
        is_edit = point_sel_count <= 2 and arc_sel_count <= 2

        has_weir_menu = has_weir or is_point_arc_two
        has_flap_menu = has_flap or is_point_arc_two
        has_sluice_menu = has_sluice or is_point_arc_two
        above = len(menu_list) > 0

        if above and has_weir_menu:
            menu_list.append(None)

        self._add_structure_menu_items(
            'Weir', 'WRS_NUMBER', weir_id_result, self.data.weir_names, is_point_arc_two, has_weir, is_edit,
            'edit_weir', 'add_weir_attributes', assign, menu_list
        )

        if has_weir_menu:
            above = True
        if above and has_flap_menu:
            menu_list.append(None)

        self._add_structure_menu_items(
            'Flap Gate', 'FGT_NUMBER', flap_id_result, self.data.flap_names, is_point_arc_two, has_flap, is_edit,
            'edit_flap_gate', 'add_flap_gate_attributes', assign, menu_list
        )

        if has_flap_menu:
            above = True
        if above and has_sluice_menu:
            menu_list.append(None)

        self._add_structure_menu_items(
            'Sluice Gate', 'SLS_NUMBER', sluice_id_result, self.data.sluice_names, is_point_arc_two, has_sluice,
            is_edit, 'edit_sluice_gate', 'add_sluice_gate_attributes', assign, menu_list
        )

        if has_sluice_menu:
            above = True
        return menu_list

    def _add_structure_menu_items(
        self, type_text, id_number, bc_df, name_df, is_point_arc_two, has_structure, is_edit, edit_method, add_method,
        assign, menu_list
    ):
        """Adds structure add and edit items to menu, if appropriate.

        Args:
            type_text (:obj:`str`): The text for the type of structure this is. Visible to the user.
            id_number (:obj:`str`): The label of the id number in the dataframe bc_df. Example 'WRS_NUMBER'
            bc_df (:obj:`pd.DataFrame`): The adhparam boundary conditions dataframe.
            name_df (:obj:`pd.DataFrame`): The name dataframe.
            is_point_arc_two (:obj:`bool`): True if only two points and two arcs selected.
            has_structure (:obj:`bool`): True if a structure of that type exists.
            is_edit (:obj:`bool`): True if edit menus allowed. Only allowed if two or less points and arcs are selected.
            edit_method (:obj:`str`): The name of the method for editing this type of structure.
            add_method (:obj:`str`): The name of the method for adding this type of structure.
            assign (:obj:`dict`): A dictionary of the selection and the id files.
            menu_list (:obj:`list` of :obj:`xms.api.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
                added to the list to indicate a separator.
        """
        if is_edit and has_structure:
            # add an edit menu item per structure
            for row_data in bc_df.itertuples():
                structure_id = int(getattr(row_data, id_number))
                name_result = name_df.loc[name_df['ID'] == structure_id]
                structure_name = name_result.iloc[0].NAME
                menu_text = f'Edit {type_text} {structure_name}'
                edit_dict = {'edit_id': structure_id}
                edit_dict.update(assign)
                self._add_selection_menu_item(menu_text, edit_method, edit_dict, menu_list)
        if is_point_arc_two:
            self._add_selection_menu_item(f'Add {type_text}', add_method, assign, menu_list)

    def _add_selection_menu_item(self, command_text, command_method, param_dict, menu_list):
        """Adds a menu item with stored selection.

        Args:
            command_text (str): The menu item text the user will see.
            command_method (str): The method of this component for the action request.
            param_dict (dict): A dictionary of parameters for the function in the action request.
            menu_list (:obj:`list` of :obj:`xms.api.dmi.MenuItem`): A list of menus and menu items to be shown. Note
                    that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None"
                    may be added to the list to indicate a separator.
        """
        action = ActionRequest(
            modality='MODAL',
            method_name=command_method,
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file,
            parameters=param_dict
        )
        menu_item = MenuItem(text=command_text, action=action)
        menu_list.append(menu_item)

    def _get_att_and_comp_ids(self, target_type):
        """Reads the files referenced in the ActionRequest arguments which hold the XMS attribute ids and component ids.

        Args:
            target_type (:obj:`TargetType`): The type of ids to get.

        Returns:
            A pair of lists, the first list of XMS attribute ids, the second a list of component ids.
        """
        att_ids = []
        comp_ids = []
        for comp_id in self.comp_to_xms[self.cov_uuid][target_type].keys():
            for att_id in self.comp_to_xms[self.cov_uuid][target_type][comp_id]:
                att_ids.append(att_id)
                comp_ids.append(comp_id)
        return att_ids, comp_ids

    @staticmethod
    def _get_arc_att_and_comp_ids(params):
        """Reads the files referenced in the ActionRequest arguments which hold the arc attribute ids and component ids.

        Args:
            params (): Arguments from the ActionRequest, extracted from the list.

        Returns:
            A pair of lists, the first list of XMS attribute ids, the second a list of component ids.
        """
        att_ids = []
        comp_ids = []
        if 'arc_id_files' in params and params['arc_id_files']:
            att_ids = read_display_option_ids(params['arc_id_files'][0])
            comp_ids = read_display_option_ids(params['arc_id_files'][1])
        if 'selection' in params and 'ARC' in params['selection'] and not att_ids:
            att_ids = params['selection']['ARC']
            comp_ids = [-1 for _ in att_ids]
        elif 'selection' in params and 'ARC' in params['selection']:
            sel_att_ids = params['selection']['ARC']
            no_comp_att_ids = set(sel_att_ids) - set(att_ids)
            att_ids.extend(list(no_comp_att_ids))
            comp_ids.extend([0 for _ in no_comp_att_ids])
        return att_ids, comp_ids

    @staticmethod
    def _get_point_att_and_comp_ids(params):
        """
        Reads the files referenced in the ActionRequest arguments which hold the point attribute ids and component ids.

        Args:
            params (): Arguments from the ActionRequest, extracted from the list.

        Returns:
            A pair of lists, the first list of XMS attribute ids, the second a list of component ids.
        """
        att_ids = []
        comp_ids = []
        if 'point_id_files' in params and params['point_id_files']:
            att_ids = read_display_option_ids(params['point_id_files'][0])
            comp_ids = read_display_option_ids(params['point_id_files'][1])
        if 'selection' in params and 'POINT' in params['selection'] and not att_ids:
            att_ids = params['selection']['POINT']
            comp_ids = [-1 for _ in att_ids]
        elif 'selection' in params and 'POINT' in params['selection']:
            sel_att_ids = params['selection']['POINT']
            no_comp_att_ids = set(sel_att_ids) - set(att_ids)
            att_ids.extend(list(no_comp_att_ids))
            comp_ids.extend([0 for _ in no_comp_att_ids])
        return att_ids, comp_ids

    def _get_accepted_comp_ids(self, arc_comp_ids, dlg, point_comp_ids):
        """Gets the new component ids for the roles of the structure.

        If there was no previous component id, it will be created here.

        Args:
            arc_comp_ids (:obj:``): Selected component ids. In same order as attribute ids passed into the dialog.
            dlg (:obj:`QDialog`): A structure dialog.
            point_comp_ids (:obj:``): Selected component ids. In same order as attribute ids passed into the dialog.

        Returns:
            (:obj:`tuple`): tuple containing:
                - down_arc_comp (:obj:`int`): The downstream arc component id.
                - down_point_comp (:obj:`int`): The downstream point component id.
                - up_arc_comp (:obj:`int`): The upstream arc component id.
                - up_point_comp (:obj:`int`): The upstream point component id.
        """
        for idx, comp_id in enumerate(arc_comp_ids):
            if comp_id <= 0:
                arc_comp_ids[idx] = self.data.get_next_comp_id()
        for idx, comp_id in enumerate(point_comp_ids):
            if comp_id <= 0:
                point_comp_ids[idx] = self.data.get_next_comp_id()
        up_point_comp = point_comp_ids[0]
        down_point_comp = point_comp_ids[1]
        up_arc_comp = arc_comp_ids[0]
        down_arc_comp = arc_comp_ids[1]
        if dlg.get_point_assignment_changed():
            up_point_comp, down_point_comp = down_point_comp, up_point_comp
        if dlg.get_arc_assignment_changed():
            up_arc_comp, down_arc_comp = down_arc_comp, up_arc_comp
        return down_arc_comp, down_point_comp, up_arc_comp, up_point_comp

    def _process_new_comp_ids(self, arc_att_ids, arc_comp_ids, point_att_ids, point_comp_ids):
        """Sets the new component ids into the base class so they can be sent back to XMS. Saves the data to disk.

        Args:
            arc_att_ids (:obj:`list` of int): Selected attribute ids. In same order as component ids passed into
            the dialog.
            arc_comp_ids (:obj:`list` of int): Selected component ids. In same order as attribute ids passed into
            the dialog.
            point_att_ids (:obj:`list` of int): Selected attribute ids. In same order as component ids passed into
            the dialog.
            point_comp_ids (:obj:`list` of int): Selected component ids. In same order as attribute ids passed into
            the dialog.
        """
        for att_id, comp_id in zip(arc_att_ids, arc_comp_ids):
            self.update_component_id(TargetType.arc, att_id, comp_id)
        for att_id, comp_id in zip(point_att_ids, point_comp_ids):
            self.update_component_id(TargetType.point, att_id, comp_id)
        self._write_all_id_files()
        self.data.commit()

    @staticmethod
    def _set_attribute_ids_from_comp_ids(
        arc_att_ids, arc_comp_ids, down_arc_comp, down_point_comp, point_att_ids, point_comp_ids, up_arc_comp,
        up_point_comp
    ):
        """Builds lists of attribute ids that match the component id for the given structure role.

        Args:
            arc_att_ids (:obj:`list` of int): Selected attribute ids. In same order as component ids passed into
            the dialog.
            arc_comp_ids (:obj:`list` of int): Selected component ids. In same order as attribute ids passed into
            the dialog.
            down_arc_comp (int): The downstream arc component id.
            down_point_comp (int): The downstream point component id.
            point_att_ids (:obj:`list` of int): Selected attribute ids. In same order as component ids passed into
            the dialog.
            point_comp_ids (:obj:`list` of int): Selected component ids. In same order as attribute ids passed into
            the dialog.
            up_arc_comp (int): The upstream arc component id.
            up_point_comp (int): The upstream point component id.

        Returns:
            (:obj:`tuple`): tuple containing:
                - down_arc (:obj:`list` of int): The downstream arc component ids.
                - down_point (:obj:`list` of int): The downstream point component ids.
                - up_arc (:obj:`list` of int): The upstream arc component ids.
                - up_point (:obj:`list` of int): The upstream point component ids.
        """
        up_arc, down_arc, up_point, down_point = [], [], [], []
        for att_id, comp_id in zip(arc_att_ids, arc_comp_ids):
            if comp_id == up_arc_comp:
                up_arc.append(att_id)
            elif comp_id == down_arc_comp:
                down_arc.append(att_id)
        for att_id, comp_id in zip(point_att_ids, point_comp_ids):
            if comp_id == up_point_comp:
                up_point.append(att_id)
            elif comp_id == down_point_comp:
                down_point.append(att_id)
        return down_arc, down_point, up_arc, up_point

    def _write_all_id_files(self):
        """Overwrites all of the id files with current structure assignments."""
        weir_up_arc, weir_down_arc, weir_up_point, weir_down_point = \
            StructuresIO.get_structure_string_ids(self.data.bc.weirs, 'WS_UPSTREAM', 'WS_DOWNSTREAM')
        flap_up_arc, flap_down_arc, flap_up_point, flap_down_point = \
            StructuresIO.get_structure_string_ids(self.data.bc.flap_gates, 'FS_UPSTREAM', 'FS_DOWNSTREAM')
        sluice_up_arc, sluice_down_arc, sluice_up_point, sluice_down_point = \
            StructuresIO.get_structure_string_ids(self.data.bc.sluice_gates, 'SS_UPSTREAM', 'SS_DOWNSTREAM')

        multiple_point = StructuresIO._clean_structure_sets(
            weir_up_point, weir_down_point, flap_up_point, flap_down_point, sluice_up_point, sluice_down_point
        )
        multiple_arc = StructuresIO._clean_structure_sets(
            weir_up_arc, weir_down_arc, flap_up_arc, flap_down_arc, sluice_up_arc, sluice_down_arc
        )
        write_display_option_ids("struct_point_weir_upstream.display_ids", list(weir_up_point))
        write_display_option_ids("struct_point_weir_downstream.display_ids", list(weir_down_point))
        write_display_option_ids("struct_point_flap_upstream.display_ids", list(flap_up_point))
        write_display_option_ids("struct_point_flap_downstream.display_ids", list(flap_down_point))
        write_display_option_ids("struct_point_sluice_upstream.display_ids", list(sluice_up_point))
        write_display_option_ids("struct_point_sluice_downstream.display_ids", list(sluice_down_point))
        write_display_option_ids("struct_point_multiple.display_ids", list(multiple_point))

        write_display_option_ids("struct_arc_weir_upstream.display_ids", list(weir_up_arc))
        write_display_option_ids("struct_arc_weir_downstream.display_ids", list(weir_down_arc))
        write_display_option_ids("struct_arc_flap_upstream.display_ids", list(flap_up_arc))
        write_display_option_ids("struct_arc_flap_downstream.display_ids", list(flap_down_arc))
        write_display_option_ids("struct_arc_sluice_upstream.display_ids", list(sluice_up_arc))
        write_display_option_ids("struct_arc_sluice_downstream.display_ids", list(sluice_down_arc))
        write_display_option_ids("struct_arc_multiple.display_ids", list(multiple_arc))
        self.display_option_list.append(
            XmsDisplayMessage(file=self.arc_disp_opts_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
        )
        self.display_option_list.append(
            XmsDisplayMessage(file=self.point_disp_opts_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
        )
