"""Read and write AdH data for testing."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy
import pandas as pd
import param
import xarray as xr

# 3. Aquaveo modules
from xms.data_objects._data_objects.parameters import FilterLocation
from xms.data_objects.parameters import Arc, Coverage, Point, Polygon

# 4. Local modules


def coverage_to_dict(coverage: Coverage) -> dict:
    """
    Creates a dictionary representation of the coverage details provided by the `Coverage` object.

    Args:
        coverage: The coverage object.

    Returns:
        A JSON-compatible dictionary.
    """

    def point_to_dict(point: Point) -> dict:
        """Convert a point to a dictionary."""
        return {
            "point id": point.id,
            "coordinates": {"x": point.x, "y": point.y, "z": point.z}
        } if point.id != -1 else None

    def arc_to_dict(arc: Arc) -> dict:
        """Convert an arc to a dictionary."""
        arc_points = arc.get_points(FilterLocation.LOC_ALL)
        points_list = [{"x": point.x, "y": point.y, "z": point.z} for point in arc_points]
        return {"arc id": arc.id, "points": points_list}

    def polygon_to_dict(polygon: Polygon) -> dict:
        """Convert a polygon to a dictionary."""
        poly_points = polygon.get_points(FilterLocation.LOC_ALL)
        points_list = [{"x": point.x, "y": point.y, "z": point.z} for point in poly_points]
        return {"polygon id": polygon.id, "points": points_list}

    data = {}

    # Only process points and arcs if no polygons exist
    polygons = coverage.polygons
    if len(polygons) == 0:
        points = coverage.get_points(FilterLocation.LOC_ALL)
        if len(points) > 0:
            data["points"] = []
            for point in coverage.get_points(FilterLocation.LOC_ALL):
                point_dict = point_to_dict(point)
                if point_dict:
                    data["points"].append(point_dict)
        arcs = coverage.arcs
        if len(arcs) > 0:
            data["arcs"] = []
            for arc in coverage.arcs:
                data["arcs"].append(arc_to_dict(arc))
    else:
        data["polygons"] = []
        for polygon in coverage.polygons:
            data["polygons"].append(polygon_to_dict(polygon))

    return data


def make_json_serializable(data):
    """
    Converts a dictionary's values to JSON serializable formats.

    Args:
        data (dict): The original dictionary.

    Returns:
        dict: A JSON serializable dictionary.
    """

    def serialize_value(value):
        if isinstance(value, numpy.integer):
            return int(value)
        elif isinstance(value, numpy.floating):
            return float(value)
        elif isinstance(value, numpy.bool_):
            return bool(value)
        elif isinstance(value, numpy.ndarray):
            return value.tolist()  # Convert arrays to lists
        elif isinstance(value, pd.DataFrame):
            return value.to_dict(orient="records")  # Serialize DataFrames
        elif isinstance(value, pd.Series):
            return value.tolist()  # Serialize Series as lists
        elif isinstance(value, param.Parameterized):
            return parameterized_to_dict(value)
        elif isinstance(value, dict):
            return make_json_serializable(value)  # Recursively handle nested dictionaries
        else:
            # Ensure everything else is converted if necessary
            return value

    output = {k: serialize_value(v) for k, v in data.items()}

    # Remove keys with None values
    output = {k: v for k, v in output.items() if v is not None}
    return output


def parameterized_to_dict(parameterized_obj: param.Parameterized):
    """
    Converts a `param.Parameterized` object to a dictionary representation.

    Args:
        parameterized_obj (param.Parameterized):
            The `param.Parameterized` object to be converted.

    Returns:
        dict:
            A dictionary representation of the `Parameterized` object, including non-default,
            non-empty, and valid parameter values. Nested objects are also represented
            recursively in dictionary form.
    """
    result = {}
    for param_name, param_value in parameterized_obj.param.objects().items():
        current_value = getattr(parameterized_obj, param_name)
        default_value = param_value.default
        if current_value is None:
            continue
        if param_name == "name":
            continue

        if isinstance(current_value, pd.DataFrame):
            if not current_value.empty and not current_value.equals(default_value):
                result[param_name] = current_value.to_dict(orient="records")

        elif isinstance(current_value, param.Parameterized):
            nested_dict = parameterized_to_dict(current_value)
            if nested_dict:  # Only include if non-empty
                result[param_name] = nested_dict

        elif current_value != default_value:
            result[param_name] = current_value

    return result


def include_dataframe_if_not_empty(dataframe):
    """
    Converts a DataFrame to a dictionary with the "records" orientation if it contains data.

    Args:
        dataframe (pd.DataFrame): The input Pandas DataFrame to be evaluated and converted.

    Returns:
        list[dict] | None:
            - If the DataFrame is not empty, returns a list of dictionaries, where each dictionary
              represents a row in the DataFrame with column names as keys.
            - If the DataFrame is empty, returns `None`.
    """
    return dataframe.to_dict(orient="records") if not dataframe.empty else None


def filtered_info_attrs(attrs: dict) -> dict:
    """
    Filters a dictionary of attributes to exclude specific keys.

    This function removes key-value pairs where:
    - The key is "VERSION".
    - The key ends with "uuid" (case-insensitive).

    Args:
        attrs (dict): A dictionary of attributes to filter.

    Returns:
        dict: A filtered dictionary excluding the specified keys.
    """
    return {
        key: value
        for key, value in attrs.items()
        if key != "VERSION" and not key.lower().endswith("uuid")
    }


def include_if_xarray_not_empty(xarray_obj):
    """
    Checks whether the input xarray object is non-empty and returns it if so.

    Behaviour:
    - For `xarray.DataArray`: it is considered empty if it has no elements (size == 0).
    - For `xarray.Dataset`: it is considered empty if it has no data variables.

    If the xarray object is empty, the function returns `None`.

    Args:
        xarray_obj: The xarray object to be evaluated. This could be either:
            - `xarray.DataArray`
            - `xarray.Dataset`

    Returns:
        xarray_obj: The provided xarray object if it is not empty.
        None: If the xarray object is empty (based on the above conditions).
    """
    if isinstance(xarray_obj, xr.DataArray) and xarray_obj.size != 0:
        return xarray_obj
    return None
