"""This is a dialog for specifying sediment material properties."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import os
import webbrowser

# 2. Third party modules
from adhparam.material_transport_properties import MaterialTransportProperties
from PySide2.QtCore import QDir, QItemSelectionModel, Qt
from PySide2.QtWidgets import (
    QCheckBox, QDialogButtonBox, QFileDialog, QGroupBox, QHBoxLayout, QLabel, QLineEdit, QPushButton, QSizePolicy,
    QSpacerItem, QSplitter, QStackedWidget, QTabWidget, QVBoxLayout, QWidget
)

# 3. Aquaveo modules
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.dialogs.polygon_display_options import PolygonDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.number_corrector import NumberCorrector  # noqa: AQU103
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets.display_option_icon_factory import DisplayOptionIconFactory

# 4. Local modules
from xms.adh.data.sediment_constituents_io import SedimentConstituentsIO
from xms.adh.data.sediment_materials_io import SedimentMaterialsIO
from xms.adh.gui.widgets.bed_layer_table_widget import BedLayerTableWidget
from xms.adh.gui.widgets.consolidation_table_widget import ConsolidationTableWidget
from xms.adh.gui.widgets.sediment_material_table_widget import SedimentMaterialTableWidget
from xms.adh.gui.widgets.transport_constituent_mat_assignment_widget import TransportConstituentMatAssignmentWidget


class SedimentMaterialDialog(XmsDlg):
    """A dialog for assigning sediment material properties."""
    def __init__(
        self, win_cont: QWidget, title: str, materials_data: SedimentMaterialsIO,
        constituents_data: SedimentConstituentsIO
    ):
        """Initializes the class, sets up the ui, and writes the model control values.

        Args:
            win_cont (QWidget): Parent window
            title (str): Window title
            materials_data (SedimentMaterialsIO): Materials object
            constituents_data (SedimentConstituentsIO): The sediment transport constituents data.
        """
        super().__init__(win_cont, 'xms.adh.gui.sediment_material_dialog')
        self.data = copy.deepcopy(materials_data)
        self.constituents = constituents_data
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:ADH_Sediment_Library_Control'

        self.mat_id_to_page = {}
        self.mat_id_to_transport = {}

        self._BED_LAYER_TAB = 0
        self._TRANSPORT_TAB = 1

        self.number_corrector = NumberCorrector(self)
        self.dbl_validator = QxDoubleValidator(parent=self)
        self.dbl_validator.setBottom(0.0)

        self.widgets = dict()

        self.setWindowTitle(title)
        self._setup()
        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())
        self.last_import_export_path = None
        self.deleted_mat_ids = []
        for mat_id in self.data.materials:
            self._add_prop_page(mat_id)

    def _setup(self):
        """Sets up the dialog."""
        # Dialog QVBoxLayout with QTabWidget then QDialogButtonBox
        self.top_layout = QVBoxLayout()
        self.setLayout(self.top_layout)

        label_text = 'To change the selected sediment constituents, go to the "Global Sediment Properties".'
        self.selection_label = QLabel(label_text, self)
        self.selection_label.setFixedHeight(self.selection_label.fontMetrics().height())
        self.top_layout.addWidget(self.selection_label)

        # Create a splitter widget.
        self.main_splitter = QSplitter()
        self.top_layout.addWidget(self.main_splitter)

        # Add a table for the left side of the dialog listing all materials.
        self.material_widget = SedimentMaterialTableWidget(self, self.data)
        self.main_splitter.addWidget(self.material_widget)

        # Add buttons for the bottom of the dialog.
        self.btn_box = QDialogButtonBox()
        self.top_layout.addWidget(self.btn_box)

        # Add a stack widget for the right side of the dialog. The stack will show the currently selected material.
        self.widgets['prop_stack'] = QStackedWidget()
        self.main_splitter.addWidget(self.widgets['prop_stack'])

        # Just use a fixed starting width of 300 for the table for now
        self.main_splitter.setSizes([300, 600])
        self.main_splitter.setStyleSheet(
            'QSplitter::handle:horizontal { background-color: lightgrey; }'
            'QSplitter::handle:vertical { background-color: lightgrey; }'
        )

        # Set up the button to edit all the material textures at once.
        icon_size = self.material_widget.toolbar.iconSize()
        dummy_opt = PolygonOptions()
        icon = DisplayOptionIconFactory.get_icon(dummy_opt, icon_size.height())
        self.material_widget.toolbar.addAction(icon, 'Change all material textures', self.on_btn_change_all)

        # Set up import and export buttons for the material table.
        self.import_btn = QPushButton('Import...')
        self.export_btn = QPushButton('Export...')
        self.import_btn.clicked.connect(self.on_btn_import)
        self.export_btn.clicked.connect(self.on_btn_export)
        self.material_widget.toolbar_layout.addWidget(self.import_btn)
        self.material_widget.toolbar_layout.addWidget(self.export_btn)

        # Connect material table signals to slots in this class.
        self.material_widget.added.connect(self._add_material)
        self.material_widget.deleted.connect(self._delete_materials)
        self.material_widget.model.itemChanged.connect(self._update_name)

        # Adjust the table size and selection mode.
        self.material_widget.table_view.setMinimumSize(300, 500)
        selection_model = self.material_widget.table_view.selectionModel()
        selection_model.currentChanged.connect(self.on_material_changed)

        # QDialogButtonBox with Ok and Cancel buttons
        self.btn_box.setOrientation(Qt.Horizontal)
        self.btn_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help)
        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        self.btn_box.helpRequested.connect(self.help_requested)

    def _add_prop_page(self, mat_id):
        """Adds a page for a material onto the stacked widget if one does not exist for the material.

        Args:
            mat_id (int): The id of the material that may need a page.
        """
        if mat_id in self.mat_id_to_page:
            return
        if mat_id not in self.data.materials:
            return
        tab_widget = QTabWidget()
        tab_widget.setObjectName('prop_tab_widget')

        self._add_bed_definition_tab(mat_id, tab_widget)

        self._add_consolidation_tab(mat_id, tab_widget)

        self.mat_id_to_page[mat_id] = self.widgets['prop_stack'].addWidget(tab_widget)

        # transport tab
        trans_prop_box, trans_prop_layout = self._add_property_group_widget(mat_id, 'prop_box_trans')
        tab_widget.addTab(trans_prop_box, 'Transport')

        # Create a transport constituents widget for the transport tab.
        transport = TransportConstituentMatAssignmentWidget(trans_prop_box)
        transport.setObjectName('transport')
        trans_prop_layout.setContentsMargins(0, 0, 0, 0)
        trans_prop_layout.addWidget(transport)
        transport.set_sediment_transport('', self.constituents)

        for row in range(transport.model.rowCount()):
            id_idx = transport.model.index(row, TransportConstituentMatAssignmentWidget.SOURCE_ID_COLUMN)
            constituent_id = float(transport.model.data(id_idx))
            if constituent_id in self.data.materials[mat_id].sediment_material_properties:
                constituent = self.data.materials[mat_id].sediment_material_properties[constituent_id]
                r_idx = transport.model.index(row, TransportConstituentMatAssignmentWidget.SOURCE_REFINEMENT_COLUMN)
                d_idx = transport.model.index(row, TransportConstituentMatAssignmentWidget.SOURCE_DIFFUSION_COLUMN)
                transport.model.setData(r_idx, constituent.refinement_tolerance)
                transport.model.setData(d_idx, constituent.turbulent_diffusion_rate)

        self.mat_id_to_page[mat_id] = self.widgets['prop_stack'].addWidget(tab_widget)
        self.mat_id_to_transport[mat_id] = transport.model

    def _add_bed_definition_tab(self, mat_id, tab_widget):
        """Adds the bed definition tab to the tab widget for the material.

        Args:
            mat_id (int): The material id.
            tab_widget (QTabWidget): The tab widget to add the bed definition tab to.
        """
        # Set up the bed definition tab
        bed_prop_box, bed_prop_layout = self._add_property_group_widget(mat_id, 'prop_box_bed_layers')
        tab_widget.addTab(bed_prop_box, 'Bed Definition')
        bed_table = BedLayerTableWidget(bed_prop_box, self.data, mat_id, self.constituents, True)
        bed_table.setObjectName('bed_layer_table')
        bed_table.bed_layer_type_changed(self.data.info.attrs['bed_layer_assignment_protocol'])
        if mat_id != 0:
            self._add_bed_layer_overrides(bed_prop_layout, bed_table, mat_id)
        bed_prop_layout.addWidget(bed_table)
        if mat_id != 0:
            self._add_other_bed_properties(bed_prop_layout, mat_id)

    def _add_other_bed_properties(self, bed_prop_layout, mat_id):
        """Adds widgets for editing other bed properties.

        Args:
            bed_prop_layout (QVBoxLayout): The layout to add the widgets to.
            mat_id (int): The material id.
        """
        # Add a checkbox for turning off displacement.
        displacement_off = QCheckBox()
        displacement_off.setObjectName('displacement_off')
        displacement_off.setText('Turn off displacement')
        bed_prop_layout.addWidget(displacement_off)
        is_displacement_off = self.data.materials[mat_id].displacement_off
        displacement_off.setChecked(is_displacement_off)

        # Add a checkbox for turning on local scour.
        local_scour = QCheckBox()
        local_scour.setObjectName('local_scour')
        local_scour.setText('Local scour')
        bed_prop_layout.addWidget(local_scour)
        is_local_scour = self.data.materials[mat_id].local_scour
        local_scour.setChecked(is_local_scour)

        # Add a checkbox for using bedload diffusion.
        use_bedload_diffusion = QCheckBox()
        use_bedload_diffusion.setObjectName('use_bedload_diffusion')
        use_bedload_diffusion.setText('Specify bedload diffusion constant')
        is_use_bedload_diffusion = self.data.materials[mat_id].use_bedload_diffusion
        use_bedload_diffusion.setChecked(is_use_bedload_diffusion)
        bed_prop_layout.addWidget(use_bedload_diffusion)

        # Add a label for the bedload diffusion constant.
        bedload_diffusion_label = QLabel()
        bedload_diffusion_label.setObjectName('bedload_diffusion_label')
        bedload_diffusion_label.setText('Bedload diffusion constant:')

        # Add an edit field for the bedload diffusion constant.
        bedload_diffusion = QLineEdit()
        bedload_diffusion.setObjectName('bedload_diffusion')
        bedload_diffusion.setText(str(self.data.materials[mat_id].bedload_diffusion))
        bedload_diffusion.installEventFilter(self.number_corrector)
        bedload_diffusion.setValidator(self.dbl_validator)

        # Add a layout for the bedload diffusion widgets.
        bedload_diffusion_layout = QHBoxLayout()
        bedload_diffusion_layout.addWidget(bedload_diffusion_label)
        bedload_diffusion_layout.addWidget(bedload_diffusion)
        bedload_diffusion_layout.addSpacerItem(QSpacerItem(0, 0, QSizePolicy.Expanding))
        use_bedload_diffusion.toggled.connect(bedload_diffusion_label.setVisible)
        use_bedload_diffusion.toggled.connect(bedload_diffusion.setVisible)
        bedload_diffusion_label.setVisible(is_use_bedload_diffusion)
        bedload_diffusion.setVisible(is_use_bedload_diffusion)
        bed_prop_layout.addLayout(bedload_diffusion_layout)

    def _add_bed_layer_overrides(self, bed_prop_layout, bed_table, mat_id):
        """Adds checkboxes for overriding bed layers for the material.

        Args:
            bed_prop_layout (QVBoxLayout): The layout to add the checkboxes to.
            bed_table (BedLayerTableWidget): The bed layer widget for specifying overriding values.
            mat_id (int): The material id.
        """
        # Add a checkbox for whether this material will override the bed layer values.
        override_bed_layers = QCheckBox()
        override_bed_layers.setObjectName('override_bed_layers')
        override_bed_layers.setText('Override bed layers')
        bed_prop_layout.addWidget(override_bed_layers)

        # Add a checkbox for overriding whether cohesion is specified.
        override_cohesive = QCheckBox()
        override_cohesive.setObjectName('override_cohesive')
        override_cohesive.setText('Specify cohesion')
        bed_prop_layout.addWidget(override_cohesive)
        use_cohesive_bed_layers = self.data.materials[mat_id].bed_layer_cohesive_override
        override_cohesive.setChecked(use_cohesive_bed_layers)

        is_overriden = self.data.materials[mat_id].bed_layer_override
        override_bed_layers.setChecked(is_overriden)
        bed_table.setEnabled(is_overriden)
        override_cohesive.setEnabled(is_overriden)
        override_bed_layers.toggled.connect(bed_table.setEnabled)
        override_bed_layers.toggled.connect(override_cohesive.setEnabled)
        override_cohesive.stateChanged[int].connect(bed_table.table_view.filter_model.set_visible_cohesive_properties)
        bed_table.table_view.filter_model.set_visible_cohesive_properties(use_cohesive_bed_layers)

    def _add_consolidation_tab(self, mat_id, tab_widget):
        """Adds the consolidation tab to the tab widget of the material.

        Args:
            mat_id (int): The material id.
            tab_widget (QTabWidget): The tab widget to add the consolidation tab to.
        """
        if self.data.info.attrs['use_consolidation'] != 0:
            # Set up the consolidation tab.
            con_prop_box, con_prop_layout = self._add_property_group_widget(mat_id, 'prop_box_consolidation')
            tab_widget.addTab(con_prop_box, 'Consolidation')
            con_table = ConsolidationTableWidget(con_prop_box, self.data, mat_id, True)
            con_table.setObjectName('consolidation_table')
            if mat_id != 0:
                override_consolidation = QCheckBox()
                override_consolidation.setObjectName('override_consolidation')
                override_consolidation.setText('Override consolidation')
                con_prop_layout.addWidget(override_consolidation)
                is_overriden = self.data.materials[mat_id].consolidation_override
                override_consolidation.setChecked(is_overriden)
                con_table.setEnabled(is_overriden)
                override_consolidation.toggled.connect(con_table.setEnabled)
            con_prop_layout.addWidget(con_table)

    def _add_property_group_widget(self, mat_id, object_name):
        """Adds a group box with the material name.

        Args:
            mat_id (int): The id of the material.
            object_name (str): The Qt object name to give the QGroupBox.

        Returns:
            A tuple of (QGroupBox, QVBoxLayout). The group box contains the layout.
        """
        prop_box = QGroupBox()
        prop_box.setObjectName(object_name)
        prop_box.setTitle(self.data.materials[mat_id].name)
        prop_box.setEnabled(mat_id > 0)
        prop_layout = QVBoxLayout()
        prop_box.setLayout(prop_layout)
        return prop_box, prop_layout

    def _add_material(self, mat_id):
        """Adds a material.

        Args:
            mat_id (int): The id of the material to add.
        """
        self._add_prop_page(mat_id)

    def _delete_materials(self, mat_ids):
        """Deletes materials.

        Args:
            mat_ids (list): List of integer ids of materials that will be deleted.
        """
        self.deleted_mat_ids.extend(mat_ids)
        for mat_id in mat_ids:
            self.mat_id_to_page.pop(mat_id)
            self.mat_id_to_transport.pop(mat_id)

    def _update_name(self, item):
        """Updates the titles of group boxes to show the materials name.

        Args:
            item (QStandardItem): The item in the material table that changed its name.
        """
        mat_id = item.data(Qt.UserRole)
        mat_name = item.data(Qt.EditRole)
        if mat_id in self.mat_id_to_page:
            page_idx = self.mat_id_to_page[mat_id]
            tab_widget = self.widgets['prop_stack'].widget(page_idx)
            for tab_idx in range(tab_widget.count()):
                box_widget = tab_widget.widget(tab_idx)
                if box_widget:
                    box_widget.setTitle(mat_name)

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def on_btn_import(self):
        """Imports AdH sediment material properties."""
        file_dir = self.last_import_export_path if self.last_import_export_path else QDir.homePath()
        filename, _ = QFileDialog.getOpenFileName(self, 'Open', dir=file_dir, filter='*.adh_sed_mat')
        if filename:
            self.last_import_export_path = os.path.dirname(filename)
            self.import_sediment_materials(filename)

    def import_sediment_materials(self, file):
        """Imports sediment materials from a file.

        Args:
            file (str): The file to import the sediment materials from.
        """
        import_mat = SedimentMaterialsIO(file)
        self.data.materials = import_mat.materials
        # Remove old values from the model and re-initialize with the new materials.
        self.material_widget.model.removeRows(0, self.material_widget.model.rowCount())
        self.material_widget.setup_table_model()
        # Remove old widgets for old materials
        self.mat_id_to_page.clear()
        self.mat_id_to_transport.clear()
        num_old_widgets = self.widgets['prop_stack'].count()
        for page_idx in range(num_old_widgets, -1, -1):
            self.widgets['prop_stack'].removeWidget(self.widgets['prop_stack'].widget(page_idx))
        # Set global checkboxes based on material 0.
        consolidation_override = self.data.materials[0].consolidation_override
        self.data.info.attrs['use_consolidation'] = 1 if consolidation_override else 0
        cohesive_override = self.data.materials[0].bed_layer_cohesive_override
        self.data.info.attrs['use_cohesive_bed_layers'] = 1 if cohesive_override else 0
        self.data.materials[0].consolidation_override = False
        self.data.materials[0].bed_layer_cohesive_override = False
        # Select the unassigned material
        self.material_widget.table_view.selectionModel().setCurrentIndex(
            self.material_widget.model.index(0, 1), QItemSelectionModel.Toggle
        )
        for mat_id in self.data.materials:
            self._add_prop_page(mat_id)

    def on_btn_export(self):
        """Exports AdH material properties to a file."""
        directory = self.last_import_export_path if self.last_import_export_path else QDir.homePath()
        filename, _ = QFileDialog.getSaveFileName(self, 'Save', dir=directory, filter='*.adh_sed_mat')
        if filename:
            self.last_import_export_path = os.path.dirname(filename)
            self.export_sediment_materials(filename)

    def export_sediment_materials(self, file):
        """
        Export sediment materials to a file.

        Args:
            file (str): The file to export the sediment materials to.
        """
        self._update_material_name_and_display()
        # Set the material 0 override values based on global values.
        self.data.materials[0].consolidation_override = self.data.info.attrs['use_consolidation'] != 0
        self.data.materials[0].bed_layer_cohesive_override = self.data.info.attrs['use_cohesive_bed_layers'] != 0
        export_mat = SedimentMaterialsIO(file)
        export_mat.materials = self.data.materials
        export_mat.commit()
        # Reset material 0 to the previous value.
        self.data.materials[0].consolidation_override = False
        self.data.materials[0].bed_layer_cohesive_override = False

    def on_material_changed(self):
        """Called when the current material in the table changes.

        Sets the property stack to show widgets for the current material.
        """
        current_tab = 0
        if self.widgets['prop_stack'].currentWidget():
            current_tab = max(self.widgets['prop_stack'].currentWidget().currentIndex(), 0)

        # Disable the delete button if the OFF material currently selected.
        index = self.material_widget.table_view.selectionModel().currentIndex()
        if not index.isValid():
            return
        index_row = index.row()
        index = self.material_widget.table_view.model().index(index_row, SedimentMaterialTableWidget.MAT_NAME_COL)
        enable_delete = index_row != SedimentMaterialsIO.UNASSIGNED_MAT  # can delete all but unassigned material
        delete_material_action = self.material_widget.btn_actions[self.material_widget.delete_icon]
        self.material_widget.toolbar.widgetForAction(delete_material_action).setEnabled(enable_delete)

        mat_id = self.material_widget.table_view.model().data(index, role=Qt.UserRole)
        if mat_id not in self.mat_id_to_page:
            self._add_prop_page(mat_id)
        self.widgets['prop_stack'].setCurrentIndex(self.mat_id_to_page[mat_id])
        self.widgets['prop_stack'].currentWidget().setCurrentIndex(current_tab)

    def on_btn_change_all(self):
        """Called when the change all material texture/style button is clicked."""
        dlg = PolygonDisplayOptionsDialog(PolygonOptions(), self)
        dlg.ui.lab_line_color.setVisible(False)
        dlg.ui.color_btn.setVisible(False)
        if dlg and dlg.exec():
            new_opts = dlg.get_options()
            for row in range(self.material_widget.model.rowCount()):
                idx = self.material_widget.model.index(row, 0)
                disp_data = self.material_widget.model.data(idx, Qt.UserRole)
                disp_data.texture = new_opts.texture
                self.material_widget.model.setData(idx, disp_data, Qt.UserRole)

    def _update_transport_constituents_for_save(self):
        """Updates the transport constituents.

        Goes through all the materials, removes values for removed constituents, adds default values for added
        constituents.
        """
        for mat_id in self.data.materials:
            for _ in range(self.widgets['prop_stack'].count()):
                transport_model = self.mat_id_to_transport.get(mat_id)
                if transport_model is not None:
                    for row in range(transport_model.rowCount()):
                        id_idx = transport_model.index(row, TransportConstituentMatAssignmentWidget.SOURCE_ID_COLUMN)
                        constituent_id = float(transport_model.data(id_idx))
                        constituent = MaterialTransportProperties()
                        r_idx = transport_model.index(
                            row, TransportConstituentMatAssignmentWidget.SOURCE_REFINEMENT_COLUMN
                        )
                        d_idx = transport_model.index(
                            row, TransportConstituentMatAssignmentWidget.SOURCE_DIFFUSION_COLUMN
                        )
                        constituent.refinement_tolerance = float(transport_model.data(r_idx))
                        constituent.turbulent_diffusion_rate = float(transport_model.data(d_idx))
                        self.data.materials[mat_id].sediment_material_properties[constituent_id] = constituent

    def _update_material_name_and_display(self):
        """Updates the material data for either saving or export."""
        row_count = self.material_widget.model.rowCount()

        # Get the unassigned (material 0) display options
        display_idx = self.material_widget.model.index(0, 0)
        disp_opts = self.material_widget.model.data(display_idx, Qt.UserRole)
        self.data.materials[0].display = disp_opts

        self._update_transport_constituents_for_save()

        # Skip the unassigned material, since it is mostly disabled and cannot be edited.
        for row in range(1, row_count):
            display_idx = self.material_widget.model.index(row, 0)
            name_idx = self.material_widget.model.index(row, 1)
            mat_id = self.material_widget.model.data(name_idx, Qt.UserRole)
            mat_name = self.material_widget.model.data(name_idx, Qt.DisplayRole)
            disp_opts = self.material_widget.model.data(display_idx, Qt.UserRole)
            self.data.materials[mat_id].name = mat_name
            self.data.materials[mat_id].display = disp_opts
            # skip this iteration if nothing changed
            if mat_id not in self.mat_id_to_page:
                continue

            # Save bed layer data.
            override_bed_layers = self.get_material_check_box(mat_id, 'override_bed_layers')
            override_cohesive = self.get_material_check_box(mat_id, 'override_cohesive')
            self.data.materials[mat_id].bed_layer_override = override_bed_layers.isChecked()
            self.data.materials[mat_id].bed_layer_cohesive_override = override_cohesive.isChecked()

            # Save other bed data.
            displacement_off = self.get_material_check_box(mat_id, 'displacement_off')
            local_scour = self.get_material_check_box(mat_id, 'local_scour')
            use_bedload_diffusion = self.get_material_check_box(mat_id, 'use_bedload_diffusion')
            bedload_diffusion = self.get_material_line_edit(mat_id, 'bedload_diffusion')
            self.data.materials[mat_id].displacement_off = displacement_off.isChecked()
            self.data.materials[mat_id].local_scour = local_scour.isChecked()
            self.data.materials[mat_id].use_bedload_diffusion = use_bedload_diffusion.isChecked()
            self.data.materials[mat_id].bedload_diffusion = float(bedload_diffusion.text())

            # Save consolidation if used.
            if self.data.info.attrs['use_consolidation'] != 0:
                override_consolidation = self.get_material_check_box(mat_id, 'override_consolidation')
                self.data.materials[mat_id].consolidation_override = override_consolidation.isChecked()

    def get_material_check_box(self, mat_id: int, name: str) -> QCheckBox:
        """Get a checkbox widget for the given material ID and name.

        Args:
            mat_id (int): The material ID.
            name (str): The name of the checkbox.
        """
        page = self.mat_id_to_page[mat_id]
        widget_page = self.widgets['prop_stack'].widget(page)
        checkbox = widget_page.findChildren(QCheckBox, name)[0]
        return checkbox

    def get_material_line_edit(self, mat_id: int, name: str) -> QLineEdit:
        """Get a line edit widget for the given name.

        Args:
            mat_id (int): The material ID.
            name (str): The name of the line edit.
        """
        page = self.mat_id_to_page[mat_id]
        widget_page = self.widgets['prop_stack'].widget(page)
        line_edit = widget_page.findChildren(QLineEdit, name)[0]
        return line_edit

    def accept(self):
        """Saves the sediment material data."""
        self._update_material_name_and_display()
        return super().accept()

    def reject(self):
        """Cancels out of the dialog."""
        return super().reject()
