"""Query implementation used during normal XMS communication."""

# 1. Standard python modules
from collections.abc import Iterable

# 2. Third party modules

# 3. Aquaveo modules
import xms.data_objects._data_objects.parameters as cdop
import xms.data_objects.parameters as pydop
from xms.datasets.dataset_reader import DatasetReader

# 4. Local modules
from xms.api._xmsapi.dmi import Query as CQuery
from xms.api.Agent import XmsAgent as PyAgent
from xms.api.tree import tree_util
from xms.core.filesystem import filesystem as io_util

POINT_TARGET_TYPE = 0  # Corresponds to the xms.guipy.data.target_type.TargetType enum.
ARC_TARGET_TYPE = 1
ARC_GROUP_TARGET_TYPE = 9  # Came later for GMS, why it is out of order
POLYGON_TARGET_TYPE = 2


def wrap_data_object(do_object):
    """Returns a pure Python wrapped object given a C++ wrapped data_object.

    Currently only works with things that have a UUID (no Arc, Point, etc.)

    Args:
        do_object (xms.data_object): The data_object to wrap

    Returns:
        See description
    """
    if isinstance(do_object, cdop.Component):
        return pydop.Component(instance=do_object)
    if isinstance(do_object, cdop.Simulation):
        return pydop.Simulation(instance=do_object)
    if isinstance(do_object, cdop.Coverage):
        return pydop.Coverage(instance=do_object)
    if isinstance(do_object, cdop.Dataset):
        return DatasetReader(h5_filename=do_object.GetFilename(), group_path=do_object.GetPath())
    if isinstance(do_object, cdop.RectilinearGrid):
        return pydop.RectilinearGrid(instance=do_object)
    if isinstance(do_object, cdop.UGrid):
        if do_object.GetXmUGridFile().startswith('Error:'):
            raise RuntimeError(do_object.GetXmUGridFile())
        return pydop.UGrid(instance=do_object)
    if isinstance(do_object, str):  # May just be a filename in case of GIS data
        return do_object
    return None


class QueryImpl:
    """Query implementation used during normal XMS communication."""
    def __init__(self, **kwargs):
        """Construct the wrapper.

        Args:
            instance (xms.api._xmsapi.dmi.Query, optional): The C++ object to wrap. Mutually exclusive with 'agent'
                kwarg.
            agent (xms.api.XmsAgent, optional): The XMS interprocess communication manager, mutually exclusive with
                'instance' kwarg.
            timeout (int): The interprocess communication timeout interval in milliseconds. Defaults to 5 minutes,
                which might be a little excessive. Used to have lots of script timeouts when we serialized more
                data through the pipe.
            retries (int): Number of times to retry after a failed message request. Defaults to one. I have never
                set it to anything else, and doing sometimes causes duplicate build data being created in XMS
                from import scripts.
            progress_script (bool): True if created in a progress script.
            migrate_script (bool): True if created in a migrate script.
            blocking (bool): False if this a non-blocking script.
        """
        if 'instance' in kwargs:
            self._instance = kwargs['instance']
        elif 'agent' in kwargs:
            self._instance = CQuery(kwargs['agent']._instance)
        else:
            self._instance = CQuery()

        # Set up XmsAgent using kwargs or defaults
        c_agent = self._instance.GetXmsAgent()
        timeout = kwargs.get('timeout', 300000)
        c_agent.SetTimeout(timeout)
        retries = kwargs.get('retries', 1)
        c_agent.SetRetries(retries)

        # Set up for progress script, if needed.
        if kwargs.get('progress_script', False):
            # If we are a progress script, set the initial Context from the ProgressLoop.
            self._progress_context = self._instance.GetXmsAgent().GetSession().GetProgressLoop().GetProgressContext()
            self._instance.SetContext(self._progress_context)
            # Move up to the parent simulation because we start out in a useless place for retrieving data.
            self._instance.Select('Parent')
            self._start_context = self._instance.GetContext()
        else:
            self._progress_context = None
            self._start_context = self._instance.GetContext()
        self._root_instance = self._start_context.GetRootInstance()

        # Set migrate script flag
        self._migrate_script = kwargs.get('migrate_script', False)
        # Set the blocking flag
        self._set_blocking(kwargs.get('blocking', True))

        self._build_vertex = None
        self._selectuuid = None  # SelectUuid Context
        self._componentcoverageids = None  # ComponentCoverageIds Context
        self._componentcoverageselectedids = None  # ComponentCoverageSelectedIds Context
        self._namedexecutables = None  # NamedExecutable Context
        self._build_vertices = []
        self._model_check_errors = []
        self._unlink_items = []  # [(taker_uuid, taken_uuid)]
        self._delete_items = []  # [item_uuid]
        self._rename_items = []  # [(item_uuid, new_name)]
        self._edit_elevations = []  # [json_file]
        self._started_progress_update = False  # Set to True the first time a progress script updates percent done
        self._project_tree = None
        self._global_time = None
        self._global_time_settings = None
        self._display_projection = None
        self._copy_protection = None

    def _set_blocking(self, blocking):
        """Tell XMS if this a blocking script or not.

        Args:
            blocking (bool): If False, this is a non-blocking script
        """
        if blocking:
            return  # Don't even bother if this is a blocking script.
        # Set up the blocking Query. We need to send a flag to XMS if this is a non-blocking script. Whenever we call
        # Send(), the Query would not allow us to Get() anything
        blocking_query = CQuery()
        # Use the blocking Query so we don't mess up the main one.
        blocking_query_context = blocking_query.GetContext()
        blocking_root_instance = blocking_query_context.GetRootInstance()
        # Send the blocking flag back to XMS.
        serialize_bool = cdop.BoolLiteral(blocking)
        arg_list = [{'#description': 'Blocking', 'blocking_script': serialize_bool}]
        blocking_query.Add(arg_list, blocking_root_instance)
        blocking_query.Send()
        # Delete the blocking Query as it won't be used anymore.
        blocking_query = None

    # Data commonly retrieved at the root simulation/component level.
    @property
    def project_tree(self):
        """Returns the XMS project explorer tree."""
        if self._project_tree is None:
            current_context = self._instance.GetContext()
            self._instance.SetContext(self._start_context)
            creator = tree_util.ProjectExplorerTreeCreator()
            self._project_tree = creator.create(self)
            self._instance.SetContext(current_context)
        return self._project_tree

    def refresh_project_tree(self):
        """Returns the XMS project explorer tree."""
        try:
            current_context = self._instance.GetContext()
            self._instance.SetContext(self._start_context)
            creator = tree_util.ProjectExplorerTreeCreator()
            self._project_tree = creator.create(self)
            self._instance.SetContext(current_context)
        except:
            self._project_tree = None
        return self._project_tree

    def copy_project_tree(self, tree_node=None):
        """Make a deep copy of the project explorer tree.

        Args:
            tree_node (TreeNode): Root of the project explorer tree to deep copy. If not provided, will use XMS project
                root.

        Returns:
            TreeNode: See description
        """
        if tree_node is None:
            tree_node = self.project_tree
        return tree_util.copy_tree(tree_node)

    @property
    def global_time(self):
        """Returns the current XMS global time."""
        if self._global_time is None:
            current_context = self._instance.GetContext()
            self._instance.SetContext(self._start_context)
            result = self._instance.Get('global_time')['global_time']
            if result and result[0]:
                self._global_time = pydop.date_time_literal_to_datetime(result[0])
            self._instance.SetContext(current_context)
        return self._global_time

    @property
    def global_time_settings(self):
        """Returns the current XMS global time settings as a JSON string.

        See xms.guipy.time_format.XmsTimeFormatter for more details.
        """
        if self._global_time_settings is None:
            current_context = self._instance.GetContext()
            self._instance.SetContext(self._start_context)
            result = self._instance.Get('global_time_settings')['global_time_settings']
            if result and result[0]:
                self._global_time_settings = result[0]
            self._instance.SetContext(current_context)
        return self._global_time_settings

    @property
    def display_projection(self):
        """Returns the current XMS display projection."""
        if self._display_projection is None:
            current_context = self._instance.GetContext()
            self._instance.SetContext(self._start_context)
            result = self._instance.Get('projection')['projection']
            if result and result[0]:
                self._display_projection = pydop.Projection(instance=result[0])
            self._instance.SetContext(current_context)
        return self._display_projection

    @property
    def cp(self):
        """Returns a dict of GUI module name text to copy protection enable/disable flag."""
        if self._copy_protection is None:
            current_context = self._instance.GetContext()
            self._instance.SetContext(self._start_context)
            result = self._instance.Get('copy_protection')['copy_protection']
            if result and result[0]:
                self._copy_protection = result[0]
            self._instance.SetContext(current_context)
        return self._copy_protection

    # Pure Python wrappings for commonly used Query members.
    @property
    def xms_agent(self):
        """Returns the Query object's xms.api.XmsAgent."""
        return PyAgent(instance=self._instance.GetXmsAgent())

    @property
    def read_file(self):
        """Returns the filesystem path to the file being imported by the process.

        Note only applicable if process is an import script.
        """
        return self._instance.GetReadFile()

    # Methods for retrieving data from XMS.
    def item_with_uuid(self, item_uuid, generic_coverage=False, model_name=None, unique_name=None):
        """Retrieves an item from XMS given its UUID.

        Note:
            To get a hidden component item_uuid should be the UUID of the hidden component's owner. model_name is the
            hidden component's XML-defined model name. unique_name is the hidden component's XML-defined unique name.

        Args:
            item_uuid (str): UUID of the item
            generic_coverage (bool): True if the item is one of the dumpable generic coverages
            model_name (str): The XML model name of a hidden component. Mutually exclusive with generic_coverage.
                Must specify unique_name if model_name provided.
            unique_name (str): The XML unique name of a hidden component. Mutually exclusive with generic_coverage.
                Must specify model_name if unique_name provided.

        Returns:
            Pure python interface to the requested item.
        """
        xms_item = None
        if not item_uuid:
            return xms_item

        keyword = item_uuid
        if generic_coverage:  # Append KEYWORD_COV_WITH_ATTRIBUTES
            keyword += '#COVERAGE_WITH_ATTRIBUTES'
        elif model_name and unique_name:
            keyword += f'#{model_name}#{unique_name}'

        current_context = self._instance.GetContext()  # Save current position so we can restore it when done
        self._instance.SetContext(self._select_uuid_context())
        result = self._instance.Get(keyword)[keyword]
        if result and result[0]:  # Get a pure Python wrapping if we found the item
            if generic_coverage:
                xms_item = result[0]
            else:
                xms_item = wrap_data_object(result[0])
        self._instance.SetContext(current_context)  # Restore original Context
        return xms_item

    def current_item(self):
        """Returns a KEYWORD_NONE Get request for the current position in the Query Context.

        Note assuming Context position is at something with a UUID. Don't try to use if you are down at the Coverage
        Arc level or something like that.

        Returns:
            The KEYWORD_NONE result for the current Context position or None on failure.
        """
        result = self._instance.Get()['none']
        if result and result[0]:
            return wrap_data_object(result[0])
        return None

    def current_item_uuid(self):
        """Returns the UUID of the item at the current Context position, if it has one.

        Returns:
            str: The current item's UUID or empty string if it doesn't have one.
        """
        result = self._instance.Get('uuid')['uuid']
        return result[0] if result and result[0] else ''

    def parent_item(self):
        """Returns a KEYWORD_NONE Get request for the parent of the current position in the Query Context.

        Note assuming Context position's parent is something with a UUID. Don't try to use if you are down at the
        Coverage Arc level or something like that.

        Returns:
            The KEYWORD_NONE result for the current context position's parent or None on failure.
        """
        current_context = self._instance.GetContext()
        self._instance.Select('Parent')  # Move Context up to the parent
        parent_item = self.current_item()
        self._instance.SetContext(current_context)  # Reset Context
        return parent_item

    def parent_item_uuid(self):
        """Returns the UUID of the current Context position's parent, if it has one.

        Returns:
            str: The UUID of the current item's parent or empty string if it doesn't have one.
        """
        current_context = self._instance.GetContext()
        self._instance.Select('Parent')  # Move Context up to the parent
        parent_item_uuid = self.current_item_uuid()
        self._instance.SetContext(current_context)  # Reset Context
        return parent_item_uuid

    def named_executable_path(self, exe_name):
        """Get the path to a model executable that is defined in the XML.

        Args:
            exe_name (str): The XML-defined executable name

        Returns:
            str: Path to the named executable or empty string if not found
        """
        current_context = self._instance.GetContext()
        self._instance.Select('NamedExecutables', a_context=self._start_context)
        result = self._instance.Get(exe_name)[exe_name]
        exe_path = ''
        if result and result[0]:
            exe_path = result[0]
        self._instance.SetContext(current_context)
        return exe_path

    def load_component_ids(self, component, points, arcs, arc_groups, polygons, only_selected, delete_files):
        """Query XMS for a dump of all the current component ids of the specified entity type.

        Args:
            component (xms.components.bases.coverage_component_base.CoverageComponentBase): The coverage component to
                load id files for
            points (bool): True if you want point component id maps
            arcs (bool): True if you want arc component id maps
            arc_groups (bool): True if you want arc group component id maps
            polygons (bool): True if you want polygon component id maps
            only_selected (bool): If True, only retrieves mappings for the currently selected features
            delete_files (bool): If True, id files will be deleted before return

        Returns:
            dict: The file dict.

            key = entity_type ('POINT', 'ARC', 'POLYGON'),

            value = (att_id_file, comp_id_file)

            Note that if delete_files=True, these files will not exist upon return. If you need to make a copy of
            the component id files, set delete_files=False and manually clean up the files after copying.
        """
        file_dict = {}
        current_ctxt = self._instance.GetContext()  # Store starting position
        self._instance.SetContext(
            self._selected_component_id_context() if only_selected else self._component_id_context()
        )
        pt_keyword = f'{component.uuid}#{component.cov_uuid}#{POINT_TARGET_TYPE}'
        arc_keyword = f'{component.uuid}#{component.cov_uuid}#{ARC_TARGET_TYPE}'
        arc_group_keyword = f'{component.uuid}#{component.cov_uuid}#{ARC_GROUP_TARGET_TYPE}'
        poly_keyword = f'{component.uuid}#{component.cov_uuid}#{POLYGON_TARGET_TYPE}'
        keywords = []
        if points:
            keywords.append(pt_keyword)
        if arcs:
            keywords.append(arc_keyword)
        if arc_groups:
            keywords.append(arc_group_keyword)
        if polygons:
            keywords.append(poly_keyword)
        if not keywords:
            raise ValueError('At least one of `points`, `arcs`, `arc_groups`, `polygons` arguments must be True.')
        id_res = self._instance.Get(*keywords)

        res_map = id_res.GetParameters()  # Get result as a Python dict
        if points:  # Unpack point component coverage id map if we got them
            pt_id_result = res_map.get(pt_keyword)
            if pt_id_result and pt_id_result[0]:
                pt_id_map = pt_id_result[0][0]
                file_dict.update({key: (value[0], value[1]) for key, value in pt_id_map.items()})
        if arcs:  # Unpack arc component coverage id map if we got them
            arc_id_result = res_map.get(arc_keyword)
            if arc_id_result and arc_id_result[0]:
                arc_id_map = arc_id_result[0][0]
                file_dict.update({key: (value[0], value[1]) for key, value in arc_id_map.items()})
        if arc_groups:  # Unpack arc group component coverage id map if we got them
            arc_group_id_result = res_map.get(arc_group_keyword)
            if arc_group_id_result and arc_group_id_result[0]:
                arc_group_id_map = arc_group_id_result[0][0]
                file_dict.update({key: (value[0], value[1]) for key, value in arc_group_id_map.items()})
        if polygons:  # Unpack polygon component coverage id map if we got them
            poly_id_result = res_map.get(poly_keyword)
            if poly_id_result and poly_id_result[0]:
                poly_id_map = poly_id_result[0][0]
                file_dict.update({key: (value[0], value[1]) for key, value in poly_id_map.items()})
        component.load_coverage_component_id_map(file_dict)

        self._instance.SetContext(current_ctxt)  # Restore starting position
        if delete_files:  # Clean up temp id files
            for filenames in file_dict.values():
                io_util.removefile(filenames[0])  # XMS id file
                io_util.removefile(filenames[1])  # Component id file
        return file_dict

    def coverage_attributes(self, cov_uuid, points=False, arcs=False, polygons=False, arc_groups=False):
        """Retrieve coverage attribute table from XMS.

        Args:
            cov_uuid (str): UUID of the coverage
            points (bool): True to retrieve feature point attributes
            arcs (bool): True to retrieve feature arc attributes
            polygons (bool): True to retrieve feature polygon attributes
            arc_groups (bool): True to retrieve feature arc group attributes

        Returns:
            dict: The requested attribute tables. Keyed by 'points', 'arcs', 'polygons', and 'arc_groups'.
        """
        if not any([points, arcs, polygons, arc_groups]):
            raise RuntimeError('Must specify at least one of points, arcs, polygons, arc_groups')

        att_tables = {}
        current_context = self._instance.GetContext()  # Save current position so we can restore it when done
        self._instance.SetContext(self._select_uuid_context())
        if points:
            point_query = f'{cov_uuid}#point_base_file'
            att_tables['points'] = self._instance.Get(point_query)[point_query][0]
        if arcs:
            arc_query = f'{cov_uuid}#arc_base_file'
            att_tables['arcs'] = self._instance.Get(arc_query)[arc_query][0]
        if polygons:
            poly_query = f'{cov_uuid}#polygon_base_file'
            att_tables['polygons'] = self._instance.Get(poly_query)[poly_query][0]
        if arc_groups:
            arc_group_query = f'{cov_uuid}#arc_group_base_file'
            att_tables['arc_groups'] = self._instance.Get(arc_group_query)[arc_group_query][0]
        self._instance.SetContext(current_context)
        return att_tables

    # Methods for adding data to be built in XMS.
    def add_simulation(self, do_sim, components=None, component_keywords=None):
        """Add a simulation to XMS.

        Args:
            do_sim (pydop.Simulation): The simulation to add
            components (list of pydop.Component): The simulation's hidden components. Component objects must have the
                model name and unique name attributes set.
            component_keywords (list of dict): Parallel with components if provided. Dicts of additional component
                keywords to send with the components, such as 'actions' or 'display_options'. See kwarg docs of
                add_component() for more details.
                ::
                    [{  # Possible component data keywords
                        'actions': [],
                        'messages': [],
                        'display_options': [],
                        'component_coverage_ids': [],
                        'shapefile_atts': {},
                    }]
        """
        arg_list = [{'#description': 'BuildNoTake', 'Simulation': do_sim._instance}]
        self._build_vertices.extend(self._instance.Add(arg_list, self._ensure_build_vertex_exists()))

        if components:  # Add the hidden simulation components
            sim_vertex = self._build_vertices[-1]
            for idx, component in enumerate(components):
                keywords = component_keywords[idx] if component_keywords and idx < len(component_keywords) else None
                self._add_hidden_component(component, sim_vertex, keywords)

    def add_coverage(
        self,
        do_coverage,
        model_name=None,
        coverage_type=None,
        components=None,
        component_keywords=None,
        folder_path=None
    ):
        """Add a coverage to XMS.

        Args:
            do_coverage (Union[pydop.Coverage, xms.api._xmsapi.dmi.DataDumpIOBase]): The coverage to add. If type is an
                xmscoverage generic coverage dump, all other arguments are ignored.
            model_name (str): XML-defined name of the coverage's model. If specified, must provide coverage_type.
            coverage_type (str): XML-defined coverage type. If specified, must provide model_name.
            components (list of pydop.Component): The simulation's hidden components. Component objects must have the
                model name and unique name attributes set.
            component_keywords (list of dict): Parallel with components if provided. Dicts of additional component
                keywords to send with the components, such as 'actions' or 'display_options'. See kwarg docs of
                add_component() for more details.
                ::
                    [{  # Possible component data keywords
                        'actions': [],
                        'messages': [],
                        'display_options': [],
                        'component_coverage_ids': [],
                        'shapefile_atts': {},
                    }]
            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        is_dmi_coverage = isinstance(do_coverage, pydop.Coverage)
        arg_dict = {'#description': 'BuildNoTake'}
        if is_dmi_coverage:
            arg_dict['COVERAGE_TYPE'] = f'{model_name}#{coverage_type}'
            arg_dict['Coverage'] = do_coverage._instance
        else:  # Add an xmscoverage generic coverage dump
            arg_dict['COVERAGE_WITH_ATTRIBUTES'] = do_coverage

        # Add desired tree folder hierarchy, if specified.
        if folder_path:
            arg_dict['folder_tree_path'] = folder_path

        self._build_vertices.extend(self._instance.Add([arg_dict], self._ensure_build_vertex_exists()))

        if components and is_dmi_coverage:  # Add the hidden simulation components
            cov_vertex = self._build_vertices[-1]
            for idx, component in enumerate(components):
                keywords = component_keywords[idx] if component_keywords and idx < len(component_keywords) else None
                self._add_hidden_component(component, cov_vertex, keywords)

    def add_ugrid(self, do_ugrid):
        """Add a UGrid to be built in XMS.

        Args:
            do_ugrid (pydop.UGrid): The UGrid to add
        """
        arg_list = [{'#description': 'BuildNoTake', 'Geometry': do_ugrid._instance}]
        self._build_vertices.extend(self._instance.Add(arg_list, self._ensure_build_vertex_exists()))

    def add_dataset(self, do_dataset, folder_path=None):
        """Add a Dataset to be built in XMS.

        Args:
            do_dataset (Union[pydop.Dataset, DatasetReader, xms.datasets.dataset_writer.DatasetWriter]): The dataset to
                add
            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        if type(do_dataset) == pydop.Dataset:  # Pure python wrapped Dataset
            instance = do_dataset._instance
        else:  # Assuming DatasetReader or DatasetWriter from xmsdatasets
            # Construct a pure Python data_objects Dataset, lets us default locations.
            py_dset = pydop.Dataset(do_dataset.h5_filename, do_dataset.group_path)
            instance = py_dset._instance
        arg_dict = {'#description': 'BuildNoTake', 'solution_dataset': instance}

        # Add desired tree folder hierarchy, if specified.
        if folder_path:
            arg_dict['folder_tree_path'] = folder_path

        self._build_vertices.extend(self._instance.Add([arg_dict], self._ensure_build_vertex_exists()))

    def add_raster(self, filename, item_uuid):
        """Add a GIS raster file to be loaded in XMS.

        Args:
            filename (str): Full path to the raster file
            item_uuid (Optional[str]): UUID to assign the raster in XMS
        """
        if item_uuid:
            arg_dict = {'#description': 'BuildNoTake', 'gis_raster_file': filename, 'uuid': item_uuid}
        else:
            arg_dict = {'#description': 'BuildNoTake', 'gis_raster_file': filename}
        self._build_vertices.extend(self._instance.Add([arg_dict], self._ensure_build_vertex_exists()))

    def add_particles(self, filename: str):
        """Add a particle set to be loaded in XMS.

        Args:
            filename: Path to the particle set file. Should be a file that XMS knows how to read. For SMS, this is
                an XMDF file. GMS has no support yet.
        """
        arg_dict = {'#description': 'BuildNoTake', 'particle_file': filename}
        self._build_vertices.extend(self._instance.Add([arg_dict], self._ensure_build_vertex_exists()))

    def add_shapefile(self, filename, item_uuid):
        """Add a GIS shape file to be loaded in XMS.

        Args:
            filename (str): Full path to the shape file
            item_uuid (Optional[str]): UUID to assign the shapefile in XMS
        """
        if item_uuid:
            arg_dict = {'#description': 'BuildNoTake', 'gis_shapefile': filename, 'uuid': item_uuid}
        else:
            arg_dict = {'#description': 'BuildNoTake', 'gis_shapefile': filename}
        self._build_vertices.extend(self._instance.Add([arg_dict], self._ensure_build_vertex_exists()))

    def add_component(
        self,
        do_component,
        actions=None,
        messages=None,
        display_options=None,
        coverage_ids=None,
        shapefile_atts=None,
        folder_path=None
    ):
        """Add a non-hidden Component to be built in XMS.

        Args:
            do_component (pydop.Component): The component to add
            actions (list of pydop.ActionRequest): ActionRequests to send back with the component. See
                xms.components.bases.component_base.ComponentBase for more details.
            messages (list of tuple of str): Messages to send back with component. Should be formatted as follows:
                [('MSG_LEVEL', 'message text')...]  See xms.components.bases.component_base.ComponentBase for more
                details.
            display_options (list): Display options to send back with the component. Data should be in format returned
                by xms.components.bases.component_base.ComponentBase.get_display_options().
            coverage_ids (list): Coverage component id assignments to send back with the component. Data should be in
                format returned by
                xms.components.bases.coverage_component_base.CoverageComponentBase.get_component_coverage_ids().
            shapefile_atts (dict): Shapefile attributes to send back with the component. Data should be in
                format returned by xms.components.bases.component_base.ComponentBase.get_shapefile_att_files().
            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        arg_dict = {'#description': 'BuildNoTake', 'Component': do_component._instance}

        if actions:
            # Unwrap Python ActionRequests if given.
            arg_dict['actions'] = [action._instance for action in actions]
        if messages:
            arg_dict['messages'] = messages
        if display_options:
            arg_dict['display_options'] = display_options
        if coverage_ids:
            arg_dict['component_coverage_ids'] = coverage_ids
        if shapefile_atts:
            arg_dict['shapefile_atts'] = shapefile_atts
        if folder_path:
            arg_dict['folder_tree_path'] = folder_path
        self._build_vertices.extend(self._instance.Add([arg_dict], self._ensure_build_vertex_exists()))

    def add_model_check_errors(self, errors):
        """Adds model check errors to be sent back to XMS.

        Errors will all be set at the Context level we had at construction. It is assumed that nothing else will be
        added during the script.

        Args:
            errors (Union[pydop.ModelCheckError, Sequence[pydop.ModelCheckError]): The model check errors to add. Can
                pass an iterable or a single error
        """
        if not isinstance(errors, Iterable):  # Allow argument to be a single error
            errors = [errors]
        self._model_check_errors.extend([error._instance for error in errors])

    def link_item(self, taker_uuid, taken_uuid):
        """Link one item under another.

        Args:
            taker_uuid (str): UUID of the item to link the other item under. This should be the UUID of the parent tree
                item, even if the item is taken by a hidden component.
            taken_uuid (str): UUID of the item to link under the other item
        """
        arg_list = [{'#description': 'AddTakeUuid', '': taken_uuid, 'taker_uuid': taker_uuid}]
        self._build_vertices.extend(self._instance.Add(arg_list, self._ensure_build_vertex_exists()))

    def unlink_item(self, taker_uuid, taken_uuid):
        """Unlink one item from under another.

        Args:
            taker_uuid (str): UUID of the item to unlink the other item from. This should be the UUID of the parent tree
                item, even if the item is taken by a hidden component.
            taken_uuid (str): UUID of the item to unlink from the other item
        """
        self._unlink_items.append((taker_uuid, taken_uuid))

    def delete_item(self, item_uuid):
        """Delete an item in XMS.

        Args:
            item_uuid (str): UUID of the item to delete
        """
        self._delete_items.append(item_uuid)

    def rename_item(self, item_uuid, new_name):
        """Rename a tree item in XMS.

        Args:
            item_uuid (str): UUID of the item to rename
            new_name (str): New name to assign the item. Not guaranteed to be the name XMS assigns the item. Will be
                made unique if another item with the same name already exists.
        """
        self._rename_items.append((item_uuid, new_name))

    def edit_ugrid_top_bottom_elevations(self, json_file):
        """Edit the top and bottom elevations of a UGrid in XMS.

        Args:
            json_file (str): Path to the JSON file containing the new elevations. See xms.components.dataset_metadata
                for more information of the expected JSON structure.
        """
        self._edit_elevations.append(json_file)

    def get_ugrid_selection(self, ugrid_uuid):
        """Get the currently selected UGrid cells and points.

        Returns:
            dict: Possible keys are 'POINT' and 'CELL'. Values are lists of UGrid poiont/cell ids. If key does not
            exist, no UGrid entities of that type are selected.
        """
        current_context = self._instance.GetContext()  # Save current position so we can restore it when done
        self._instance.SetContext(self._select_uuid_context())
        keyword = f'{ugrid_uuid}#selection'
        selection = {}
        result = self._instance.Get(keyword)[keyword]
        if result and result[0]:
            selection = result[0]
        self._instance.SetContext(current_context)
        return selection

    def update_progress_percent(self, percent_done):
        """Update the progress percent of a running process in the XMS Simulation Run Queue.

        Obviously only makes sense if this is a progress script.

        Args:
            percent_done (Union[int, float]): The monitored process's new progress percent. Will be truncated if float.
        """
        if not self._started_progress_update:
            # This is the first time script has updated progress percent. Move back to the original starting progress
            # loop context.
            self._instance.SetContext(self._progress_context)
            self._started_progress_update = True
        self._instance.Set([{"": int(percent_done)}])  # XMS only accepts integer percents
        self._instance.Send()

    # Send any built data to XMS.
    def send(self, component_event=False):
        """Send data queued from add() and set() calls to XMS.

        Args:
            component_event (bool): True if send is being sent from a component event script. They do all sorts of their
                own special messing with the build context, so don't try to manage it.

        Returns:
            xms.api.dmi.SetDataResult: Result object reply from the send request
        """
        if self._model_check_errors:  # Assuming no other build data if doing model checks.
            query_errors = [{'#description': 'ModelCheck', '': error} for error in self._model_check_errors]
            self._instance.SetContext(self._start_context)
            self._instance.Add(query_errors)  # Place marks of all ModelCheck vertices will be set.
        else:
            # Root instance data has to be added last, so do it now.
            self._add_unlink_items()
            self._add_delete_items()
            self._add_rename_items()
            self._add_edit_elevations()

            # Ensure everything that has been added gets built.
            build_context = self._instance.GetContext()
            if not component_event:  # Don't mess with component event scripts. They do their own thing.
                build_context.ClearPlaceMarks()
            for build_vertex in self._build_vertices:
                build_context.SetPlaceMark(build_vertex)
            self._instance.SetContext(build_context)
            self._build_vertices = []
            self._build_vertex = None
        self._instance.Send()
        self._reset()

    def clear(self):
        """Call this to clear any data added to the Context to prevent it from being sent to XMS."""
        self._reset()
        self._instance.SetContext(self._start_context)

    # Internal methods.
    def _select_uuid_context(self):
        """Lazily select to the SelectUuid node when needed."""
        if self._selectuuid is None:
            # SelectUuid is always available from the root.
            temp_context = self._instance.GetContext()
            temp_context.ClearPlaceMarks()
            temp_context.SetPlaceMark(self._root_instance)
            self._instance.Select('SelectUuid', a_context=temp_context)
            self._selectuuid = self._instance.GetContext()
        return self._selectuuid

    def _component_id_context(self):
        """Lazily select to the ComponentCoverageIds node when needed."""
        if self._componentcoverageids is None:
            # ComponentCoverageIds is always available from the root simulation or component.
            self._instance.Select('ComponentCoverageIds', a_context=self._start_context)
            self._componentcoverageids = self._instance.GetContext()
        return self._componentcoverageids

    def _selected_component_id_context(self):
        """Lazily select to the ComponentCoverageSelectedIds node when needed."""
        if self._componentcoverageselectedids is None:
            # ComponentCoverageSelectedIds is always available from the root simulation or component.
            self._instance.Select('ComponentCoverageSelectedIds', a_context=self._start_context)
            self._componentcoverageselectedids = self._instance.GetContext()
        return self._componentcoverageselectedids

    def _named_executables_context(self):
        """Lazily select to the NamedExecutables node when needed."""
        if self._namedexecutables is None:
            # ComponentCoverageIds is always available from the root simulation or component.
            self._instance.Select('NamedExecutables', a_context=self._start_context)
            self._namedexecutables = self._instance.GetContext()
        return self._namedexecutables

    def _ensure_build_vertex_exists(self):
        """Add a root build edge to the Context the first time we add something."""
        if self._build_vertex is None:  # We are starting a build
            if self._migrate_script:
                self._build_vertex = self._root_instance
            else:
                self._instance.SetContext(self._start_context)
                # Check for an existing build edge.
                build_children = self._instance.GetInstanceChildren(self._root_instance, 'Build')
                build_children.extend(self._instance.GetInstanceChildren(self._root_instance, 'BuildComponent'))
                if build_children:  # Already have a Build root edge, use that.
                    self._build_vertex = build_children[0]
                else:  # Add a Build edge from the ultimate root to the root model simulation.
                    self._build_vertex = self._instance.AddRootVertexInstance('Build', self._root_instance)

            if self._build_vertex is not None:
                self._build_vertices.append(self._build_vertex)
        return self._build_vertex

    def _add_hidden_component(self, component, owner_id, comp_keywords):
        """Adds a hidden component to a previously added simulation or coverage.

        Args:
            component (pydop.Component): The hidden component to add
            owner_id (int): Context instance id of the parent simulation of coverage
            comp_keywords (Union[None, dict]): Optional dict of additional keywords for the component, such as 'actions'
                or 'display_options'
        """
        unique_name, model_name = component.get_unique_name_and_model_name()
        arg_dict = {'#description': f'{model_name}#{unique_name}', '': component._instance}
        if comp_keywords:
            arg_dict.update(comp_keywords)
        self._build_vertices.extend(self._instance.Add([arg_dict], owner_id))

    def _move_context_to_child(
        self, model_name, unique_name, take_name, coverage_take, geometry_take, component_take, simulation_take
    ):
        """Move current Context position to a child

        Args:
            model_name (str): XML model name of a child hidden component
            unique_name (str): XML unique name of a child hidden component
            take_name (str): XML take name of a visible child component, coverage, geometry, or simulation
            coverage_take (bool): True if taken child is a coverage
            geometry_take (bool): True if taken child is a coverage
            component_take (bool): True if taken child is a component
            simulation_take (bool): True if taken child is a simulation)
        """
        if any([coverage_take, geometry_take, component_take, simulation_take]):
            # Child is an XML-taken item
            if take_name is None:
                raise ValueError('Must provide XML-defined take name of the child.')
            self._instance.Select(take_name)  # Move to the child take
            select_keyword = 'Coverage'  # DMI or generic coverage
            if geometry_take:
                select_keyword = 'Geometry'
            elif component_take:
                select_keyword = 'Component'
            elif simulation_take:
                select_keyword = 'Simulation'
        else:  # Child is a hidden component
            if model_name is None and unique_name is None:
                raise ValueError('Must provide XML-defined unique name and model name to get hidden component child.')
            select_keyword = f'{model_name}#{unique_name}'
        self._instance.Select(select_keyword)  # Move to specified child

    def _reset(self):
        """Clear all cached data."""
        self._build_vertex = None
        self._build_vertices = []
        self._model_check_errors = []
        self._unlink_items = []
        self._delete_items = []
        self._rename_items = []
        self._edit_elevations = []

    # We found that things added to the root instance of the Context need to be added last. Not sure why, but we'll
    # just cache them up and add them before sending the data to XMS instead of figuring it out.
    def _add_unlink_items(self):
        """Add the cached unlink requests. Needs to be done last for some reason."""
        for taker_uuid, taken_uuid in self._unlink_items:
            arg_list = [{'#description': 'Unlink', taker_uuid: taken_uuid}]
            self._build_vertices.extend(self._instance.Add(arg_list, self._root_instance))

    def _add_delete_items(self):
        """Add the cached delete requests. Needs to be done last for some reason."""
        for item_uuid in self._delete_items:
            arg_list = [{'#description': 'Delete', '': item_uuid}]
            self._build_vertices.extend(self._instance.Add(arg_list, self._root_instance))

    def _add_rename_items(self):
        """Add the cached rename requests. Needs to be done last for some reason."""
        for item_uuid, new_name in self._rename_items:
            arg_list = [{'#description': 'Delete', f'tree_item_rename#{new_name}': item_uuid}]
            self._build_vertices.extend(self._instance.Add(arg_list, self._root_instance))

    def _add_edit_elevations(self):
        """Add the cached UGrid top and bottom elevation edit requests. Needs to be done last for some reason."""
        for json_file in self._edit_elevations:
            arg_list = [{'#description': 'InplaceEdit', 'ugrid_top_bottom_elev': json_file}]
            self._build_vertices.extend(self._instance.Add(arg_list, self._root_instance))

    # Debug methods
    def get_children_ids(self, parent_id, relationship=''):
        """Debug method for getting the instance children of a specified instance id in the Query Context.

        Args:
            parent_id (int): Instance id of the node to retrieve children of
            relationship (str): If provided, only the children along edges with this name will be returned

        Returns:
            list of int: Child instance ids of the specified parent instance
        """
        return self._instance.GetInstanceChildren(parent_id, relationship)

    def get_children_names(self, parent_id):
        """Debug method for getting the out edge names of a specified instance id in the Query Context.

        Args:
            parent_id (int): Instance id of the node to retrieve out edge names of

        Returns:
            list of str: Edge names between the specified parent instance and its children
        """
        return self._instance.GetInstanceChildrenRelationships(parent_id)

    def get_possible_children_names(self, parent_id):
        """Debug method for getting the out edge names of a specified instance id in the Query Context.

        Args:
            parent_id (int): Instance id of the node to retrieve out edge names of

        Returns:
            list of str: See description
        """
        return self._instance.GetContext().GetOutEdges(self._instance.GetContextDefinition(), parent_id)

    def get_vertex_type(self, instance_id):
        """Debug method for getting the type enum (as an int) of an instance in the Query Context.

        Args:
            instance_id (int): Instance id of the node to retrieve type of

        Returns:
            int: The instance's C++ ContextDataType enum as an int
        """
        return self._instance.GetVertexType(instance_id)

    @property
    def playback_folder(self):
        """Returns the location of the playback folder if a QueryPlaybackImpl, else empty string."""
        return ''
