"""This module is for the simulation hidden component."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil
import uuid

# 2. Third party modules
import h5py
import numpy
import pandas
from PySide2.QtCore import QDir
from PySide2.QtWidgets import QFileDialog

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file
from xms.constraint.contour import UGrid2dContour
from xms.core.filesystem import filesystem as io_util
from xms.data_objects.parameters import Arc, Coverage, Dataset, Point
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg

# 4. Local modules
from xms.cmsflow.components.cmsflow_component import CmsflowComponent
from xms.cmsflow.components.coverage_mapper_runner import CoverageMapperRunner
from xms.cmsflow.data.simulation_data import SimulationData
from xms.cmsflow.gui.advanced_dlg import AdvancedDlg
from xms.cmsflow.gui.arc_at_contour_dlg import ArcAtContourDlg
from xms.cmsflow.gui.dredge_dlg import DredgeDlg
from xms.cmsflow.gui.model_control_dlg import ModelControlDlg


class SimComponent(CmsflowComponent):
    """A hidden Dynamic Model Interface (DMI) component for the CMS-Flow model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.data = SimulationData(self.main_file)
        self.data.commit()
        self.tree_commands = [
            ('Model Control...', 'open_model_control'),
            ('Advanced Cards...', 'open_advanced_cards'),
            ('Dredge Module Definition...', 'open_dredge_module'),
            ('Generate Arcs at Contour...', 'open_arc_at_contour'),
            ('Generate Snap Preview', 'create_snap_preview'),
        ]  # [(menu_text, menu_method)...]
        self.renumber_file = os.path.join(os.path.dirname(self.main_file), 'renumber.txt')
        if os.path.exists(self.renumber_file):
            self.tree_commands.append(('Migrate Legacy Solutions...', 'migrate_legacy_solutions'))
        self._count = 0

    def link_event(self, link_dict, lock_state):
        """This will be called when one or more coverages, ugrids, or other components are linked to this component.

        Args:
            link_dict (dict): A dictionary with keys being UUIDs as strings representing the objects being linked into
                this component. The values of this dictionary are a list of strings of the parameter names of the
                "takes" from the XML that this is a part of.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        messages = []
        actions = []
        for link_uuid, link_xml_params in link_dict.items():
            for xml_param in link_xml_params:
                if xml_param in ['grid', 'cgrid']:
                    if self.data.info.attrs['domain_uuid']:  # Already have a linked CGrid or Quadtree
                        params = {'old_uuid': self.data.info.attrs['domain_uuid']}
                        action = ActionRequest(
                            main_file=self.main_file,
                            modality='NO_DIALOG',
                            class_name=self.class_name,
                            module_name=self.module_name,
                            method_name='delete_old_domain',
                            comp_uuid=self.uuid,
                            parameters=params
                        )
                        actions.append(action)
                    self.data.info.attrs['domain_uuid'] = link_uuid
        self.data.commit()
        return messages, actions

    def unlink_event(self, unlinks, lock_state):
        """This will be called when a coverage, or a ugrid, or another component is unlinked from this component.

        Args:
            unlinks (list of str): A list of UUIDs as strings representing the objects being unlinked.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        if self.data.info.attrs['domain_uuid'] in unlinks:
            self.data.info.attrs['domain_uuid'] = ''
        self.data.commit()
        return [], []

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, UNLOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                    start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                    data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                    the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        # Update external file references if saving new project
        if save_type == 'SAVE':  # Update paths in existing main file
            error = self.data.update_file_paths()
            if error:
                messages.append(('ERROR', error))
        elif save_type in ['SAVE_AS', 'PACKAGE']:  # Update filepaths in the new main file.
            is_package = save_type == 'PACKAGE'
            if is_package:  # If doing a Save As Package copy all referenced external files.
                error = self.copy_external_files(new_main_file)
            else:
                error = self.update_proj_dir(new_main_file, True)
            if error:
                messages.append(('ERROR', error))
        elif save_type == 'UNLOCK':
            # SMS invokes this when it opens a project for the first time, but it doesn't seem like a good idea to
            # update anything since all we have is the main-file's path, which is in the temp directory. All the paths
            # stored in the component should already be relative to the actual project directory, so it's probably
            # okay to just leave them alone.
            pass

        return new_main_file, messages, action_requests

    @staticmethod
    def delete_old_domain(query, params):
        """Delete the existing linked domain when another is linked to the simulation.

        This is needed because we want to allow using an SMS 2DMesh or UGrid object as the domain. The xml has both a
        take_mesh2d and take_ugrid parameter with a limit of 1. If we get a link event for the domain and we already
        have one, we need to remove the old because it is of a different type

        Args:
            query (xms.api.dmi.Query.Query): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        old_uuid = None
        sim_uuid = None
        if params and params[0]:
            old_uuid = params[0].get('old_uuid')
            # Need the UUID of the parent tree item, even though it is the hidden component that actually has the XML
            # take parameter.
            sim_uuid = query.parent_item_uuid()
        if old_uuid and sim_uuid:
            query.unlink_item(taker_uuid=sim_uuid, taken_uuid=old_uuid)
        return [], []

    def open_model_control(self, query, params, win_cont):
        """Opens the Model Control dialog and saves component data state on OK.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        pe_tree = query.project_tree
        sim_uuid = query.parent_item_uuid()
        dlg = ModelControlDlg(self.data, pe_tree, sim_uuid, win_cont)
        if dlg.exec_():
            self.data.commit()
        return [], []

    def open_dredge_module(self, query, params, win_cont):
        """Opens the Dredge Module Definition dialog and saves component data state on OK.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        pe_tree = query.project_tree
        sim_uuid = query.parent_item_uuid()
        dlg = DredgeDlg(self.data, pe_tree, sim_uuid, win_cont)
        if dlg.exec_():
            self.data.commit()
        return [], []

    def open_advanced_cards(self, query, params, win_cont):
        """Opens the Advanced Cards dialog and saves component data state on OK.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        dlg = AdvancedDlg(self.data, win_cont)
        if dlg.exec_():
            self.data.commit()
        return [], []

    @staticmethod
    def create_snap_preview(query, params, win_cont):
        """Creates mapped components to display CMS-Flow data on a quadtree.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with SMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.

        """
        query.xms_agent.set_timeout(300000)
        note = ''
        worker = CoverageMapperRunner(query)
        error_str = 'Error(s) encountered applying coverages to simulation. Review log output for more details.'
        warning_str = 'Warning(s) encountered applying coverages to simulation. Review log output for more details.'
        display_text = {
            'title': 'CMS-Flow Snap Preview',
            'working_prompt': 'Applying coverages to quadtree. Please wait...',
            'error_prompt': error_str,
            'warning_prompt': warning_str,
            'success_prompt': 'Successfully created snap preview',
            'note': note,
            'auto_load': 'Close this dialog automatically when exporting is finished.'
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms', worker, win_cont)
        if feedback_dlg.exec() and not LogEchoQSignalStream.logged_error:
            worker.send()
        return [], []

    def migrate_legacy_solutions(self, query, params, win_cont):
        """Read in solutions that use the old numbering.

        SMS version 11.2 and prior used a different numbering for their quadtrees. As such, datasets need to be
        renumbered to map to the new grid and appear correctly in SMS.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict'): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        # Find the Quadtree linked to the simulation
        sim_item = tree_util.find_tree_node_by_uuid(query.project_tree, query.parent_item_uuid())
        quad_item = tree_util.descendants_of_type(
            sim_item, xms_types=['TI_UGRID_PTR', 'TI_CGRID2D_PTR'], recurse=False, allow_pointers=True, only_first=True
        )
        if not quad_item:
            return [('ERROR', 'Could not get quadtree to add datasets to.')], []

        # Get the name of the legacy solution that is being renumbered
        path = os.environ.get('XMS_PYTHON_APP_PROJECT_PATH', '')
        if not os.path.exists(path):
            path = os.path.join(QDir.homePath(), 'Documents')
        file_filter = 'HDF5 (*.h5)'
        dialog = QFileDialog(win_cont, 'Select File', path, file_filter)
        dialog.setLabelText(QFileDialog.Accept, 'Select')
        if dialog.exec_():
            # Get a temporary directory to put the new datasets in.
            # dataset_dir = os.path.join(query.xms_temp_directory, str(uuid.uuid4()))
            # os.makedirs(dataset_dir)

            # Create new solution files with renumbered datasets to location of the old files with a modified name.
            old_dataset_files = dialog.selectedFiles()
            new_dataset_files = []
            for dset_file in old_dataset_files:
                dataset_dir = os.path.dirname(os.path.realpath(dset_file))
                dataset_filename = os.path.basename(dset_file)
                renumbered_dataset_filename = 'Migrated_' + dataset_filename

                new_dataset_files.append(os.path.join(dataset_dir, renumbered_dataset_filename))
                shutil.copy(dset_file, new_dataset_files[-1])

            # Build a map for renumbering the datasets
            renumber = pandas.read_csv(self.renumber_file, header=None).values
            if renumber is None or renumber.size == 0:
                return [('ERROR', 'Could not get quadtree renumber map.')], []

            # Renumbering the datasets in the new file
            self._renumber_solution_datasets(new_dataset_files, old_dataset_files, quad_item.uuid, renumber)
            # AKZ ToDo - This created a flake error - trying it without the return value
            # dataset_args = self._renumber_solution_datasets(
            #    new_dataset_files, old_dataset_files, quad_item.uuid, renumber
            # )
        return [], []

    @classmethod
    def _renumber_solution_datasets(cls, new_dataset_files, old_dataset_files, quadtree_uuid, renumber):
        """Renumbers the dataset that is read in the old files and puts it in the new.

        This method assumes that the new_dataset_files were copied from the old_dataset_files.

        Args:
            new_dataset_files (:obj:`list` of str): A list of dataset files to write to.
            old_dataset_files (:obj:`list` of str): A list of dataset files to read from.
            quadtree_uuid (str): The UUID of the quadtree the datasets will belong to.
            renumber (:obj:`list` of :obj:`list` of int): New indexes in old numbering order. Inner lists always size 1.

        Returns:
            (:obj:`list`): A list of Dataset instances to be added to Query.
        """
        slice_index = len('/Values') * -1
        renumber_index = [0 for _ in renumber]
        # Was new index, in old tel file order. Change to old index in new index order.
        for old_index, new_index_list in enumerate(renumber):
            renumber_index[new_index_list[0]] = old_index
        dataset_args = []
        for dset_file, old_dset_file in zip(new_dataset_files, old_dataset_files):
            datasets_to_change = []
            with h5py.File(old_dset_file, 'r') as old_file:
                with h5py.File(dset_file, 'a') as file:
                    cls._find_datasets(datasets_to_change, file, '')
                    for dset in datasets_to_change:
                        old_dset = old_file[dset][...]
                        values = []
                        for timestep in old_dset:
                            values.append(timestep[renumber_index])
                        file[dset][...] = numpy.array(values)
                        # Build new dataset objects.
                        dataset = Dataset(dset_file, dset[:slice_index], 'CELL', 'NULL')
                        dataset.geom_uuid = quadtree_uuid
                        dataset_args.append(dataset)
                    file.flush()
        return dataset_args

    @classmethod
    def _find_datasets(cls, datasets_to_change, h5_object, h5_path):
        """Finds datasets named 'Values' inside of the h5 file.

        Args:
            datasets_to_change (:obj:`list` of  str): A list of h5 paths to datasets.
            h5_object (h5py.Group): The group or file that is to be parsed.
            h5_path (str): The h5 path of the h5_object.
        """
        for item in h5_object.keys():
            h5_dset_path = f'{h5_path}/{item}'
            if isinstance(h5_object[item], h5py.Dataset):
                if item.endswith('Values'):
                    datasets_to_change.append(h5_dset_path)
            else:
                cls._find_datasets(datasets_to_change, h5_object[item], h5_dset_path)

    @staticmethod
    def open_arc_at_contour(query, params, win_cont):
        """Creates an arc on a new coverage at a specific dataset value on a quadtree.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with SMS.
            params (:obj:`dict'): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        # Get the grid from the simulation
        sim_item = tree_util.find_tree_node_by_uuid(query.project_tree, query.parent_item_uuid())
        quad_item = tree_util.descendants_of_type(
            sim_item, xms_types=['TI_UGRID_PTR', 'TI_CGRID2D_PTR'], recurse=False, allow_pointers=True, only_first=True
        )
        if not quad_item:
            return [('ERROR', 'Could not find the geometry for the simulation.')], []
        quad_geom = query.item_with_uuid(quad_item.uuid)
        grid = read_grid_from_file(quad_geom.cogrid_file)

        pe_tree = tree_util.trim_project_explorer(query.project_tree, quad_item.uuid)
        dialog = ArcAtContourDlg(pe_tree, query)
        if dialog.exec_():
            contour_value = float(dialog.value_edit.text())
            contour_uuid = dialog.get_selected_item_uuid()
            contours = UGrid2dContour(grid.ugrid)
            # Get the grid cell/point scalars from self.wse_uuid
            if contour_uuid:
                dataset = query.item_with_uuid(contour_uuid)
                if not dataset:
                    return [('ERROR', 'Could not find the selected dataset.')], []

                contour_ts_data = dataset.values[dialog.get_selected_time_step_index()]
                if dataset.null_value is not None:
                    contours.set_no_data_value(dataset.null_value)
                # Determine the location at which the data will be stored.
                if dataset.location == 'points':
                    contours.set_grid_point_scalars(contour_ts_data, [], 0)
                else:
                    contours.set_grid_cell_scalars(contour_ts_data, [], 1)
                contours.set_extract_scalars([contour_value])
                contours.set_contour_length_threshold(0.0)
                lines = contours.extract_contour_segments()

                arcs = []
                pt_id = 1
                for line_idx, line in enumerate(lines[contour_value]):
                    pt_1 = Point(*line[0], feature_id=pt_id)
                    pt_id += 1
                    pt_2 = Point(*line[-1], feature_id=pt_id)
                    pt_id += 1
                    vertices = [Point(*pt) for pt in line[1:-1]]
                    arc = Arc(start_node=pt_1, end_node=pt_2, vertices=vertices, feature_id=line_idx + 1)
                    arcs.append(arc)

                cov = Coverage(name=f'Contour {contour_value}', uuid=str(uuid.uuid4()))
                cov.arcs = arcs
                cov.complete()
                query.add_coverage(cov)
        return [], []

    def update_proj_dir(self, new_main_file, convert_filepaths):
        """Called when saving a project for the first time or saving a project to a new location.

        All referenced filepaths should be converted to relative from the new project location. If the file path is
        already relative, it is relative to the old project directory. After updating file paths, update the project
        directory in the main file.

        Args:
            new_main_file (str): The location of the new main file.
            convert_filepaths (bool): False if only the project directory should be updated.

        Returns:
            (str): Message on failure, empty string on success

        """
        new_data = SimulationData(new_main_file)
        if not convert_filepaths:
            # This case is to handle opening a package project for the first time.
            comp_folder = os.path.dirname(self.main_file)
            package_proj_dir = os.path.normpath(os.path.join(comp_folder, '../../..'))
            new_data.info.attrs['proj_dir'] = package_proj_dir
            new_data.commit()  # Save the updated project directory
            return ''
        err_msg = new_data.update_proj_dir()
        # Copy the newly saved file to temp.
        io_util.copyfile(new_main_file, self.main_file)
        return err_msg

    @staticmethod
    def copy_external_files(new_main_file):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Args:
            new_main_file (str): The location of the new component main file in the package.

        Returns:
            (str): Message on failure, empty string on success

        """
        new_data = SimulationData(new_main_file)
        new_data.copy_external_files()
        new_data.commit()
        return ''
