"""Map locations and attributes of applicable linked coverages to the CMS-Flow domain."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint.ugrid_builder import UGridBuilder
from xms.grid.ugrid import UGrid
from xms.snap.snap_polygon import SnapPolygon

# 4. Local modules
from xms.cmsflow.mapping.bc_mapper import BcMapper
from xms.cmsflow.mapping.rm_structures_mapper import RmStructuresMapper
from xms.cmsflow.mapping.save_points_mapper import SavePointsMapper


class CoverageMapper:
    """Class for mapping coverages to a quadtree for CMS-Flow."""
    def __init__(self, generate_snap):
        """Constructor.

        Args:
            generate_snap (bool): True if generating components for snap preview.
        """
        super().__init__()
        self._logger = logging.getLogger('xms.cmsflow')
        self._generate_snap = generate_snap
        self._quadtree = None
        self._quadtree_ugrid = None
        self._small_quadtree = None
        self._bc_cov = None
        self._rm_cov = None
        self._save_pts_cov = None
        self._bc_comp = None
        self._rm_comp = None
        self._save_pts_comp = None
        self._cell_activity = None
        self._wkt = ''

        self.mapped_comps = []
        self.small_id_to_original_id = []
        self.save_pt_att_id_to_cell_id = {}

    @property
    def quadtree_ugrid(self):
        """Cache unwrapping of the Python XmUGrid."""
        if self._quadtree_ugrid is None and self._quadtree is not None:
            self._quadtree_ugrid = self._quadtree.ugrid
        return self._quadtree_ugrid

    def check_mappable_items(self):
        """Check for mappable items."""
        if not self._bc_cov and not self._rm_cov and not self._save_pts_cov and not self._cell_activity:
            return False
        return True

    def set_quadtree(self, quadtree, wkt):
        """Sets the quadtree geometry of the domain.

        Args:
            quadtree (CoGrid): A quadtree geometry of the domain.
            wkt (str): The 'well known text' of the projection.
        """
        self._quadtree = quadtree
        self._wkt = wkt

    def get_quadtree(self):
        """Get the quadtree geometry of the domain."""
        return self._quadtree

    def get_cell_activity(self):
        """Get the activity out of the coverage mapper."""
        return self._cell_activity

    def set_activity(self, activity_cov):
        """Sets the active areas of the domain.

        This method assumes that a valid quadtree has already been set.

        Args:
            activity_cov (Coverage): An activity coverage.

        Returns:
            Union[CoGrid, None]: The small quadtree built from active cells, or None if no small grid built (every cell
                is active).
        """
        if not self._quadtree:
            return None
        cell_count = self.quadtree_ugrid.cell_count
        if cell_count < 1:
            return None

        if activity_cov:  # Activity defined on a coverage (13.1 and earlier)
            self._cell_activity = [True for _ in range(cell_count)]
            self._set_activity_from_coverage(activity_cov)
        elif self._quadtree.model_on_off_cells:
            self._cell_activity = list(self._quadtree.model_on_off_cells)
        elif self._quadtree.cell_elevations:
            self._cell_activity = [
                1 if not math.isclose(elev, -999.0, abs_tol=1e-10) else 0 for elev in self._quadtree.cell_elevations
            ]
        else:
            self._cell_activity = [True for _ in range(cell_count)]  # Guess everyone is active
            return None
        self._small_quadtree = self._build_small_grid()
        return self._small_quadtree

    def set_boundary_conditions(self, boundary_conditions, component):
        """Sets the boundary conditions coverage.

        Args:
            boundary_conditions (Coverage): A boundary conditions coverage.
            component (BcComponent): A boundary conditions component belonging to the coverage.
        """
        self._bc_cov = boundary_conditions
        self._bc_comp = component

    def set_save_points(self, coverage, component):
        """Sets the rubble mound structures coverage.

        Args:
            coverage (Coverage): A Save Points coverage.
            component (SavePointsComponent): A Save Points component belonging to the coverage.
        """
        self._save_pts_cov = coverage
        self._save_pts_comp = component

    def set_rubble_mound(self, rubble_mound, component):
        """Sets the rubble mound structures coverage.

        Args:
            rubble_mound (Coverage): A rubble mound jetties coverage.
            component (RMStructuresComponent): A rubble mound jetties component belonging to the coverage.
        """
        self._rm_cov = rubble_mound
        self._rm_comp = component

    def do_map(self):
        """Performs the snapping of the coverages to the domain."""
        try:
            self._map_boundary_conditions()
            self._map_rubble_mound()
            self._map_save_points()
        except:  # pragma: no cover  # noqa
            self._logger.exception('Error generating snap.')  # pragma: no cover

    def get_bc_arc_snapper(self):
        """Gets a snapper for boundary condition arcs.

        Returns:
            A snapper for boundary condition arcs.
        """
        mapper = BcMapper(self, self._generate_snap)
        return mapper.snapper

    def get_save_points_snapper(self):
        """Gets a snapper for Save Points coverage.

        Returns:
            A snapper for Save Points coverage.
        """
        mapper = SavePointsMapper(self, self._generate_snap)
        return mapper.snapper

    def get_rm_poly_snapper(self):
        """Gets a snapper for rubble mound jetty polygons.

        Returns:
            A snapper for rubble mound jetty polygons.
        """
        mapper = RmStructuresMapper(self, self._generate_snap)
        return mapper.snapper

    def get_snap_id_to_original_id(self):
        """Gets the ids of the original quadtree in a list that is as long as the number of cells in the small quadtree.

        Returns:
            A list of original quadtree cell ids.
        """
        if self.small_id_to_original_id:
            return self.small_id_to_original_id
        else:
            return [idx for idx in range(self.quadtree_ugrid.cell_count)]

    def _build_small_grid(self):
        """Creates a potentially smaller geometry of only the active cells.

        Returns:
            A CoGrid with only smaller cells in it.
        """
        # If there is no activity, then return the whole quadtree unchanged.
        if not self._cell_activity:
            return self._quadtree
        cell_stream = self.quadtree_ugrid.cellstream
        # All elements should have a cell type, a cell point count, and 4-8 point ids
        split_cell_stream = []
        idx = 0
        while idx < len(cell_stream):
            # Add one for the cell point count and add however many points are present
            # Add one more to get past the last point
            new_idx = idx + cell_stream[idx + 1] + 2
            split_cell_stream.append(cell_stream[idx:new_idx])
            idx = new_idx
        if len(split_cell_stream) != len(self._cell_activity):
            raise Exception('Unable to create a grid of active cells.')
        small_cell_stream = []
        # Only add back in cells that are active.
        for idx, cell in enumerate(split_cell_stream):
            if self._cell_activity[idx]:
                # Check if this an isolated cell. A cell is considered isolated by CMS-Flow if no other active cells
                # share an edge. Point adjacency is not enough.
                found_neighbor = False
                for edge_idx in range(self.quadtree_ugrid.get_cell_edge_count(idx)):
                    for adj_cell_idx in self.quadtree_ugrid.get_cell_edge_adjacent_cells(idx, edge_idx):
                        if self._cell_activity[adj_cell_idx]:
                            # This neighbor shares an edge and is active, we are good to add the cell.
                            small_cell_stream.extend(cell)
                            self.small_id_to_original_id.append(idx)
                            found_neighbor = True
                            break
                    if found_neighbor:
                        break
                else:  # This is an isolated ocean cell. Log a warning and make it inactive.
                    self._logger.warning(f'Isolated ocean cell found. ID = {idx + 1}. This cell will be deactivated.')
                    self._cell_activity[idx] = 0

        # Build the new grid.
        xmugrid = UGrid(self.quadtree_ugrid.locations, small_cell_stream)
        co_builder = UGridBuilder()
        co_builder.set_is_2d()
        co_builder.set_ugrid(xmugrid)
        small_grid = co_builder.build_grid()
        return small_grid

    def _set_activity_from_coverage(self, activity_cov):
        """Map activity coverage polygons to the Quadtree.

        Args:
            activity_cov (Coverage): An activity coverage.
        """
        snap = SnapPolygon()
        snap.set_grid(self._quadtree, True)
        polygons = activity_cov.m_cov.polygons
        snap.add_polygons(polygons)
        for idx in range(len(activity_cov.m_activity)):
            if not activity_cov.m_activity[idx]:
                cells = snap.get_cells_in_polygon(idx)
                for cell in cells:
                    self._cell_activity[cell] = False  # mark as inactive

    def _map_boundary_conditions(self):
        """Maps the boundary conditions coverage onto the domain."""
        if not self._bc_cov:
            return
        self._logger.info('Mapping boundary conditions coverage to quadtree.')
        mapper = BcMapper(self, generate_snap=self._generate_snap)
        do_comp, comp = mapper.do_map()
        if do_comp is not None:
            self.mapped_comps.append((do_comp, [comp.get_display_options_action()]))
        self._logger.info('Finished mapping boundary conditions coverage to quadtree.')

    def _map_rubble_mound(self):
        """Maps the boundary conditions coverage onto the domain."""
        if not self._rm_cov:
            return
        self._logger.info('Mapping rubble mound jetties coverage to quadtree.')
        mapper = RmStructuresMapper(self, generate_snap=self._generate_snap)
        do_comp, comp = mapper.do_map()
        if do_comp is not None:
            self.mapped_comps.append((do_comp, [comp.get_display_options_action()]))
        self._logger.info('Finished rubble mound jetties coverage to quadtree.')

    def _map_save_points(self):
        """Maps the Save Points coverage onto the domain."""
        if not self._save_pts_cov:
            return
        self._logger.info('Mapping Save Points coverage to quadtree.')
        mapper = SavePointsMapper(self, generate_snap=self._generate_snap)
        do_comp, comp = mapper.do_map()
        if do_comp is not None:
            self.mapped_comps.append((do_comp, [comp.get_display_options_action()]))
        self._logger.info('Finished mapping Save Points coverage to quadtree.')
