"""StructuresData class."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
import pkg_resources
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem
from xms.guipy.data.polygon_texture import PolygonTexture

# 4. Local modules

DEFAULT_STRUCTURES_COLORS = [
    (255, 0, 0),  # red
    (0, 0, 255),  # blue
    (0, 255, 51),  # green
    (255, 204, 0),  # yellow
    (0, 204, 255),  # cyan
    (255, 0, 204),  # magenta
    (153, 0, 255),  # purple
    (255, 102, 0),  # orange
    (51, 153, 51),  # dark green
    (153, 102, 0),  # brown
    (102, 153, 153),  # grey blue
    (153, 51, 51),  # dark red
    (255, 153, 255),  # light magenta
]


def struct_type_id_from_name(struct_name):
    """Returns the type id from the structure name.

    Args:
        struct_name (:obj:`str`): name of the structure type

    Returns:
        (:obj:`int`): the integer associated with the structure type
    """
    struct_name_lower = struct_name.lower()
    if struct_name_lower == 'bathymetry modification':
        return 1
    elif struct_name_lower == 'wave runup':
        return 2
    elif struct_name_lower == 'floating breakwater':
        return 3
    elif struct_name_lower == 'wall breakwater':
        return 4
    elif struct_name_lower == 'rubble-mound':
        return 5
    elif struct_name_lower == 'highly permeable':
        return 6
    elif struct_name_lower == 'semi-permeable':
        return 7
    else:
        return 1  # Used to be "unassigned". Now "bathymetry modification" is the default


class StructuresData(XarrayBase):
    """Class for storing the Structure Coverages Structure properties."""
    UNASSIGNED_STRUCT = 0  # The unassigned structure id (and row in structures list)
    # constants for columns in the structures data set
    COL_ID = 0
    COL_COLOR = 1
    COL_NAME = 2
    COL_CBX_TYPE = 3
    COL_TOG_MOD = 4
    COL_TXT_MOD = 5
    COL_EDT_MOD = 6

    def __init__(self, filename):
        """Constructor.

        Args:
            filename (:obj:`str`): file name
        """
        super().__init__(filename)
        self.info.attrs['FILE_TYPE'] = 'STRUCTURE_COVERAGES_STRUCTURE_DATA'
        # This migration needs to happen before loading the structures. Will blow while getting default display options
        if 'next_struct_color' not in self.info.attrs:
            self.info.attrs['next_struct_color'] = -1
        self._structures = None  # structures data sets
        self._migrate()
        self.close()

    def _migrate(self):
        """Migrate old data to be compatible with the current version."""
        if 'cov_uuid' not in self.info.attrs:
            self.info.attrs['cov_uuid'] = ''  # gets set later
        if 'display_uuid' not in self.info.attrs:
            self.info.attrs['display_uuid'] = ''
        # We used to have an unassigned type, but it didn't make sense. Change to 'Bathymetry modification'.
        self.structures['type'] = xr.where(
            self.structures['type'] == 'Unassigned', 'Bathymetry modification', self.structures['type']
        )

    @property
    def structures(self):
        """Get the structure component data parameters.

        Returns:
            (:obj:`xarray.Dataset`): The structure list dataset

        """
        if self._structures is None:
            self._structures = self.get_dataset('structures', False)
            if self._structures is None:
                self._structures = self._default_structures()
        return self._structures

    @structures.setter
    def structures(self, dset):
        """Setter for the structures dataset."""
        if dset:
            self._structures = dset

    def default_data_dict(self):
        """Returns a dict of default values for a new structure."""
        return {
            'id': [0],
            'color': self._get_structure_poly_display(),
            'name': ['bathymetry modification'],
            'type': ['Bathymetry modification'],
            'use_mod': [0],
            'mod_type': [''],
            'mod_val': [0.0]
        }

    def _default_structures(self):
        """Creates a default structure data set.

        Returns:
            (:obj:`xarray.Dataset`): The structure list dataset
        """
        return pd.DataFrame(self.default_data_dict()).to_xarray()

    def add_structures(self, struct_data):
        """Adds the structures from struct_data to this instance of StructuresData.

        Args:
            struct_data (StructuresData): another Structure Data instance

        Returns:
            (:obj:`dict`): The old ids of the struct_data as key and the new ids as the data
        """
        struct_df = self.structures.to_dataframe()
        struct_names = set(struct_df['name'].to_list())
        struct_data_df = struct_data.structures.to_dataframe()
        old_to_new_ids = {0: 0}  # UNASSIGNED_STRUCT must always be in the dict
        next_struct_id = max(struct_df['id']) + 1
        next_idx = len(struct_df)
        for i in range(1, len(struct_data_df)):  # start at 1 to skip UNASSIGNED_STRUCT
            struct_id = struct_data_df.iloc[i, StructuresData.COL_ID]
            new_struct_id = next_struct_id
            next_struct_id += 1
            old_to_new_ids[struct_id] = new_struct_id
            struct_data_df.iloc[i, StructuresData.COL_ID] = new_struct_id
            struct_name = struct_data_df.iloc[i, StructuresData.COL_NAME]
            cnt = 1
            orig_struct_name = struct_name
            while struct_name in struct_names:
                struct_name = f'{orig_struct_name} ({cnt})'
                cnt += 1
            struct_data_df.iloc[i, StructuresData.COL_NAME] = struct_name
            struct_df.loc[next_idx] = struct_data_df.values[i]
            next_idx += 1
        self._structures = struct_df.to_xarray()
        return old_to_new_ids

    @staticmethod
    def struct_name_from_type_id(struct_type):
        """Returns the type id from the structure name.

        Args:
            struct_type(:obj:`int`): the integer associated with the structure type
        Returns:
            (:obj:`str`): name of the structure type
        """
        if struct_type == 1:
            return 'Bathymetry modification'
        elif struct_type == 2:
            return 'Wave runup'
        elif struct_type == 3:
            return 'Floating breakwater'
        elif struct_type == 4:
            return 'Wall breakwater'
        elif struct_type == 5:
            return 'Rubble-mound'
        elif struct_type == 6:
            return 'Highly permeable'
        elif struct_type == 7:
            return 'Semi-permeable'
        else:
            return 'Bathymetry modification'  # Default with a value of 0.0

    def _get_structure_poly_display(self):
        """Get a set of randomized polygon fill display options.

        Returns:
            (str): The randomized display options as a string. Formatted as 'R G B texture', where R, G, B,
            and texture are integers.
        """
        # Check if this structure already has display attributes specified.
        if self.info.attrs['next_struct_color'] == -1:
            self.info.attrs['next_struct_color'] += 1
            return '0 0 0 1'

        if self.info.attrs['next_struct_color'] >= len(DEFAULT_STRUCTURES_COLORS):
            self.info.attrs['next_struct_color'] = 0
        clr = DEFAULT_STRUCTURES_COLORS[self.info.attrs['next_struct_color']]
        clr_str = f'{clr[0]} {clr[1]} {clr[2]} {int(PolygonTexture.null_pattern)}'
        self.info.attrs['next_struct_color'] += 1
        return clr_str

    def struct_id_from_info(self, info_type_id, info_use_mod, info_mod_val):
        """Returns the type id from the structure info.

        Args:
            info_type_id (:obj:`int`): structure type id
            info_use_mod (:obj:`bool`): if the structure is using a modification
            info_mod_val (:obj:`float`): value for the modification

        Returns:
            (:obj:`int`): the structure id
        """
        struct_df = self.structures.to_dataframe()
        for i in range(1, len(struct_df)):  # start at 1 to skip UNASSIGNED_STRUCT
            struct_type_str = struct_df.iloc[i, StructuresData.COL_CBX_TYPE]
            type_id = struct_type_id_from_name(struct_type_str)
            if type_id != info_type_id:
                continue
            use_mod = struct_df.iloc[i, StructuresData.COL_TOG_MOD]
            if bool(use_mod) != info_use_mod:
                continue
            if use_mod:
                mod_val = struct_df.iloc[i, StructuresData.COL_EDT_MOD]
                if mod_val != info_mod_val:
                    continue
            # If we got here, it's a match, so return the id
            return i

        # If we got here, it's a new structure, so add it
        new_struct_id = max(struct_df['id']) + 1
        data = {
            'id': [new_struct_id],
            'color': [self._get_structure_poly_display()],
            'name': [f's{new_struct_id}'],
            'type': [self.struct_name_from_type_id(info_type_id)],
            'use_mod': [int(info_use_mod)],
            'mod_type': [''],
            'mod_val': [info_mod_val]
        }
        new_row = pd.DataFrame(data)

        struct_df = pd.concat([struct_df, new_row]).reset_index(drop=True)
        self._structures = struct_df.to_xarray()
        return new_struct_id

    def commit(self):
        """Save in memory datasets to the NetCDF file."""
        self.info.attrs['VERSION'] = pkg_resources.get_distribution('xmscmswave').version
        filesystem.removefile(self._filename)  # Always vacuum the file. Structures dataset is small enough.
        super().commit()
        self.structures.to_netcdf(self._filename, group='structures', mode='a')

    def close(self):
        """Closes the H5 file and does not write any data that is in memory."""
        super().close()
        if self._structures is not None:
            self._structures.close()
