"""Execute a script in a parallel process."""

# 1. Standard Python modules
import os

# 2. Third party modules
import orjson

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xfs

# 4. Local modules
from xms.cmswave.data import cmswave_consts as const

DSET_SCRIPT_ARGS_JSON = 'write_dsets.json'


def write_dataset_script_args(query, cogrid_file, sim_data, sim_name):
    """Write a JSON file with the argument required to execute the spatial dataset export script.

    Args:
        query (:obj:`Query`): The XMS interprocess communicator
        cogrid_file (:obj:`str`): Absolute path to the simulation's CoGrid file
        sim_data (:obj:`SimulationData`): The simulation data
        sim_name (:obj:`str`): Name of the simulation being exported

    Returns:
        (:obj:`str`): Absolute path to the JSON file
    """
    json_data = {
        'sim_data': sim_data._filename,
        'sim_name': sim_name,
        'cogrid_file': cogrid_file,
    }
    # Check if we have currents
    use_curr = sim_data.info.attrs['current_interaction']
    if use_curr == "Use dataset":
        dset = query.item_with_uuid(sim_data.info.attrs['current_uuid'])
        if dset:
            json_data['curr'] = dset.h5_filename, dset.group_path

    # Check if we have wind data
    terms = sim_data.info.attrs['source_terms']
    if terms != "Propagation only":  # not propagation only
        use_const_wind = sim_data.info.attrs['wind']
        if use_const_wind != "Constant value":  # varying wind values
            dset = query.item_with_uuid(sim_data.info.attrs['wind_uuid'])
            if dset:
                json_data['wind'] = dset.h5_filename, dset.group_path

    # Check if we have surge data
    tidal_method = sim_data.info.attrs['surge']
    if tidal_method == const.TEXT_USE_DATASET:  # variable tidal surge values
        dset = query.item_with_uuid(sim_data.info.attrs['surge_uuid'])
        if dset:
            json_data['surge'] = dset.h5_filename, dset.group_path

    # Check if we have friction data
    use_fric = sim_data.info.attrs['friction']
    if use_fric == "Darcy-Weisbach dataset":
        dset = query.item_with_uuid(sim_data.info.attrs['darcy_uuid'])
        if dset:
            json_data['friction'] = dset.h5_filename, dset.group_path
    elif use_fric == "Manning dataset":
        dset = query.item_with_uuid(sim_data.info.attrs['manning_uuid'])
        if dset:
            json_data['friction'] = dset.h5_filename, dset.group_path

    # Check if we have forward reflection data
    use_forward_reflection = sim_data.info.attrs['forward_reflection']
    if use_forward_reflection != "Constant value":  # variable forward reflection values
        dset = query.item_with_uuid(sim_data.info.attrs['forward_reflection_uuid'])
        if dset:
            json_data['forward_reflection'] = dset.h5_filename, dset.group_path

    # Check if we have backward reflection data
    use_backward_reflection = sim_data.info.attrs['backward_reflection']
    if use_backward_reflection != "Constant value":  # variable backward reflection values
        dset = query.item_with_uuid(sim_data.info.attrs['backward_reflection_uuid'])
        if dset:
            json_data['backward_reflection'] = dset.h5_filename, dset.group_path

    # write muddy bed file
    use_muddy_bed = sim_data.info.attrs['muddy_bed']
    if use_muddy_bed != "None":  # variable muddy bed
        dset = query.item_with_uuid(sim_data.info.attrs['muddy_bed_uuid'])
        if dset:
            json_data['muddy_bed'] = dset.h5_filename, dset.group_path

    with open(os.path.join(os.getcwd(), DSET_SCRIPT_ARGS_JSON), 'wb') as file:
        data = orjson.dumps(json_data)
        file.write(data)


def read_dataset_script_args():
    """Write a JSON file with the argument required to execute the spatial dataset export script.

    Returns:
        (:obj:`dict`): The arguments dict read from the JSON file
    """
    filename = os.path.join(os.getcwd(), DSET_SCRIPT_ARGS_JSON)
    with open(filename, 'rb') as file:
        data = orjson.loads(file.read())
    xfs.removefile(filename)
    return data
