"""Module for ComponentWithMenusBase class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['ComponentWithMenusBase', 'MenuCallback', 'MessagesAndRequests']

# 1. Standard Python modules
from pathlib import Path
from typing import Callable, final, Iterable, Optional, TypeAlias, Union
import warnings

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Menu, MenuItem, Query, XmsEnvironment as XmEnv
from xms.core.filesystem import filesystem as io_util

# 4. Local modules
from xms.components.bases.coverage_component_base import CoverageComponentBase

#: Type hint for the return value of methods that return messages.
#:
#: When a method returns messages, XMS will handle each message by displaying a message box for the message. Each
#: message is a tuple of `(level, text)`.
#:
#: `level` is the severity level. It controls the icon displayed on the message box. It can be one of 'DEBUG', 'ERROR',
#: 'WARNING', or 'INFO'.
#:
#: `text` is the actual text to display.
Messages: TypeAlias = list[tuple[str, str]]

#: Type hint for list of action requests.
#:
#: When a method returns Requests, XMS will handle each request by calling the method in the request.
Requests: TypeAlias = list[ActionRequest]

#: Type hint for the return value of methods that return messages and action requests.
#:
#: See `Messages` and `Requests` above.
MessagesAndRequests: TypeAlias = tuple[Messages, Requests]

#: Type hint for methods that handle menu items.
#:
#: Args:
#:     query: Interprocess communication object.
#:     params: A list 0 or 1 dicts. If 1 dict, the dict may or may not have a 'selection' key. If it does, the key's
#:         value is a list of selected feature IDs. Why it's a list is beyond me.
#:     parent: Parent widget to use when creating windows.
#:
#: Returns:
#:     Tuple of `(messages, requests)`. `messages` is a list of `(level, text)`, where `level` is one of 'DEBUG',
#:     'ERROR', 'WARNING', or 'INFO' and controls the icon shown in the dialog XMS shows. `text` is the text to show.
#:     `requests` is a list of further action requests for XMS to call after the method returns.
MenuCallback: TypeAlias = Callable[[Query, list[dict], QWidget], Optional[MessagesAndRequests]]

#: Type hint for a menu command. A command includes a label and some kind of handler. If the handler is a callable that
#: is compatible with the MenuCallback type hint, then the (label, MenuCallback) pair will result in a menu item with
#: the given label that calls the given handler when clicked. If the handler is a list instead, it will result in a
#: submenu, and the contents of the list will be commands on the submenu.
Command: TypeAlias = Union[tuple[str, MenuCallback], tuple[str, Iterable['Command']]]

#: XMS runs the same event on duplication that it does on receiving a file from an import script. The handling for these
#: events is different, so we create this file in the component's directory to signal the difference.
duplicate_event_file = 'run_duplicate_handler_component_was_duplicated'


class ComponentWithMenusBase(CoverageComponentBase):
    """
    A basic component with minimal functionality.

    This component implements some common features on top of ComponentBase:

    - self.uuid, self.class_name and self.module_name are all initialized for you.
    - If main_file is None, a new main-file will be generated. Its name will be based on the XML for this component.
    - Tree commands can be added by appending to self.tree_commands. Menus will be automatically created.
    - Create event handlers can be appended to self._create_event_handlers, which has a simpler API and is less
      error-prone (it's harder to forget to call the base class version).
    """
    def __init__(self, main_file: Path | str = None):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        if main_file is None:
            # This is typical when building a component from Python, e.g. model native reads or mapped coverages. It
            # might be a bit slow since it involves parsing some XML files, but in most cases there are just a small
            # number of files, they're small, we only do this a few times at once, we're already accessing a bunch of
            # other files anyway, and we already have a feedback dialog going. So in practice it's fast enough.
            #
            # There's a circular dependency here and the alternative is preferred, so we'll import and warn here.
            warnings.warn(
                'Constructing a component without a path is deprecated. '
                'Use xms.components.component_builders.main_file_maker instead.',
                DeprecationWarning,
                stacklevel=2
            )
            from xms.components.component_builders.main_file_maker import make_main_file
            main_file = make_main_file(self)

        super().__init__(main_file)

        #: Commands to show in the project explorer tree when right-clicking the coverage. The first element of each
        #: tuple is the text to display, and the second is the method (on `self`) to call when the item is clicked.
        #:
        #: Menu items are displayed in the reverse of the order they appear in this list. If each class appends items
        #: to the end of the list, then the most derived one's items will appear first. Since users tend to interact
        #: with the most derived one's items the most, this usually makes life more convenient for the user.
        #:
        #: It is possible to make submenus by making the second element of the tuple a sequence instead. Each element
        #: of the sequence should be another list of commands, the same as this member. This example shows making a
        #: three level deep menu:
        #:
        #: self.tree_commands.append((
        #:     'level 1', (
        #:         ('level 2, item 1', self.level_2_item_1),
        #:         ('level 2, item 2', self.level_2_item_2),
        #:         ('level 2 submenu', (
        #:             ('level 3, item 1', self.level_3_item_1),
        #:             ('level 3, item 2', self.level_3_item_2),
        #:         ))
        #:     )
        #:  ))
        #:
        #: Submenus with only one item in them are folded into the parent instead. This yields a single menu item named
        #: 'level 1 - level 2, item 1`, rather than a submenu named `level 1` with an item named `level 2, item 1`:
        #:
        #: self.tree_commands.append((
        #:     'level 1', [
        #:         ('level 2, item 1', self.level_2_item_1)
        #:     ]
        #: ))
        self.tree_commands: list[Command] = []

        #: The handler to call when the user double-clicks the tree item.
        #:
        #: If specified, this method will be called whenever the user double-clicks the component's tree item (or that
        #: of whatever it's attached to, in the case of hidden components).
        #:
        #: If None (the default), then the first menu item (as shown to the user) will be used as the handler instead.
        #:
        #: Note that the menu is displayed to the user in the reverse of the order in self.tree_commands to put the most
        #: derived component's items first, so the handler chosen when this is None will be the one closest to the end
        #: of self.tree_commands.
        #:
        #: Also note that submenus are skipped when searching for a handler. If the item at the end of
        #: self.tree_commands is a submenu, then it and all of its children will be skipped in favor of the next one.
        #: If everything is skipped, then an error will be logged to the process temp folder (since this is probably a
        #: bug) and the component will suppress any further response.
        #:
        #: If the component isn't supposed to respond to double-clicks, this can be made explicit by adding a
        #: `double_click="false"` attribute to the component's `<component_menus>` tag in the XML. This will also
        #: suppress the error logging mentioned above.
        self.default_tree_command: Optional[MenuCallback] = None

        #: Messages that XMS should display to the user after an event finishes. Event handlers can append messages to
        #: this list. Each element of the list is a tuple of `(level, text)`. `level` controls the icon XMS shows in the
        #: message box it displays for the message. Allowed options are  'DEBUG', 'ERROR', 'WARNING', 'INFO'. `text` is
        #: the message to show in the message box.
        self.messages: Messages = []

        #: Requests for XMS to call additional methods after the current one completes. Event handlers can append
        #: requests to this list.
        self.requests: Requests = []

        #: Methods to call when the component is "created". A component is created when XMS itself creates the coverage
        #: it's attached to via the New Coverage menu item in the project explorer.
        #:
        #: Each callback will be passed a `Query` for interprocess communication. Handlers are called in the order they
        #: appear in the list. Handlers should generally only be appended to the list, as earlier ones may be
        #: responsible for establishing preconditions expected by later ones. The return value of any handler is
        #: ignored. Handlers that want to send messages or action requests back to XMS should see `self.messages` and
        #: `self.requests`, respectively.
        self._create_event_handlers: list[Callable[[Query], None]] = []

        #: Methods to call when the component is "initialized". A component is initialized when XMS receives it from
        #: an import script.
        #:
        #: Each callback will be passed a `Query` for interprocess communication. Handlers are called in the order they
        #: appear in the list. Handlers should generally only be appended to the list, as earlier ones may be
        #: responsible for establishing preconditions expected by later ones. The return value of any handler is
        #: ignored. Handlers that want to send messages or action requests back to XMS should see `self.messages` and
        #: `self.requests`, respectively.#:
        #:
        #: ***REQUIRED PRECONDITION***: The component *must* have the `use_display="true"` attribute on its
        #: `<component>` tag in the XML. If the attribute is missing or `false`, XMS won't raise the event this depends
        #: on, so there's no way for Python to know the component was just initialized. So far though, the only user is
        #: the VisibleCoverageComponentBase, which requires `use_display="true"` in order to have display options, so
        #: this limitation hasn't been an issue yet.
        self._initialize_event_handlers: list[Callable[[Query], None]] = []

        #: Methods to call when the component is "duplicated". A component is duplicated when the Duplicate action is
        #: invoked from the right-click menu of the coverage the component is attached to. Event handlers are run on the
        #: new component.
        #:
        #: Each callback will be passed a `Query` for interprocess communication. Handlers are called in the order they
        #: appear in the list. Handlers should generally only be appended to the list, as earlier ones may be
        #: responsible for establishing preconditions expected by later ones. The return value of any handler is
        #: ignored. Handlers that want to send messages or action requests back to XMS should see `self.messages` and
        #: `self.requests`, respectively.#:
        self._duplicate_event_handlers: list[Callable[[Query], None]] = []

        # There's no known mechanism for components to learn when they were received from an ActionRequest callback.
        # All initialization must be done before sending it to XMS. Coverage components can use the component keyword
        # mechanism to associate component and feature IDs and set up the display. See the CoverageComponentBuilder.
        # We'd just use that mechanism for everything except that it was only implemented for ActionRequests.

    @property
    def class_name(self) -> str:
        """
        The name of this class.

        xmsapi needs to know this so it can reconstruct the component. This overrides the base class version to remove
        assignment, which is only present for legacy code and complicates initialization for this component.
        """
        return type(self).static_class_name()

    @classmethod
    def static_class_name(cls):
        """The name of this class, available on the type."""
        return cls.__name__

    @classmethod
    def unique_name(cls) -> str:
        """
        The unique name for this component.

        XMS needs this to be unique for a model. By convention, we just make this equal to the class name.
        """
        return cls.static_class_name()

    @property
    def module_name(self):
        """
        The name of the module containing this class.

        xmsapi needs to know this so it can reconstruct the component. This overrides the base class version to remove
        assignment, which is only present for legacy code and complicates initialization for this component.
        """
        return type(self).static_module_name()

    @classmethod
    def static_module_name(cls):
        """The name of the module containing this class, available on the type."""
        return cls.__module__

    def get_project_explorer_menus(self, main_file_list: list[tuple[str, int]]) -> list[Menu | MenuItem | None]:
        """
        Get a list of menu items for when the component is right-clicked.

        Args:
            main_file_list: List of tuples where the first element is the path to a main_file, and the second
                element is whether that component is locked.

        Returns:
            A list of menus and menu items to be shown. Note that this list can have objects of type `xms.api.dmi.Menu`
            as well as `xms.api.dmi.MenuItem`. `None` may be added to the list to indicate a separator.
        """
        if len(main_file_list) != 1 or not self.tree_commands:
            return []  # Multi-select, nothing selected, or no project explorer menu commands for this component

        menu_list = [None]  # None == spacer
        # This used to add menu items in reverse on the assumption that we'd want the most derived component to be the
        # one that picks the top item, but after a few models it was apparent the only components that cared about the
        # order were sim components that had to go out of their way to undo the reversal.
        for command_text, command_method in self.tree_commands:
            item = self._make_menu_item(command_text, command_method)
            menu_list.append(item)

        return menu_list

    def get_double_click_actions(self, lock_state: bool) -> MessagesAndRequests:
        """
        Get the menu item to run when the component is double-clicked.

        Args:
            lock_state: Whether the component is locked for editing. Do not change the files if locked.

        Returns:
            Messages and requests for XMS to handle.
        """
        messages: Messages = []

        if self.default_tree_command:
            requests = [self._make_request(self.default_tree_command)]
            return messages, requests

        main_files = [(self.main_file, lock_state)]
        items = self.get_project_explorer_menus(main_file_list=main_files)

        for item in items:
            if isinstance(item, MenuItem):
                requests = [item.action_request]
                return messages, requests

        XmEnv.report_error(
            f'{self.module_name}.{self.class_name}\n'
            'is identified as supporting double-clicking in its XML, but an explicit handler was not assigned\n'
            'and a suitable default handler could not be found.'
        )
        return messages, []

    def _make_menu_item(
        self, text: str, command: MenuCallback | list[Command], params: Optional[dict] = None
    ) -> Menu | MenuItem:
        """
        Make a menu item.

        Args:
            text: Text to display on the menu item.
            command: Method to call when the item is clicked, or list of items for a submenu.
            params: Parameters to pass to `command_method` when it gets called. They will be wrapped in a list before
                the call, due to XMS weirdness. All keys must be the same type, and either str or int.

        Returns:
            The built menu item.
        """
        if isinstance(command, Callable):  # Should be MenuCallback, but can't use that since it's only a type hint
            action = self._make_request(command, params=params)
            item = MenuItem(text=text, action=action)
            return item

        submenu = Menu(text)
        for sub_label, sub_command in command:
            sub_item = self._make_menu_item(sub_label, sub_command, params)
            if isinstance(sub_item, Menu):
                submenu.add_menu(sub_item)
            else:
                submenu.add_menu_item(sub_item)
        return submenu

    def _make_request(
        self,
        command_method: MenuCallback,
        params: Optional[dict] = None,
        needs_window: bool = True,
        main_file: Optional[str | Path] = None
    ) -> ActionRequest:
        """
        Build an ActionRequest.

        Args:
            command_method: The method to build a request for.
            params: Parameters to pass to `command_method` when it gets called. They will be wrapped in a list before
                the call, due to XMS weirdness. All keys must be the same type, and either str or int.
            needs_window: Whether the command method should be called with a window.

        Returns:
            The built ActionRequest.
        """
        main_file = str(main_file or self.main_file)  # xmsapi doesn't like Path

        action = ActionRequest(
            main_file=main_file,
            modality='MODAL' if needs_window else 'NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name=command_method.__name__,
            comp_uuid=self.uuid,
        )

        if params:
            action.action_parameters = params

        return action

    @final
    def create_event(self, lock_state: bool) -> MessagesAndRequests:
        """
        This will be called when XMS receives the component from an import script.

        This seems to only be called when the component is sent to XMS via Query from an import script. Creating a new
        coverage from XMS, changing a coverage's type to one that prompts XMS to create the component, and sending the
        component from any kind of component event, don't seem to trigger create events at all. This should not be
        relied on for anything that needs to happen outside an import script.

        Derived classes are discouraged from overriding this, since doing so is likely to break display options for
        the component. See `self._create_event_handlers` if create-time actions are required.

        Args:
            lock_state: Whether the component is locked for editing. Do not change the files if locked.

        Returns:
            Messages and requests for XMS to handle.
        """
        if self._create_event_handlers:
            params = {'locked': lock_state}
            request = self._make_request(self._create_event, params=params, needs_window=False)
            return [], [request]

        return [], []

    @final
    def _create_event(self, query: Query, _params) -> MessagesAndRequests:
        """
        Handle the component's create event.

        Derived classes are discouraged from overriding this, since doing so is likely to break display options for
        the component. See `self._create_event_handlers` if create-time actions are required.

        Args:
            query: Interprocess communication object.
            _params: List of one dictionary. The dictionary should have a key 'locked', which is whether the component
                is locked for editing. This is currently unused since a newly created component should probably always
                be unlocked.

        Returns:
            Messages and requests for XMS to handle.
        """
        for handler in self._create_event_handlers:
            handler(query)

        return self.messages, self.requests

    @final
    def get_initial_display_options(self, query: Query, _params: list[dict]) -> MessagesAndRequests:
        """
        Called when XMS receives the component from an import script or just duplicated it.

        Args:
            query: Query with a context at the component instance level.
            _params: See `self._create_event`.

        Returns:
            Messages and requests.
        """
        flag_file = Path(self.main_file).parent / duplicate_event_file
        if flag_file.exists():
            flag_file.unlink()
            handlers = self._duplicate_event_handlers
        else:
            handlers = self._initialize_event_handlers

        for handler in handlers:
            handler(query)

        return self.messages, self.requests

    def save_to_location(self, new_path: str, save_type: str) -> tuple[str, Messages, Requests]:
        """Save component files to a new location.

        Args:
            new_path: Path to the new save location.
            save_type: One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                - DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                  start with.
                - PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                  data must use relative file paths.
                - SAVE happens when re-saving this project.
                - SAVE_AS happens when saving a project in a new location. This happens the first time we save a
                  project.
                - UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder
                  in the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            A tuple of (new_main_file, messages, action_requests).
                - new_main_file: Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages: Messages for XMS to display.
                - action_requests: List of actions for XMS to perform.
        """
        messages = []
        action_requests = []

        new_main_file = Path(new_path) / Path(self.main_file).name

        if save_type == 'SAVE':  # We already updated ourselves, so copy our mainfile from temp to the new location.
            io_util.copyfile(self.main_file, new_main_file)
        if save_type == 'DUPLICATE':
            new_folder = Path(new_main_file).parent
            (new_folder / duplicate_event_file).touch()

        return str(new_main_file), messages, action_requests
