"""Module for VisibleCoverageComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['VisibleCoverageComponentBaseData']

# 1. Standard Python modules
from abc import ABC, abstractmethod
from pathlib import Path
from typing import Optional, Sequence

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.components.bases.xarray_base import MainFileBase


class VisibleCoverageComponentBaseData(MainFileBase, ABC):
    """Data manager for a visible coverage component."""
    @abstractmethod
    def __init__(self, main_file: str | Path):
        """
        Initialize the class.

        Args:
            main_file: Path to the data manager's data file. Will be created if it does not exist.
        """
        super().__init__(main_file)
        #: Mapping from TargetType -> (mapping from feature_id -> component_id).
        #: Access like `component_id_map[TargetType.point][point_feature_id]`.
        #: Will be initialized by VisibleCoverageComponentBase.load_coverage_component_id_map() to have one key for
        #: each target type. Algorithms can assume that, if this is not None, then each target type is a valid key,
        #: whether any features of that type exist or not.
        self.component_id_map: Optional[dict[TargetType, dict[int, int]]] = None

    @abstractmethod
    def touch(self):
        """
        Ensure the data manager's main-file exists.

        It's up to the data manager what data, if any, is put into the main-file. The data manager may be constructed
        later with a main-file that has been touched, but otherwise uninitialized, so either touching should ensure the
        file has acceptable content, or the constructor should be ready to deal with whatever content touching creates.

        The main-file *will* be touched multiple times, with writes in between. Touching should be ready to deal with
        main-files that are already initialized, without losing data.
        """
        pass

    @property
    @abstractmethod
    def coverage_uuid(self) -> str:
        """The UUID of the coverage the component belongs to."""
        pass

    @coverage_uuid.setter
    @abstractmethod
    def coverage_uuid(self, uuid: str):
        """The UUID of the coverage the component belongs to."""
        pass

    @abstractmethod
    def commit(self):
        """Save the current state of the data manager to disk."""
        pass

    @abstractmethod
    def component_ids_with_type(self, target: TargetType, feature_type: str) -> Sequence[int]:
        """
        Get all the component IDs that have a particular feature type.

        Args:
            target: Target type.
            feature_type: The type of the feature to get component IDs for. Should be one of the internal names in
                the component's `features` member.

        Returns:
            The component IDs that have the provided target and feature types.
        """
        pass
