"""Module for DisplayOptionsHelper."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = [
    'DisplayOptionsHelper',
    'FeatureDict',
    'FeatureList',
    'TARGETS',
    'UNASSIGNED_TYPE',
    'MULTIPLE_TYPES',
    'MULTIPLE_TYPES_LABEL',
]

# 1. Standard Python modules
from array import array
import ast
from collections import Counter
from itertools import count
from pathlib import Path
from typing import cast, Iterator, Sequence, TypeAlias
import uuid

# 2. Third party modules
from PySide2.QtGui import QColor, QIcon
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.core.filesystem.filesystem import removefile
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.line_style import LineOptions
from xms.guipy.data.point_symbol import PointOptions
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon
from xms.guipy.settings import SettingsManager

# 4. Local modules
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids,
    write_display_option_line_locations, write_display_options_to_json
)
from xms.components.display.xms_display_message import XmsDisplayMessage

FeatureList: TypeAlias = list[tuple[str, str] | tuple[int, int]]
FeatureDict: TypeAlias = dict[TargetType, FeatureList]

#: A feature type for a feature which has no selected type.
UNASSIGNED_TYPE = 'generated-unassigned {665ffa81-a3a9-4224-8832-bc2cf4e4774a}'

#: A feature type for a feature which has multiple selected types.
MULTIPLE_TYPES = 'generated-multiple {676f0ee5-0e3c-4ab0-b764-62930597ead9}'

MULTIPLE_TYPES_LABEL = 'Multiple types'

INITIAL_POINT_ATT_ID_FILE = 'initial_point.attids'
INITIAL_POINT_COMP_ID_FILE = 'initial_point.compids'
INITIAL_ARC_ATT_ID_FILE = 'initial_arc.attids'
INITIAL_ARC_COMP_ID_FILE = 'initial_arc.compids'
INITIAL_POLYGON_ATT_ID_FILE = 'initial_polygon.attids'
INITIAL_POLYGON_COMP_ID_FILE = 'initial_polygon.compids'

TARGETS = [TargetType.point, TargetType.arc, TargetType.polygon]

PT_JSON = 'pt_display.json'
ARC_JSON = 'arc_display.json'
POLY_JSON = 'poly_display.json'

DEFAULT_COLORS = [
    (255, 0, 0),  # red
    (0, 0, 255),  # blue
    (85, 170, 0),  # lighter green
    (211, 165, 0),  # dark gold
    (0, 160, 203),  # cyan
    (255, 0, 204),  # magenta
    (153, 0, 255),  # purple
    (255, 102, 0),  # orange
    (51, 153, 51),  # dark green
    (153, 102, 0),  # brown
    (102, 153, 153),  # grey blue
    (153, 51, 51),  # dark red
    (255, 153, 255),  # light magenta
]


class ColorPool:
    def __init__(self, colors: list[QColor]):
        counter = Counter()
        for q_color in colors:
            color = (q_color.red(), q_color.green(), q_color.blue())
            counter[color] += 1

        self._used_colors = counter
        self._color_pool = []

    def next_color(self) -> QColor:
        if not self._color_pool:
            self._color_pool = [color for color in DEFAULT_COLORS]
            self._color_pool.reverse()

        while self._color_pool:
            color = self._color_pool.pop()
            if self._used_colors[color] > 0:
                self._used_colors[color] -= 1
            else:
                return QColor(color[0], color[1], color[2], 255)

        return self.next_color()


class DisplayOptionsHelper:
    """
    Display options helper.

    This helper is designed to be used with `xms.components.bases.visible_coverage_component_base`. You would typically
    instantiate it by passing the `.main_file` attribute of that component to this helper.

    This is mainly used by the component itself after it has been created. See
    `xms.components.coverage_component_builder` for something that can be used to initialize the component when creating
    it in a feedback thread.
    """
    def __init__(self, main_file: str | Path):
        """
        Initialize the helper.

        Args:
            main_file: The main-file of the component to apply display options to. Typically the `.main_file` attribute
                of the component.
        """
        main_file = Path(main_file)
        self.display_directory = main_file.parent
        self.cats: list[CategoryDisplayOptionList] = []

    def __enter__(self):
        """Read everything from disk."""
        self.read_display_options()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        """Save everything to disk."""
        if exc_type is None and exc_value is None and traceback is None:
            # If something crashed, we might have garbage data, so we won't try to write it.
            self.write_display_options()

    def read_display_options(self):
        """Read display options from disk."""
        for target in TARGETS:
            json_file = self._json_for_target(target)

            # json_dict will be {} if the file doesn't exist.
            json_dict = read_display_options_from_json(str(json_file))
            categories = CategoryDisplayOptionList()
            # categories.from_dict has no effect if json_dict is {}
            categories.from_dict(json_dict)
            categories.target_type = target  # Redundant if file exists, necessary if it doesn't
            self.cats.append(categories)

    def write_display_options(self):
        """Write display options to disk."""
        for target in TARGETS:
            category_list = self.cats[target]
            if len(category_list.categories) < 1:
                # read_display_options() ensures every target exists, whether the component supports it or not.
                # initialize_display_options() ensures there will be at least one category for anything the component
                # supports. If we got here, either the component doesn't support the target, or someone neglected to
                # initialize the display options.
                continue
            disp_opts_file = self._json_for_target(target)
            write_display_options_to_json(str(disp_opts_file), category_list)

    def initialize_display_options(
        self, comp_uuid: str, module_name: str, features: FeatureDict, default_features: dict[TargetType, str]
    ):
        """
        Creates default display option .json files for a coverage component.

        All display options files are created. Existing ones will be overwritten with defaults.

        If the user saved defaults in the registry, they will be applied. Otherwise default ones will be used. In the
        event that there are registry defaults for one target but not another, then registry defaults will be used where
        they exist, and hard-coded defaults will be used where they don't.

        N.B. the files will not exist on disk until `self.write_display_options()` is called, which will happen
        automatically when the context manager goes out of scope.

        See `self.update_features` if you need to update the list of supported features after initialization.

        Args:
            comp_uuid: UUID of the component. See its `self.uuid` member.
            module_name: The component's module name. See its `self.module_name` member.
            features: The components supported features. See its `self.features` member.
            default_features: The component's default features. See its `self.default_features` member.
        """
        for target in TARGETS:
            if target not in features:
                continue  # Don't make files for unsupported features

            _check_features_unique(features[target])
            self.cats[target] = self._get_registry_display_options(module_name, target)
            self._patch_display_options(comp_uuid, target, features[target])
            self._assign_default_type(target, features[target], default_features)

    @staticmethod
    def _get_registry_display_options(module_name: str, target: TargetType) -> CategoryDisplayOptionList:
        """
        Get display options from the registry, if available.

        If there are no registry display options, some default-initialized ones will be returned instead.

        Args:
            module_name: The component's module name. See its `self.module_name` member.
            target: Target to get display options for.

        Returns:
            The default display options found in the registry, or None if no defaults were found.
        """
        settings = SettingsManager()
        value = settings.get_setting(module_name, _name_for_target(target), b'')
        value = cast(bytes, value)

        try:
            parsed = ast.literal_eval(value.decode())
        except (SyntaxError, ValueError):
            parsed = {}

        if not parsed or not isinstance(parsed, dict):
            parsed = {}

        categories = CategoryDisplayOptionList()
        categories.from_dict(parsed)
        return categories

    def _patch_display_options(self, comp_uuid: str, target: TargetType, features: FeatureList):
        """
        Fix up any errors in the display options.

        Adds missing features, removes extras, fixes the UUIDs, etc.

        Args:
            comp_uuid: The component's UUID.
            target: Type of feature.
            features: List of feature types the component wants to support. Each element is a tuple of
                (internal_name, label).

        Returns:
            Display options for the target.
        """
        target_options = self.cats[target]
        target_options.target_type = target
        target_options.file = self._json_for_target(target)
        target_options.comp_uuid = comp_uuid
        target_options.uuid = str(uuid.uuid4())

        self._drop_extra_features(target, features)
        self._add_missing_features(target, features)
        self._color_features(target)

        for index, feature_options in enumerate(target_options.categories):
            feature_options.id = index
            prefix = _id_file_prefix(target)
            feature_options.file = f'{prefix}cat{index}.ids'

    def _assign_default_type(self, target: TargetType, features: FeatureList, defaults: dict[TargetType, str]):
        """Mark the default type as the default."""
        name = defaults.get(target, '')  # Nonexistent keys are used to imply no default feature for target
        matching_labels = (l for n, l in features if n == name)
        label = next(iter(matching_labels), '')
        for category in self.cats[target].categories:
            if category.description == label:
                category.is_unassigned_category = True

    def update_features(self, target: TargetType, features: FeatureList):
        """
        Update the features supported for a given target.

        This is similar to `self.initialize_display_options` in that it makes the supported features match what the
        component wants. If `initialize_display_options` is used to reinitialize the display options, it will throw out
        any user-specified settings in favor of reloading the defaults. This method preserves existing settings, but
        requires those settings to have already been created by `initialize_display_options`.

        Args:
            target: Target type to update.
            features: The features the component now supports.
        """
        self._drop_extra_features(target, features)
        self._add_missing_features(target, features)
        self._color_features(target)

        target_options = self.cats[target].categories
        max_existing = max(feature_options.id for feature_options in target_options)
        new_ids = count(start=max_existing + 1)

        for target_option in target_options:
            if target_option.id == -1:
                new_id = next(new_ids)
                target_option.id = new_id
                prefix = _id_file_prefix(target)
                target_option.file = f'{prefix}cat{new_id}.ids'

    def _drop_extra_features(self, target: TargetType, features: FeatureList):
        """Drop any features that are in the display options but not the features list."""
        target_options = self.cats[target]

        needed_labels = {label for _name, label in features}

        dropped_categories = [
            feature_options
            for feature_options in target_options.categories if feature_options.description not in needed_labels
        ]
        for dropped_category in dropped_categories:
            (self.display_directory / dropped_category.file).unlink(missing_ok=True)

        target_options.categories = [
            feature_options
            for feature_options in target_options.categories if feature_options.description in needed_labels
        ]

    def _add_missing_features(self, target: TargetType, features: FeatureList):
        """Add any features that are in the feature list but not in the display options."""
        target_options = self.cats[target]

        have_labels = {feature_options.description for feature_options in target_options.categories}
        needed_labels = [label for _name, label in features if label not in have_labels]
        if _needs_multiple_type(features):
            needed_labels.append(MULTIPLE_TYPES_LABEL)

        for label in needed_labels:
            target_options.categories.append(CategoryDisplayOption())
            c = target_options.categories[-1]
            c.options = _option_type_for_target(target)
            c.description = label
            c.id = -1

    def _color_features(self, target: TargetType):
        """
        Assign colors to unassigned features.

        Assumes the feature options' ID is -1 if it needs colors.
        """
        target_options = self.cats[target]
        used_colors = [feature_options.options.color for feature_options in target_options.categories]
        pool = ColorPool(used_colors)
        for feature_options in self.cats[target].categories:
            if feature_options.id == -1:
                feature_options.options.color = pool.next_color()

    def run_display_options_dialog(self, parent: QWidget, module_name: str, cov_uuid: str) -> list[XmsDisplayMessage]:
        """
        Run the display options dialog.

        Args:
            parent: Parent widget for the dialog.
            module_name: Module name of the component running the dialog. See the component's `self.module_name` member.
            cov_uuid: UUID of the coverage the component belongs to. See the components `self.cov_uuid` member.

        Returns:
            List of display messages. The component should extend its display message list with these to signal XMS to
            update the display.
        """
        reg_keys = [_name_for_target(target) for target in TARGETS]

        dlg = CategoryDisplayOptionsDialog(self.cats, parent, package_name=module_name, registry_keys=reg_keys)

        dlg.setWindowIcon(QIcon(get_xms_icon()))
        dlg.setModal(True)
        if dlg.exec():
            self.cats = dlg.get_category_lists()
            return self.get_update_messages(cov_uuid)
        else:
            return []

    def _json_for_target(self, target: TargetType):
        """
        Get the path to the .json file for a target.

        Args:
            target: Target to get the .json file for.

        Returns:
            Absolute path to the .json file.
        """
        names = {TargetType.point: PT_JSON, TargetType.arc: ARC_JSON, TargetType.polygon: POLY_JSON}

        return self.display_directory / names[target]

    def request_feature_update(self, target: TargetType, component_ids: Sequence[int], feature_ids: Sequence[int]):
        """
        Request that the given target, component ID, and feature ID be associated with each other.

        XMS needs to know which feature IDs are associated with which component IDs in order to display things
        correctly. The helper is not able to tell XMS directly; only a component can do that. This method saves off the
        feature and component IDs and feature types. When the component is created, it can call
        `self.features_needing_update()` to retrieve the IDs, then tell XMS itself.

        Args:
            target: The target type of the features to request updates for.
            component_ids: The component IDs. Each component ID will be associated with the feature ID in `feature_ids`
                at the same index. Duplicates are allowed; the component ID will be associated with multiple features if
                duplicates are present.
            feature_ids: The feature IDs. Parallel to `component_ids`. Each feature ID will be associated with the
                component id in `component_ids` at the same index. Duplicates will invoke weird behavior in XMS and are
                forbidden for this reason.
        """
        if len(set(feature_ids)) != len(feature_ids):
            raise ValueError('Duplicate feature ID detected')
        if len(feature_ids) != len(component_ids):
            raise ValueError('feature_ids and component_ids must be parallel.')

        mapping = {
            TargetType.point: (INITIAL_POINT_ATT_ID_FILE, INITIAL_POINT_COMP_ID_FILE),
            TargetType.arc: (INITIAL_ARC_ATT_ID_FILE, INITIAL_ARC_COMP_ID_FILE),
            TargetType.polygon: (INITIAL_POLYGON_ATT_ID_FILE, INITIAL_POLYGON_COMP_ID_FILE),
        }
        feature_file, component_file = mapping[target]
        feature_file = str(self.display_directory / feature_file)
        component_file = str(self.display_directory / component_file)

        write_display_option_ids(feature_file, feature_ids)
        write_display_option_ids(component_file, component_ids)

    def features_needing_update(self) -> Iterator[tuple[TargetType, int, int]]:
        """
        Get all the features needing an update.

        This can be used by a component during its create event to retrieve most of the data passed to
        `self.request_feature_update()` when the component was built. The return values are those that the component
        should call its `self.update_component_id()` with. Feature types are not returned since those are only needed
        by the display options helper, and handled internally.

        Returns:
            Iterable of tuples of `(target, component_id, feature_id)` identifying features that need to be updated.
        """
        file_names = [
            (INITIAL_POINT_ATT_ID_FILE, INITIAL_POINT_COMP_ID_FILE, TargetType.point),
            (INITIAL_ARC_ATT_ID_FILE, INITIAL_ARC_COMP_ID_FILE, TargetType.arc),
            (INITIAL_POLYGON_ATT_ID_FILE, INITIAL_POLYGON_COMP_ID_FILE, TargetType.polygon),
        ]

        for feature_id_file_name, component_id_file_name, target_type in file_names:
            feature_id_file = self.display_directory / feature_id_file_name
            component_id_file = self.display_directory / component_id_file_name

            # This could use an `or` check, but then it would silently ignore bugs where one file exists and the other
            # doesn't. It uses `and` so they're noticed.
            if not feature_id_file.exists() and not component_id_file.exists():
                continue

            feature_ids = read_display_option_ids(str(feature_id_file))
            removefile(feature_id_file)

            component_ids = read_display_option_ids(str(component_id_file))
            removefile(component_id_file)

            # The lists should always be the same length. This uses strict so bugs are more likely to be noticed.
            for feature_id, component_id in zip(feature_ids, component_ids, strict=True):
                yield target_type, component_id, feature_id

    def assign_component_ids_for_feature_type(
        self, target: TargetType, feature_label: str, component_ids: Sequence[int]
    ):
        """
        Assign the component IDs that should be displayed with a particular feature type.

        This completely overwrites the existing assignment.

        Args:
            target: Target type.
            feature_label: The feature's display label, *not* its internal name.
            component_ids: Component IDs to display with the given target and feature type.
        """
        file_name = self._component_id_file_for_feature_type(target, feature_label)
        file_path = str(self.display_directory / file_name)
        write_display_option_ids(file_path, component_ids)

    def get_update_messages(self, coverage_uuid: str) -> list[XmsDisplayMessage]:
        """
        Get update messages to send to XMS to prompt it to update the coverage's display.

        Messages for all targets supported by the component will be returned. The messages should be extended into the
        component's `display_option_list` member. Doing so will result in XMS updating the display for the component.

        Args:
            coverage_uuid: The UUID of the coverage the component is attached to.

        Returns:
            Messages that can be extended into the component's `display_option_list` member.
        """
        messages = []

        for target in TARGETS:
            # An earlier revision checked whether the .json file existed, but this runs before the files are created,
            # so checking the categories is more reliable.
            if len(self.cats[target].categories) > 0:
                file = self._json_for_target(target)
                messages.append(XmsDisplayMessage(file=str(file), edit_uuid=coverage_uuid))

        return messages

    def assign_component_id(self, target: TargetType, component_id: int, feature_label: str):
        """
        Assign a component ID to a target type and feature type.

        This method is meant to be called by the component when it assigns new IDs. It is not suitable for use when
        building a coverage. See `self.request_feature_update()` for an alternative that is.

        Args:
            target: Target type of the feature the component ID refers to.
            component_id: Component ID to assign.
            feature_label: Label of the feature the component ID refers to. See the component's
                `self.feature_label_from_name` to get this label from an internal name.
        """
        target_options: list[CategoryDisplayOption] = self.cats[target].categories
        file = ''
        for feature_options in target_options:
            if feature_options.description == feature_label:
                file = str(self.display_directory / feature_options.file)
                break

        if not file:
            raise ValueError('Unknown feature label')

        component_ids: array = read_display_option_ids(file)
        component_ids.append(component_id)
        write_display_option_ids(file, component_ids)

    def _component_id_file_for_feature_type(self, target: TargetType, feature_label: str) -> Path:
        """
        Get the name of the file that stores component IDs for a given feature type.

        Args:
            target: Target type.
            feature_label: The feature's display label, *not* its internal name.
        """
        target_options = self.cats[target]
        for feature_options in target_options.categories:
            feature_options: CategoryDisplayOption
            if feature_options.description == feature_label:
                return self.display_directory / feature_options.file
        raise AssertionError(f'Unknown label {feature_label!r} for target {target}')

    def reset_display_uuids(self, component_uuid: str):
        """
        Reset the UUIDs in the display .json files.

        This is mostly needed when the component is duplicated. The newly duplicated component will initially have
        the same UUIDs in its .json files as the component it started from, which confuses XMS. This assigns new ones
        to keep XMS happy.

        Args:
            component_uuid: The component's new UUID.
        """
        for target_category in self.cats:
            target_category.uuid = str(uuid.uuid4())
            target_category.comp_uuid = component_uuid

    def apply_to_drawing(self, main_file: str | Path):
        """
        Apply the display options to a mapped coverage.

        Args:
            main_file: Main-file of the new mapped coverage.
        """
        main_file = Path(main_file)
        component_uuid = main_file.parent.name
        for target_category in self.cats:
            target_category.uuid = str(uuid.uuid4())
            target_category.comp_uuid = component_uuid
            target_category.is_ids = 0
        self.display_directory = main_file.parent

    def draw_lines(self, feature_label: str, lines: Sequence[Sequence[Sequence[float]]]):
        """
        Draw lines in a mapped coverage.

        The coverage must have already been assigned display options for drawing on, such as by calling
        `self.apply_to_drawing()`.

        Args:
            feature_label: Label of the feature to draw lines for.
            lines: The lines to draw. Each line is a list of points. Each point is a
        """
        id_file = self._component_id_file_for_feature_type(TargetType.arc, feature_label)
        converted_lines = []
        for line in lines:
            converted_lines.append([])
            for point in line:
                converted_lines[-1].extend(point)
        write_display_option_line_locations(str(id_file), converted_lines)

    def turn_off_labels(self):
        """Turn off labels for the coverage."""
        for target in self.cats:
            for feature_type in target.categories:
                feature_type.label_on = False


def _check_features_unique(features: FeatureList):
    """Check that the features have unique names and labels."""
    expected_length = len(features)

    unique_names = set(name for name, label in features)
    unique_labels = set(label for name, label in features)
    assert len(unique_names) == expected_length and len(unique_labels) == expected_length


def _name_for_target(target: TargetType) -> str:
    """
    Get a name for a target.

    Args:
        target: The target to get the name of.

    Returns:
        A string like 'point', 'arc', or 'poly'.
    """
    names = {
        TargetType.point: 'point',
        TargetType.arc: 'arc',
        TargetType.polygon: 'poly',
    }

    return names[target]


def _option_type_for_target(target: TargetType) -> PointOptions | LineOptions | PolygonOptions:
    """Get the option type for a target."""
    if target == TargetType.point:
        return PointOptions()
    elif target == TargetType.arc:
        return LineOptions()
    elif target == TargetType.polygon:
        return PolygonOptions()
    else:
        raise ValueError('Unsupported target')  # pragma: nocover


def _id_file_prefix(target: TargetType) -> str:
    """Get the prefix for an ID file."""
    prefixes = {
        TargetType.point: 'pt_',
        TargetType.arc: 'arc_',
        TargetType.polygon: 'poly_',
    }

    return prefixes[target]


def _needs_multiple_type(features: FeatureList) -> bool:
    """Check whether a list of features needs the "Multiple types" type added."""
    for name, _label in features:
        if name == MULTIPLE_TYPES:
            return False

    return True
