"""UGrid2dFromUGrid3dCreator class."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard python modules
from operator import itemgetter
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.grid.ugrid import UGrid

# 4. Local modules
from xms.constraint import UGridBuilder


def cogrid_2d_from_locations_and_cells(locations, cells, uuid_function):
    """Returns a cogrid made using the locations and cells."""
    cell_stream = []
    cell_types = {-1: UGrid.cell_type_enum.POLYGON, 3: UGrid.cell_type_enum.TRIANGLE, 4: UGrid.cell_type_enum.QUAD}
    for cell in cells:
        npts = len(cell)
        cell_type = cell_types[-1] if npts not in cell_types else cell_types[npts]
        cs = [cell_type, npts] + cell
        cell_stream.extend(cs)
    ugrid = UGrid(locations, cell_stream)
    ugrid_builder = UGridBuilder()
    ugrid_builder.set_is_2d()
    ugrid_builder.set_ugrid(ugrid)
    cogrid_2d = ugrid_builder.build_grid()
    cogrid_2d.uuid = uuid_function()
    return cogrid_2d


class UGrid2dFromUGrid3dCreator:
    """Tool to create a 2D UGrid from the top or bottom of a 3D UGrid with point/cell ordering the same."""

    def __init__(self):
        """Initializes the class."""
        pass

    def create_2d_cogrid(self, cogrid_3d, top_or_bottom):
        """Creates the 2D cogrid.

        Args:
            cogrid_3d (UGrid3d): The 3D grid.
            top_or_bottom (str): 'Top' or 'Bottom'

        Returns:
            cogrid.
        """
        ugrid = cogrid_3d.ugrid
        face_orientation = self._get_face_orientation(top_or_bottom)
        cells = self._cells_to_consider(cogrid_3d, top_or_bottom, ugrid.cell_count)

        # Get top or bottom faces as a list of tuples of cell index, face index (list[(int, int)])
        faces = self._get_outer_faces(ugrid, cells=cells, face_orientation=face_orientation)
        points = self._get_unique_face_points(ugrid, faces)

        # Get new locations
        points.sort()  # Sort points so that the ordering will be the same in the new UGrid
        locs = ugrid.locations
        new_locations = [locs[point] for point in points]

        # Create cells
        cells = []  # list of lists of points
        old_to_new_points = {point: i for i, point in enumerate(points)}
        faces.sort(key=itemgetter(0))  # Sort faces by cell index so cell ordering will be the same in the new UGrid
        for cell_idx, face in faces:
            face_points = ugrid.get_cell_3d_face_points(cell_idx, face)
            cell_points = [old_to_new_points[face_point] for face_point in face_points]
            if top_or_bottom == 'Bottom':
                cell_points.reverse()
            cells.append(cell_points)

        cogrid = cogrid_2d_from_locations_and_cells(new_locations, cells, self.get_uuid)
        return cogrid

    def _get_face_orientation(self, top_or_bottom):
        """Returns the face orientation given the user's choice of top or bottom.

        Args:
            top_or_bottom (str): 'Top' or 'Bottom'

        Returns:
            (UGrid.face_orientation_enum): ORIENTATION_TOP or ORIENTATION_BOTTOM.
        """
        ug_foe = UGrid.face_orientation_enum  # for short
        return ug_foe.ORIENTATION_TOP if top_or_bottom == 'Top' else ug_foe.ORIENTATION_BOTTOM

    def _the_layer_from_inputs(self, cogrid_3d, top_or_bottom):
        """Returns either the top of bottom layer number given the user's choice and the UGrid cell ordering.

        Args:
            cogrid_3d (UGrid3d): The 3D grid.
            top_or_bottom (str): 'Top' or 'Bottom'

        Returns:
            (int): The layer
        """
        cell_ordering = cogrid_3d.ugrid.calculate_cell_ordering()
        if top_or_bottom == 'Top':
            if cell_ordering == UGrid.cell_ordering_enum.CELL_ORDER_INCREASING_DOWN:
                the_layer = min(cogrid_3d.cell_layers)
            else:
                the_layer = max(cogrid_3d.cell_layers)
        else:
            if cell_ordering == UGrid.cell_ordering_enum.CELL_ORDER_INCREASING_DOWN:
                the_layer = max(cogrid_3d.cell_layers)
            else:
                the_layer = min(cogrid_3d.cell_layers)
        return the_layer

    def _cells_to_consider(self, cogrid_3d, top_or_bottom, cell_count):
        """Returns the cells in the top or bottom layer, if using layers, or all the cells if not.

        Args:
            cogrid_3d (UGrid3d): The 3D grid.
            top_or_bottom (str): 'Top' or 'Bottom'
            cell_count (int): Number of cells in the grid.

        Returns:
            (list[int]): cell indexes.
        """
        # Get the cells we will be considering. Use layers if they exist
        if cogrid_3d.cell_layers:
            the_layer = self._the_layer_from_inputs(cogrid_3d, top_or_bottom)
            cell_lay = cogrid_3d.cell_layers
            cells = [idx for idx in range(cell_count) if cell_lay[idx] == the_layer]
        else:
            cells = [cell_idx for cell_idx in range(cell_count)]
        return cells

    @staticmethod
    def get_uuid():
        """Returns a random uuid string (or maybe not so random if testing).

        Returns:
            (str): uuid string.
        """
        return str(uuid.uuid4())

    @staticmethod
    def _get_outer_faces(ugrid, cells, face_orientation: UGrid.face_orientation_enum | None = None):
        """Returns list of faces on the outside of the UGrid as list of (cell index, face index) tuples.

        Args:
            ugrid (UGrid): The UGrid.
            cells (list[int]): Optional list of cells which, if provided, will be the only cells considered.
            face_orientation (UGrid.face_orientation_enum): Optional face orientation (only top and bottom make sense).
        """
        outer_faces = []
        if face_orientation is not None:
            for cell_idx in cells:
                face_count = ugrid.get_cell_3d_face_count(cell_idx)
                for face_idx in range(face_count):
                    orientation = ugrid.get_cell_3d_face_orientation(cell_idx, face_idx)
                    if orientation == face_orientation:
                        if ugrid.get_cell_3d_face_adjacent_cell(cell_idx, face_idx) == -1:
                            outer_faces.append((cell_idx, face_idx))
        else:
            for cell_idx in cells:
                face_count = ugrid.get_cell_3d_face_count(cell_idx)
                for face_idx in range(face_count):
                    adjacent_cell = ugrid.get_cell_3d_face_adjacent_cell(cell_idx, face_idx)
                    if adjacent_cell == -1:
                        outer_faces.append((cell_idx, face_idx))
        return outer_faces

    def _get_unique_face_points(self, ugrid, faces):
        """Returns list of unique UGrid point indexes on the given faces.

        Args:
            ugrid (UGrid): The UGrid.
            faces (list[tuple[int, int]]): list of (cell index, face index) tuples.
        """
        unique_points = set()  # Use a set to eliminate duplicates
        for cell_idx, face in faces:
            face_points = ugrid.get_cell_3d_face_points(cell_idx, face)
            unique_points.update(face_points)
        return list(unique_points)
