"""Python wrapping for xms._data_objects.parameters.Component."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.data_objects._data_objects.parameters import Component as CComponent


class Component:
    """The pure Python wrapper for C++ exposed xms._data_objects.parameters.Component objects."""
    def __init__(self, **kwargs):
        """Construct the wrapper.

        Args:
            **kwargs:
                name (str): Name of the component
                comp_uuid (str): UUID of the component
                main_file (str): Full path to the component's main file
                model_name (str): XML-defined model name of the component
                unique_name (str): XML-defined unique name of the component
                class_name (str): Component's Python class name
                module_name (str): Component's full module import name
                locked (bool): True if the component is currently locked for editing
                instance (CComponent): The C++ Component object to wrap
        """
        self._instance = kwargs.get('instance')
        if not self._instance:
            self._instance = CComponent()

        # Make sure XML model and unique names are given together
        have_model_name = 'model_name' in kwargs
        have_unique_name = 'unique_name' in kwargs
        if have_model_name != have_unique_name:
            raise ValueError('Must construct with both model_name and unique_name or neither.')
        # Make sure Python class and module names are given together
        have_class_name = 'class_name' in kwargs
        have_module_name = 'module_name' in kwargs
        if have_class_name != have_module_name:
            raise ValueError('Must construct with both class_name and module_name or neither.')

        if 'name' in kwargs:
            self._instance.SetName(kwargs['name'])
        if 'comp_uuid' in kwargs:
            self._instance.SetUUID(kwargs['comp_uuid'])
        if 'main_file' in kwargs:
            self._instance.SetMainFile(kwargs['main_file'])
        if have_model_name:
            self._instance.SetUniqueNameAndModelName(kwargs['unique_name'], kwargs['model_name'])
        if have_class_name:
            self._instance.SetClassAndModule(kwargs['class_name'], kwargs['module_name'])
        if 'locked' in kwargs:
            self._instance.SetLocked(kwargs['locked'])

    @property
    def name(self):
        """Returns the component's name."""
        return self._instance.GetName()

    @name.setter
    def name(self, comp_name):
        """Sets the component's name ("Tree item name", if not hidden).

        Args:
            comp_name (str): name of the component
        """
        self._instance.SetName(comp_name)

    @property
    def uuid(self):
        """Returns the component's UUID."""
        return self._instance.GetUUID()

    @uuid.setter
    def uuid(self, comp_uuid):
        """Sets the component's UUID.

        Args:
            comp_uuid (str): UUID of the component
        """
        self._instance.SetUUID(comp_uuid)

    @property
    def main_file(self):
        """Returns the component's main file."""
        return self._instance.GetMainFile()

    @main_file.setter
    def main_file(self, comp_main_file):
        """Sets the component's main file.

        Args:
            comp_main_file (str): main file of the component
        """
        self._instance.SetMainFile(comp_main_file)

    def get_class_and_module(self):
        """Returns a tuple containing a python class and module of the component definition."""
        return self._instance.GetClassAndModule()

    def set_class_and_module(self, class_name, module_name):
        """Sets the python class and module of the component definition.

        Args:
            class_name (str): class name of the component definition
            module_name (str): module of the component definition
        """
        self._instance.SetClassAndModule(class_name, module_name)

    def get_unique_name_and_model_name(self):
        """Returns a tuple containing the xml unique name and model name of the component definition."""
        return self._instance.GetUniqueNameAndModelName()

    def set_unique_name_and_model_name(self, unique_name, model_name):
        """Sets the xml unique name and model name of the component definition.

        Args:
            unique_name (str): unique name of the component as defined in the xml
            model_name (str): model name of the component as defined in the xml
        """
        self._instance.SetUniqueNameAndModelName(unique_name, model_name)

    @property
    def locked(self):
        """Returns True if the component is locked."""
        return self._instance.IsLocked()

    @locked.setter
    def locked(self, comp_lockstate):
        """Sets the lockstate of the component.

        Args:
            comp_lockstate (bool): lockstate of the component
        """
        self._instance.SetLocked(comp_lockstate)
