"""Helper methods for converting data_objects DateTimeLiterals and TimeDurationLiterals."""
# 1. Standard python modules
import datetime

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.data_objects._data_objects.parameters import DateTimeLiteral as CDateTimeLiteral


INVALID_DATE = datetime.datetime(1400, 1, 1, 0, 0, 0)


def julian_to_datetime(julian):
    """Convert a Julian double to a Python datetime.

    Args:
        julian (float): The Julian double to convert

    Returns:
        datetime.datetime: A Python datetime representing julian, or None if date is invalid.
    """
    do_dt = CDateTimeLiteral()
    do_dt.SetFromDouble(julian)
    py_dt = date_time_literal_to_datetime(do_dt)
    return py_dt if py_dt != INVALID_DATE else None


def datetime_to_julian(py_datetime):
    """Convert a Python datetime to a Julian double.

    Args:
        datetime.datetime: A Python datetime to convert

    Returns:
        float: The Julian double representation of the date
    """
    do_dt = datetime_to_date_time_literal(py_datetime)
    return do_dt.GetAsDouble()


def date_time_literal_to_datetime(do_dt):
    """Convert a data_objects DateTimeLiteral to a native Python datetime.

    Args:
        do_dt (xms.data_objects._data_objects.parameters.DateTimeLiteral): The C++ exposed data_object timestamp

    Returns:
        datetime.datetime: The timestamp as a native Python datetime
    """
    return datetime.datetime(
        year=do_dt.GetYear(),
        month=do_dt.GetMonth(),
        day=do_dt.GetDay(),
        hour=do_dt.GetHour(),
        minute=do_dt.GetMinute(),
        second=do_dt.GetSecond(),
        microsecond=do_dt.GetMicrosecond()
    )


def datetime_to_date_time_literal(py_datetime):
    """Convert a native Python datetime to a data_objects DateTimeLiteral.

    Args:
        py_datetime (datetime.datetime): The timestamp as a native Python datetime

    Returns:
        xms.data_objects._data_objects.parameters.DateTimeLiteral: The C++ exposed data_object timestamp
    """
    return CDateTimeLiteral(
        py_datetime.year,
        py_datetime.month,
        py_datetime.day,
        py_datetime.hour,
        py_datetime.minute,
        py_datetime.second,
        py_datetime.microsecond
    )


def time_duration_literal_to_timedelta(do_dt):
    """Convert a data_objects TimeDurationLiteral to a native Python timedelta.

    Args:
        do_dt (xms.data_objects._data_objects.parameters.TimeDurationLiteral): The C++ exposed data_object timespan

    Returns:
        datetime.timedelta: The timespan as a native Python timedelta
    """
    return datetime.timedelta(
        hours=do_dt.GetHour(),
        minutes=do_dt.GetMinute(),
        seconds=do_dt.GetSecond(),
        microseconds=do_dt.GetMicrosecond()
    )
