"""Code to handle XMS DMI component events for EwnCoverageComponent."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Menu, MenuItem
from xms.core.filesystem import filesystem as xmf

# 4. Local modules
from xms.ewn.components import ewn_component_display as ecd
from xms.ewn.data.ewn_cov_data import EwnCovData
from xms.ewn.data.sediment_cov_data import SedimentCovData


class EwnComponentManager:
    """Helper class to operate on an EwnCoverageComponent."""
    def __init__(self, comp):
        """Initializes the helper class.

        Args:
            comp (EwnCoverageComponent): The component to manage
        """
        self._comp = comp

    def handle_save(self, new_path, save_type, cov_type='EWN'):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

            cov_type (:obj:`str`): coverage type

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
                if necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        # Check if we are already in the new location
        new_main_file = os.path.join(new_path, os.path.basename(self._comp.main_file))
        if new_main_file != self._comp.main_file:
            if os.path.exists(new_main_file):
                os.remove(new_main_file)
            shutil.copyfile(self._comp.main_file, new_main_file)

        if save_type == 'UNLOCK':  # If unlocking, we are all done.
            return new_main_file, [], []

        is_sediment = False
        if cov_type == 'EWN':
            data = EwnCovData(new_main_file)
        elif cov_type == 'Sediment':
            is_sediment = True
            data = SedimentCovData(new_main_file)

        if save_type == 'DUPLICATE':  # Reset coverage and display ids on duplicate.
            json_dict = ecd.duplicate_display_opts(new_path, is_sediment)
            data.info.attrs['cov_uuid'] = ''
            data.info.attrs['display_uuid'] = json_dict['uuid']
            data.commit()

        return new_main_file, [], []

    def get_tree_menus(self):
        """Get the menus for coverage tree item right-click menu.

        Returns:
            menu_items (:obj:`list[xms.api.dmi.MenuItem]`): A list of menus and menu items to be shown. Note
            that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        menu_list = [None]  # None == spacer
        # Add the top-level project explorer menus
        for command_text, command_method in self._comp.tree_commands:
            action = ActionRequest(
                modality='MODAL',
                method_name=command_method,
                class_name=self._comp.class_name,
                module_name=self._comp.module_name,
                main_file=self._comp.main_file,
                comp_uuid=self._comp.uuid
            )
            menu_item = MenuItem(text=command_text, action=action)
            menu_list.append(menu_item)

        # Add the Tools submenu, if the component has tools.
        tool_menu = Menu(text='Tools')
        has_tools = False
        for command_text, command_method in self._comp.tool_commands:
            has_tools = True
            action = ActionRequest(
                modality='MODAL',
                method_name=command_method,
                class_name=self._comp.class_name,
                module_name=self._comp.module_name,
                main_file=self._comp.main_file,
                comp_uuid=self._comp.uuid
            )
            menu_item = MenuItem(text=command_text, action=action)
            tool_menu.add_menu_item(menu_item)
        if has_tools:
            menu_list.append(tool_menu)
        return menu_list

    def get_poly_menus(self, selection, id_files):
        """Get the EWN coverage polygon menus.

        Args:
            selection (:obj:`dict`): A dictionary with the key being a string of the feature entity type
                (POINT, ARC, POLYGON). The value of the dictionary is a list of IntegerLiteral ids of the selected
                feature objects.

            id_files (:obj:`dict`): Key is entity type string, value is tuple of two str where first is the file
                location of the XMS coverage id binary file. Second is file location of the component coverage id binary
                file. Only applicable for coverage selections. File will be deleted after event. Copy if need to
                persist.

        Returns:
            menu_items (:obj:`list[xms.api.dmi.MenuItem]`): A list of menus and menu items to be shown. Note
            that this list can have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        menu_list = [None]  # None == spacer
        # Copy all the id files to a temporary location. XMS will delete them once this method returns.
        temp_dir = os.path.join(os.path.dirname(self._comp.main_file), 'temp')
        os.makedirs(temp_dir, exist_ok=True)

        unpacked_id_files = {}
        for entity, filenames in id_files.items():
            if not os.path.exists(filenames[0]) or not os.path.exists(filenames[1]):
                id_files[entity] = ('', '')
                continue
            temp_xms_file = os.path.join(temp_dir, os.path.basename(filenames[0]))
            temp_comp_file = os.path.join(temp_dir, os.path.basename(filenames[1]))
            xmf.copyfile(filenames[0], temp_xms_file)
            xmf.copyfile(filenames[1], temp_comp_file)
            unpacked_id_files[entity] = (temp_xms_file, temp_comp_file)

        if 'POLYGON' in selection:
            poly_id_files = unpacked_id_files['POLYGON'] if 'POLYGON' in unpacked_id_files else None
            for command_text, command_method in self._comp.polygon_commands:
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self._comp.class_name,
                    module_name=self._comp.module_name,
                    main_file=self._comp.main_file,
                    comp_uuid=self._comp.uuid,
                    parameters={
                        'id_files': poly_id_files,
                        'selection': selection['POLYGON']
                    }
                )
                menu_item = MenuItem(text=command_text, action=action)
                menu_list.append(menu_item)
        if 'ARC' in selection:
            arc_id_files = unpacked_id_files['ARC'] if 'ARC' in unpacked_id_files else None
            for command_text, command_method in self._comp.arc_commands:
                action = ActionRequest(
                    modality='MODAL',
                    method_name=command_method,
                    class_name=self._comp.class_name,
                    module_name=self._comp.module_name,
                    main_file=self._comp.main_file,
                    comp_uuid=self._comp.uuid,
                    parameters={
                        'id_files': arc_id_files,
                        'selection': selection['ARC']
                    }
                )
                menu_item = MenuItem(text=command_text, action=action)
                menu_list.append(menu_item)
        if not menu_list:
            shutil.rmtree(temp_dir, ignore_errors=True)  # Delete the id files if no menus were added.
        return menu_list
