"""This module is a group of utility functions for dealing with files."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
import shutil

# 2. Third party modules
import h5py

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.core.filesystem import filesystem
from xms.gdal.utilities import gdal_utils as gu

# 4. Local modules


def paths_are_equal(path1, path2):
    """Returns true if the two filesystem paths are equal.

    Args:
        path1 (:obj:`str`): The first path.
        path2 (:obj:`str`): The second path.

    Returns:
        (:obj:`bool`): True if both paths exists and are equivalent.

    """
    if not os.path.exists(path1) or not os.path.exists(path2):
        return False
    return os.path.samefile(path1, path2)


def copyfile(src, dest):
    """Copy a file, ignoring shutil.SameFileError.

    Args:
        src (:obj:`str`): Source file path.
        dest (:obj:`str`): Destination file path.
    """
    try:
        shutil.copyfile(src, dest)
    except shutil.SameFileError:
        pass


def remove(file):
    """Delete a file, ignoring any system errors.

    Args:
        file (:obj:`str`): File path to delete.

    """
    try:
        os.remove(file)
    except Exception:
        pass


def delete_h5_groups(filename, groups):
    """Delete a group or dataset from an H5 or NetCDF file.

    Need to use H5 calls because the xarray and Python NetCDF libraries do not provide
    a convenient enough way to overwrite existing datasets in a file without overwriting
    the entire file.

    Args:
        filename (:obj:`str`): File path to the H5/NetCDF file
        groups (:obj:`list`): List of paths in the file to remove

    """
    with h5py.File(filename, 'a') as f:
        for group in groups:
            try:
                del f[group]
            except Exception:
                pass  # Try to clean up the other groups


def convert_to_relative(path, base_path):
    """Convert an absolute path to relative from a base path.

    I am currently letting this sucker throw because I have code using it that reports errors.

    Args:
        path: The absolute path to convert.
        base_path: The base path to make the absolute path relative to

    Returns:
        (:obj:`str`): The absolute path converted to relative from the base path.

    """
    if Path(base_path).is_file():
        base_path = os.path.dirname(base_path)
    return os.path.relpath(path, base_path)


def comp_uuid_from_mainfile_path(mainfile_path):
    """Get a component's UUID from the absolute path to its mainfile."""
    return os.path.basename(os.path.dirname(mainfile_path))


def does_file_exist(file, proj_dir):
    """Determine if a file in our persistent data still exist.

    If file is not absolute, will check if relative from the project directory exists.

    Args:
       file (:obj:`str`): Relative or absolute file path to check the existence of
       proj_dir (:obj:`str`): Project path to resolve relative paths to

    Returns:
        (:obj:`bool`): True if the file exists

    """
    try:
        if not os.path.isabs(file):  # Convert relative to absolute
            file = filesystem.resolve_relative_path(proj_dir, file)
        return os.path.exists(file)
    except Exception:
        return False


def compare_h5_files(output_file, base_file, to_ignore):
    """Compare two HDF5 files to see if they are the same.
    """
    with h5py.File(output_file, 'r') as f1:
        with h5py.File(base_file, 'r') as f2:
            try:
                implicit_ignore = ['dim_0',
                                   'DIMENSION_LIST',
                                   'REFERENCE_LIST',
                                   '_Netcdf4Dimid',
                                   '_NCProperties',
                                   '_Netcdf4Coordinates',
                                   '_nc3_strict',
                                   'CLASS',
                                   'NAME',
                                   '_FillValue']
                ignore = to_ignore
                ignore.extend(implicit_ignore)
                result = _compare_h5_groups('/', f1, f2, ignore)
                message = f'{result[1]}\n{output_file}\n{base_file}\n'
                return result[0], message
            except Exception as e:
                return False, f'compare_h5_files exception: {str(e)}'


def _compare_h5_groups(path, group_out, group_base, to_ignore):
    """Compares two HDF5 groups. It will recurse into subgroups.
    """
    # print(f'path: {path}')
    g_base = set(group_base.keys())  # & g_out

    for sub_group in g_base:
        sub_path = f'{path}{sub_group}/'
        # print(f'path: {sub_path}')

        # pandas uses 'index' while xarray uses 'dim_0', both work equally well,
        # and the values in them are not relevant to our tests.
        # Check to make sure at least one of them is present.
        if sub_group == 'index' or sub_group == 'dim_0':
            if 'index' in group_out:
                sub_group_out = group_out['index']
            elif 'dim_0' in group_out:
                sub_group_out = group_out['dim_0']
            else:
                sub_group_out = None
                error = f'{sub_path}: Dataset missing index.'
                return False, error
            continue
        if sub_group not in group_out:
            error = f'{sub_path}: Dataset missing.'
            return False, error
        is_dset_out = isinstance(group_out[sub_group], h5py.Dataset)
        is_dset_base = isinstance(group_base[sub_group], h5py.Dataset)
        sub_group_out = group_out[sub_group]
        sub_group_base = group_base[sub_group]

        if sub_group in to_ignore:
            continue

        # check the attributes
        a_set_base = set(sub_group_base.attrs.keys())
        for att in a_set_base:
            # print(f'att: {att}')
            if att in to_ignore:
                continue
            attr_out = sub_group_out.attrs[att]
            attr_base = sub_group_base.attrs[att]
            if attr_out != attr_base:
                error = f'{sub_path}: Attribute value mismatch ({att}))'
                return False, error
            if sub_group_out.attrs[att]:
                type_out = str(type(sub_group_out.attrs[att][0]))
            else:
                type_out = None
            if sub_group_base.attrs[att]:
                type_base = str(type(sub_group_base.attrs[att][0]))
            else:
                type_base = None
            if type_out != type_base:
                error = f'{sub_path}: Attribute type mismatch ({type_out} != {type_base})'
                return False, error

        if is_dset_out and is_dset_base:
            if sub_group_out.shape != sub_group_base.shape:
                error = f'{sub_path}: Dataset shape mismatch.'
                return False, error
            if sub_group_out.dtype != sub_group_base.dtype:
                error = f'{sub_path}: Dataset type mismatch.'
                return False, error
            array_out = sub_group_out[()]
            array_base = sub_group_base[()]
            equal = (array_out == array_base).all()
            if not equal:
                error = f'{sub_path}: Dataset value mismatch.'
                return False, error
        elif not is_dset_out and not is_dset_base:
            # recurse
            result = _compare_h5_groups(sub_path, sub_group_out, sub_group_base, to_ignore)
            if not result[0]:
                return result
        else:
            error = f'{sub_path}: Dataset/group mismatch.'
            return False, error
    return True, ""


def logging_filename(abs_path):
    """Get the filename for logging (basename if testing).

    Args:
        abs_path (:obj:`str`): Aboslute path to the file to log

    Returns:
        (:obj:`str`): abs_path or base_name of abs_path if testing
    """
    if XmEnv.xms_environ_running_tests() == 'TRUE':
        return os.path.basename(abs_path)
    return abs_path


def load_wkt(wkt):
    """Parse a WKT in a GDAL object.

    Args:
        wkt (:obj:`str`): The WKT

    Returns:
        (:obj:`osr.SpatialReference`): See description
    """
    # Use GDAL to parse the horizontal datum from the display projection's WKT.
    try:
        sr = gu.wkt_to_sr(wkt)
    except ValueError as err:
        raise err
    return sr
