"""Module for GmiCoverageComponentBase."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['CoverageComponentBase']

# 1. Standard Python modules
from abc import abstractmethod
from contextlib import suppress
from functools import cached_property
from pathlib import Path
from typing import Optional, TextIO

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.components.bases.component_with_menus_base import MessagesAndRequests
from xms.components.bases.visible_coverage_component_base import VisibleCoverageComponentBase
from xms.components.display.display_options_helper import MULTIPLE_TYPES, UNASSIGNED_TYPE
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.gmi.component_bases.dataset_selector_mixin import DatasetSelectorMixin
from xms.gmi.data.generic_model import GenericModel, Section
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData, table_to_text_file
from xms.gmi.gui.section_dialog import SectionDialog


class CoverageComponentBase(DatasetSelectorMixin, VisibleCoverageComponentBase):
    """A Dynamic Model Interface (DMI) component base for GMI-based coverage components."""
    def __init__(self, main_file: Optional[str | Path]):
        """
        Initialize the component.

        Args:
            main_file: The component's main-file.
        """
        super().__init__(main_file)
        self._section_dialog_keyword_args = {'dataset_callback': self._dataset_callback}
        self._query: Optional[Query] = None

    @abstractmethod
    def _get_section(self, target: TargetType) -> Section:
        """Get a Section with parameters for a given target."""
        pass  # pragma: nocover

    @property
    def features(self) -> dict[TargetType, list[tuple[str, str]]]:
        """The features this coverage supports."""
        feature_dict = {}
        for target in [TargetType.point, TargetType.arc, TargetType.polygon]:
            section = self._get_section(target)
            for group_name in section.group_names:
                feature_dict.setdefault(target, [])
                label = section.group(group_name).label
                feature_dict[target].append((group_name, label))

        return feature_dict

    @property
    def default_features(self) -> dict[TargetType, str]:
        """
        Defines how unassigned features are rendered.

        Mapping from `target_type -> name`. The name is the first element of one of the tuples returned by
        `self.features`. `self.features[target_type][i][0] == name` must be true for exactly one integer i. The feature
        for which it is true will be considered the default one, and unassigned features will be rendered as if they
        have been assigned that type.

        Targets that are not present in the result will be rendered as if they have no type assigned.
        """
        features = self.features
        defaults = {}
        for target in features:
            section = self._get_section(target)
            for group_name in section.group_names:
                if section.group(group_name).is_default:
                    defaults[target] = group_name
        return defaults

    @cached_property
    def data(self) -> CoverageBaseData:
        """The component's data manager."""
        return CoverageBaseData(self.main_file)

    def _internal_assign_feature(
        self, params: list[dict], parent: QWidget, query: Query, dialog_name: str, window_title: str, target: TargetType
    ) -> MessagesAndRequests:
        self._query = query
        # noinspection PyProtectedMember
        return super()._internal_assign_feature(params, parent, query, dialog_name, window_title, target)

    def _assign_feature(
        self, parent: QWidget, dialog_name: str, window_title: str, target: TargetType, feature_ids: list[int]
    ):
        """
        Display the Assign feature dialog and persist data if accepted.

        Args:
            parent: Parent widget for any dialog windows created.
            dialog_name: Suggested name for the dialog. Used to store dialog settings in the registry.
            window_title: title of window
            target: Point, arc, or polygon.
            feature_ids: IDs of selected features.
        """
        section = self._get_section(target)
        component_id = self.get_comp_id(target, feature_ids[0])
        values = self.data.feature_values(target, component_id)
        section.restore_values(values)

        if len(section.group_names) == 1:
            section.group(section.group_names[0]).is_active = True

        dlg = SectionDialog(
            parent=parent,
            section=section,
            is_interior=False,
            dlg_name=dialog_name,
            window_title=window_title,
            **self._section_dialog_keyword_args,
        )
        if dlg.exec():
            values = dlg.section.extract_values()
            active_group = dlg.section.active_group_name(UNASSIGNED_TYPE, MULTIPLE_TYPES)
            component_id = self.data.add_feature(target, values, active_group)

            self.assign_feature_ids(target, active_group, feature_ids, component_id)
            self.data.commit()

    def to_text_file(self, model: GenericModel, file: str | Path):
        """
        Export the component to a text file suitable for comparing to a test baseline.

        This is intended to be used after building the component and preparing it to be sent to XMS. The output data
        includes all the attributes assigned in self.data, the UUID of the coverage the component expects to be attached
        to, and a mapping of feature and component IDs to verify that the right things were associated with each other.

        Args:
            model: Model used to interpret contents of data manager.
            file: Where to export to.
        """
        with open(file, 'w') as f:
            f.write('coverage\n')
            f.write(f'{self.cov_uuid}\n\n')
            f.write('updates\n')
            self._ids_to_text_file(f)
            self.data.to_text_file(model, f)

    def _ids_to_text_file(self, file: TextIO):
        """
        Write the contents of self.update_ids and self.label_texts to a text file.

        This data determines whether feature IDs have been correctly associated with component IDs. If the output is
        wrong, then features be associated with the wrong attributes, or no attributes at all.

        Args:
            file: Where to write to.
        """
        table = [('target', 'feature_id', 'component_id', 'label')]
        targets = self.update_ids.get(self.cov_uuid, {})
        for target in targets:
            name = TargetType(target).name
            for feature_id, component_id in targets.get(target, {}).items():
                label = ''
                with suppress(KeyError):
                    label = self.label_texts[self.cov_uuid][target][component_id]

                table.append((name, feature_id, component_id, label))

        table_to_text_file(table, file)
