"""CoverageComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from itertools import cycle
import os
from pathlib import Path
import shutil
from typing import Optional
import uuid

# 2. Third party modules
from PySide2.QtGui import QColor, QIcon
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.components.display.display_options_io import (
    read_display_option_ids, read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import DrawType, XmsDisplayMessage
from xms.core.filesystem import filesystem as io_util
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.line_style import LineOptions
from xms.guipy.data.point_symbol import PointOptions
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.category_display_options_list import CategoryDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import get_xms_icon

# 4. Local modules
from xms.gmi.components.gmi_component import GmiComponent, UNINITIALIZED_COMP_ID
from xms.gmi.data.coverage_data import CoverageData
from xms.gmi.data.generic_model import GenericModel, Section
from xms.gmi.gui.group_set_dialog import GroupSetDialog

INITIAL_POINT_ATT_ID_FILE = 'initial_point.attids'
INITIAL_POINT_COMP_ID_FILE = 'initial_point.compids'
INITIAL_ARC_ATT_ID_FILE = 'initial_arc.attids'
INITIAL_ARC_COMP_ID_FILE = 'initial_arc.compids'
INITIAL_POLYGON_ATT_ID_FILE = 'initial_polygon.attids'
INITIAL_POLYGON_COMP_ID_FILE = 'initial_polygon.compids'

PT_JSON = 'pt_display.json'
ARC_JSON = 'arc_display.json'
POLY_JSON = 'poly_display.json'

GMI_MULTIPLE_ASSIGNED_TYPE = -1
GMI_NONE_ASSIGNED_TYPE = -2
GMI_MULTIPLE_ASSIGNED_DISPLAY = "Multiple assigned"

DEFAULT_COLORS = [
    (255, 0, 0),  # red
    (0, 0, 255),  # blue
    (85, 170, 0),  # lighter green
    (255, 204, 0),  # yellow
    (0, 160, 203),  # cyan
    (255, 0, 204),  # magenta
    (153, 0, 255),  # purple
    (255, 102, 0),  # orange
    (51, 153, 51),  # dark green
    (153, 102, 0),  # brown
    (102, 153, 153),  # grey blue
    (153, 51, 51),  # dark red
    (255, 153, 255),  # light magenta
]


class CoverageComponent(GmiComponent):
    """
    A hidden Dynamic Model Interface (DMI) component for a GMI coverage.

    The CoverageComponent manages feature coverages. See `MaterialComponent`
    for material coverages.
    """
    def __init__(self, main_file: str, generic_model: GenericModel | None = None):
        """
        Initialize the component class.

        Args:
            main_file: The main file associated with this component.
            generic_model: Parameter definitions.
        """
        super().__init__(main_file, generic_model)
        self.cov_uuid = self.data.coverage_uuid
        self.tree_commands = [('Display Options...', 'open_display_options')]

        self.point_commands = [('Assign Properties...', 'open_assign_point')]
        self.arc_commands = [('Assign Properties...', 'open_assign_arc')]
        self.polygon_commands = [('Assign Properties...', 'open_assign_polygon')]
        self.point_dlg_title = 'Point Properties'
        self.arc_dlg_title = 'Arc Properties'
        self.polygon_dlg_title = 'Polygon Properties'

        # Derived classes may want to override these.
        self._category_dict = self._make_category_dict()
        self._inactive_group_name = 0  # Group name in GenericModel to use if no groups are active.
        self._multiple_active_group_name = -1  # Group name in GenericModel to use if multiple groups are active.
        self._show_groups = True  # Whether to show the group panel in the parameter assignment dialog.

        self._dlg_message = ''
        self._banner = None

        self._selected_att_ids = []
        self._selected_comp_ids = []

        self._ensure_display_option_files_exist()

    def _make_category_dict(self):
        """
        Initialize some internal data.

        This is needed during the constructor, and derived classes might want to change it, but passing it in
        seems inappropriate since the user doesn't care, so it's a method to allow overriding.
        """
        comp_dir = os.path.dirname(self.main_file)
        return {
            TargetType.point:
                {
                    'disp_opts_file': os.path.join(comp_dir, PT_JSON),
                    'option_class': PointOptions,
                    'id_file_prefix': 'pt_',
                    'reg_key': f'PT_{self.module_name}',
                },
            TargetType.arc:
                {
                    'disp_opts_file': os.path.join(comp_dir, ARC_JSON),
                    'option_class': LineOptions,
                    'id_file_prefix': 'arc_',
                    'reg_key': f'ARC_{self.module_name}',
                },
            TargetType.polygon:
                {
                    'disp_opts_file': os.path.join(comp_dir, POLY_JSON),
                    'option_class': PolygonOptions,
                    'id_file_prefix': 'poly_',
                    'reg_key': f'POLY_{self.module_name}',
                },
        }

    def _section(self, target_type: TargetType) -> Section:
        """
        Get a section from the coverage's model based on its `TargetType`.

        Args:
            target_type: `TargetType` for the desired section.

        Returns:
            The section for the feature type.
        """
        return self.data.generic_model.section_from_target_type(target_type)

    def create_event(self, lock_state: bool) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        Called when the component is created from nothing.

        Args:
            lock_state: Whether the component is locked for editing. Do not change the files if locked.

        Returns:
            A tuple of (messages, action_requests).

            - messages: List of tuples with the first element of the tuple being the message
              level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
            - action_requests: List of actions for XMS to perform.
        """
        messages = []
        action_requests = [self.get_display_options_action()]
        return messages, action_requests

    def save_to_location(self, new_path: str, save_type: str) -> tuple[str, list[tuple[str, str]], list[ActionRequest]]:
        """
        Save component files to a new location.

        Args:
            new_path: Path to the new save location.
            save_type: One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                - DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                  start with.
                - PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                  data must use relative file paths.
                - SAVE happens when re-saving this project.
                - SAVE_AS happens when saving a project in a new location. This happens the first time we save a
                  project.
                - UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder
                  in the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            A tuple of (new_main_file, messages, action_requests).
                - new_main_file: Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages: List of tuples with the first element of the tuple being the message level (DEBUG, ERROR,
                  WARNING, INFO) and the second element being the message text.
                - action_requests: List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            self.update_display_options_file(new_main_file, new_path)

        return new_main_file, messages, action_requests

    def get_initial_display_options(self, query: Query,
                                    params: list[dict]) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query: Object for communicating with XMS.
            params: Generic map of parameters. Contains selection map and component id files.

        Returns:
            A tuple of (messages, action_requests).

            - messages: List of tuples with the first element of the tuple being the message
              level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
            - action_requests: List of actions for XMS to perform.
        """
        # Query XMS for parent coverage's UUID if we don't know it yet.
        if not self.cov_uuid:
            self.cov_uuid = query.parent_item_uuid()
            self.data.coverage_uuid = self.cov_uuid
            self.data.commit()
        if not self.cov_uuid:
            return [('ERROR', 'Could not get the coverage UUID.')], []

        for id_file in [INITIAL_POINT_ATT_ID_FILE, INITIAL_ARC_ATT_ID_FILE, INITIAL_POLYGON_ATT_ID_FILE]:
            initial_att_file = os.path.join(os.path.dirname(self.main_file), id_file)
            if os.path.isfile(initial_att_file):  # Came from a model native read, initialize the component ids.
                self._update_component_ids_from_files()

        # Send the display options json files to XMS.
        self.display_option_list = []
        for d in self._category_dict.values():
            file = d['disp_opts_file']
            if Path(file).is_file():
                self.display_option_list.append(XmsDisplayMessage(file=file, edit_uuid=self.cov_uuid))
        return [], []

    def _update_component_ids_from_files(self):
        """Read att and comp ids from files and update in XMS.

        Called in get_initial_display_options() after a model native import.
        """
        comp_folder = os.path.dirname(self.main_file)
        file_names = [
            (INITIAL_POINT_ATT_ID_FILE, INITIAL_POINT_COMP_ID_FILE, TargetType.point),
            (INITIAL_ARC_ATT_ID_FILE, INITIAL_ARC_COMP_ID_FILE, TargetType.arc),
            (INITIAL_POLYGON_ATT_ID_FILE, INITIAL_POLYGON_COMP_ID_FILE, TargetType.polygon),
        ]

        for att_file, comp_file, target_type in file_names:
            att_file = os.path.join(comp_folder, att_file)
            comp_file = os.path.join(comp_folder, comp_file)

            att_ids = read_display_option_ids(att_file)
            comp_ids = read_display_option_ids(comp_file)
            io_util.removefile(att_file)
            io_util.removefile(comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(target_type, att_id, comp_id)

    def refresh_component_ids(self, new_comp_id: int, target_type: TargetType):
        """
        Load all the currently used component ids and clean out the unused ones.

        Called on OK of the assign feature dialogs.

        Keeps the files fresh and the data has been small enough so far.

        Args:
            new_comp_id: Component id of the feature that was just assigned.
            target_type: feature type.
        """
        # Load all currently used component ids
        self._query.load_component_ids(self, points=True, arcs=True, polygons=True)
        # Drop unused component ids from the xarray datasets
        disp_opts_file = self._category_dict[target_type]['disp_opts_file']
        comp_ids = list(self.comp_to_xms.get(self.cov_uuid, {}).get(target_type, {}).keys())
        comp_ids.append(new_comp_id)
        self.data.drop_unused_features(target_type, comp_ids)
        self.display_option_list.append(XmsDisplayMessage(file=disp_opts_file, edit_uuid=self.cov_uuid))

        # Rewrite the display option id files now to clear out non-existent component ids.
        self.update_id_files()
        self.data.commit()

    def update_id_files(self):
        """Write the display id files."""
        comp_dir = os.path.dirname(self.main_file)
        for k, v in self._category_dict.items():
            type_list = self._section(k).group_names
            prefix = v['id_file_prefix']
            for idx, ftype in enumerate(type_list):
                # id_file = f'{prefix}{ftype}.ids'
                id_file = os.path.join(comp_dir, f'{prefix}cat{idx}.ids')
                component_ids_with_type = self.data.component_ids_with_type(k, ftype)
                if len(component_ids_with_type) > 0:
                    write_display_option_ids(id_file, component_ids_with_type)
                else:
                    io_util.removefile(id_file)

    def update_display_options_file(self, new_main_file: str, new_path: str):
        """
        Generate new UUIDs for the component and display lists.

        Will commit data file in this method.

        Args:
            new_main_file: Path to the new component's main file
            new_path: The new component's directory.
        """
        new_data = CoverageData(new_main_file)
        # If duplicating, clear the coverage UUID. Will query XMS for the new coverage's UUID on the create event.
        new_data.coverage_uuid = ''
        # Set component UUID in arc display options file.
        new_comp_uuid = os.path.basename(new_path)
        for category in self._category_dict.values():
            disp_file = category['disp_opts_file']
            basename = os.path.basename(disp_file)
            fname = os.path.join(new_path, basename)
            json_dict = read_display_options_from_json(fname)
            json_dict['uuid'] = str(uuid.uuid4())  # Generate a new arc display list UUID.
            json_dict['comp_uuid'] = new_comp_uuid
            categories = CategoryDisplayOptionList()
            categories.from_dict(json_dict)
            write_display_options_to_json(fname, categories)
        new_data.commit()

    def open_assign_point(self, query: Query, params: list[dict],
                          win_cont: QWidget) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        Run the Assign Point dialog.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            win_cont: The window container.

        Returns:
            A tuple of (messages, action_requests).

            - messages: List of tuples with the first element of the tuple being the message
              level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
            - action_requests: List of actions for XMS to perform.
        """
        dlg_name = f'{self.module_name}.point_dlg'
        return self._assign_feature(
            params=params,
            parent=win_cont,
            query=query,
            dlg_name=dlg_name,
            window_title=self.point_dlg_title,
            target_type=TargetType.point
        )

    def open_assign_arc(self, query: Query, params: list[dict],
                        win_cont: QWidget) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        Opens the Assign arc dialog and saves component data state on OK.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            win_cont: The window container.

        Returns:
            A tuple of (messages, action_requests).

            - messages: List of tuples with the first element of the tuple being the message
              level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
            - action_requests: List of actions for XMS to perform.
        """
        dlg_name = f'{self.module_name}.arc_dlg'
        return self._assign_feature(
            params=params,
            parent=win_cont,
            query=query,
            dlg_name=dlg_name,
            window_title=self.arc_dlg_title,
            target_type=TargetType.arc
        )

    def open_assign_polygon(self, query: Query, params: list[dict],
                            win_cont: QWidget) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        Opens the Assign polygon dialog and saves component data state on OK.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            win_cont: The window container.

        Returns:
            A tuple of (messages, action_requests).

            - messages: List of tuples with the first element of the tuple being the message
              level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
            - action_requests: List of actions for XMS to perform.
        """
        dlg_name = f'{self.module_name}.polygon_dlg'
        return self._assign_feature(
            params=params,
            parent=win_cont,
            query=query,
            dlg_name=dlg_name,
            window_title=self.polygon_dlg_title,
            target_type=TargetType.polygon
        )

    def _assign_feature(
        self, params: list[dict], parent: QWidget, query: Query, dlg_name: str, window_title: str,
        target_type: TargetType
    ) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        Display the Assign feature dialog and persist data if accepted.

        Args:
            params: The ActionRequest parameter map
            parent: The parent window
            query: Object for communicating with XMS
            dlg_name: Unique name for this dialog. site-packages import path would make sense.
                            Name used as registry key for the dialog location and size.
            window_title: title of window
            target_type: feature type

        Returns:
            A tuple of (messages, action_requests).

            - messages: List of tuples with the first element of the tuple being the message
              level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
            - action_requests: List of actions for XMS to perform.
        """
        self._query = query
        comp_id = self._unpack_xms_data(params[0], target_type)
        if not self.selected_att_ids:
            return [('INFO', 'No feature selected. Select one or more features to assign boundary conditions.')], []

        section = self._section(target_type)
        values = self.data.feature_values(target_type, comp_id)
        if values:
            section.restore_values(values)
        dlg = GroupSetDialog(
            parent=parent,
            section=section,
            get_curve=self.data.get_curve,
            add_curve=self.data.add_curve,
            is_interior=False,
            dlg_name=dlg_name,
            window_title=window_title,
            multi_select_message=self._dlg_message,
            banner=self._banner,
            show_groups=self._show_groups,
            enable_unchecked_groups=False,
            dataset_callback=self._dataset_callback,
        )
        if dlg.exec():
            # Update the attribute datasets
            values = dlg.section.extract_values()
            # get the checked items
            active_group = dlg.section.active_group_name(self._inactive_group_name, self._multiple_active_group_name)
            comp_id = self.data.add_feature(target_type, values, active_group)
            # Associate all selected features with the new component id.
            for feature_id in self.selected_att_ids:
                self.update_component_id(target_type, feature_id, comp_id)
            self.refresh_component_ids(comp_id, target_type)
            self.data.commit()
        return [], []

    def _read_display_options(self) -> list[CategoryDisplayOptionList]:
        """
        Read display options from disk.

        Returns:
            The read display options, one for each target type.
        """
        cat_list = []
        for key, category in self._category_dict.items():
            disp_opts_file = category['disp_opts_file']
            json_dict = read_display_options_from_json(disp_opts_file)
            if len(json_dict) > 0:
                categories = CategoryDisplayOptionList()
                categories.from_dict(json_dict)
                cat_list.append(categories)
            else:
                cat_list.append(CategoryDisplayOptionList())
                cat_list[-1].target_type = key
        return cat_list

    def _write_display_options(self, category_lists: list[CategoryDisplayOptionList]):
        """
        Write display options to disk.

        Args:
            category_lists: Display options to write to disk. One for each target type.
        """
        for category_list in category_lists:
            if len(category_list.categories) < 1:
                continue
            disp_opts_file = self._category_dict[category_list.target_type]['disp_opts_file']
            write_display_options_to_json(disp_opts_file, category_list)
            self.display_option_list.append(
                XmsDisplayMessage(file=disp_opts_file, edit_uuid=self.cov_uuid, draw_type=DrawType.draw_at_ids)
            )

    def open_display_options(self, query: Query, params: list[dict],
                             parent: QWidget) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        Run the display options dialog.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            parent: The window container.

        Returns:
            A tuple of (messages, action_requests).

            - messages: List of tuples with the first element of the tuple being the message
              level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
            - action_requests: List of actions for XMS to perform.
        """
        cat_list = self._read_display_options()
        reg_keys = [self._category_dict[target_type]['reg_key'] for target_type in self._category_dict.keys()]
        dlg = CategoryDisplayOptionsDialog(cat_list, parent, package_name='xmsgmi', registry_keys=reg_keys)

        dlg.setWindowIcon(QIcon(get_xms_icon()))
        dlg.setModal(True)
        if dlg.exec():
            category_lists = dlg.get_category_lists()
            self._write_display_options(category_lists)
        return [], []

    def _ensure_display_option_files_exist(self):
        """Copies default point, arc, and polygon display option JSON files to the component directory if needed.

        Will create new random UUIDs for the display lists. Should only be called by the unmapped coverage on
        creation.
        """
        for k, v in self._category_dict.items():
            disp_opts_file = v['disp_opts_file']
            if not os.path.exists(disp_opts_file):
                categories = self._default_display_categories(k)
                if categories:
                    categories.file = disp_opts_file
                    categories.comp_uuid = self.uuid
                    write_display_options_to_json(disp_opts_file, categories)

    def _default_display_categories(self, target_type: TargetType) -> Optional[CategoryDisplayOptionList]:
        """
        Get the default display categories for arcs.

        Args:
            target_type: Type of feature (arcs, pts, polys).

        Returns:
            List of display categories
        """
        cats = CategoryDisplayOptionList()
        cats.target_type = target_type
        section = self._section(target_type)
        types_list = list(section.group_names)
        if len(types_list) < 1:
            return None
        pool = cycle(DEFAULT_COLORS)

        for idx, t in enumerate(types_list):
            cats.categories.append(CategoryDisplayOption())
            c = cats.categories[-1]
            c.options = self._category_dict[target_type]['option_class']()
            c.description = section.group(t).label
            c.id = idx
            prefix = self._category_dict[target_type]['id_file_prefix']
            # c.file = f'{prefix}{t}.ids'
            c.file = f'{prefix}cat{idx}.ids'
            clr = next(pool)
            c.options.color = QColor(clr[0], clr[1], clr[2], 255)
        return cats

    def _unpack_xms_data(self, param_map: dict, target_type: TargetType) -> int:
        """
        Unpack the selection info and component id maps sent by XMS.

        Args:
            param_map: The ActionRequest parameter map.
            target_type: Feature type.

        Returns:
            Component id of atts to display.
        """
        # Get the XMS attribute ids of the selected features (if any)
        self.selected_att_ids = param_map.get('selection', [])
        if not self.selected_att_ids:
            return UNINITIALIZED_COMP_ID

        # Get the component id map of the selected features (if any).
        comp_id = UNINITIALIZED_COMP_ID
        id_files = param_map.get('id_files', [])
        if id_files and id_files[0]:
            if target_type == TargetType.point:
                files_dict = {'POINT': (id_files[0], id_files[1])}
            elif target_type == TargetType.arc:
                files_dict = {'ARC': (id_files[0], id_files[1])}
            else:  # target_type == TargetType.polygon:
                files_dict = {'POLYGON': (id_files[0], id_files[1])}
            self.load_coverage_component_id_map(files_dict)
            comp_id = self._check_selected_types(target_type)
        # Clean up temp files dumped by SMS.
        shutil.rmtree(os.path.join(os.path.dirname(self.main_file), 'temp'), ignore_errors=True)
        return comp_id

    def _check_selected_types(self, target_type: TargetType) -> int:
        """
        Determine which attributes to display in Assign feature dialog and any warning message that should be added.

        Args:
            target_type: Feature type.

        Returns:
            Component id of attributes to display.
        """
        num_features = len(self.selected_att_ids)
        feature_dict = {
            TargetType.point: 'points',
            TargetType.arc: 'arcs',
            TargetType.polygon: 'polygons',
        }
        fstr = feature_dict[target_type]
        if num_features == 1:  # 1 feature selected, use those atts
            comp_id = self.get_comp_id(target_type, self.selected_att_ids[0])
            self._selected_comp_ids.append(comp_id)
            return comp_id
        else:  # More than one feature selected, check types
            self._selected_comp_ids = list(self.comp_to_xms[self.cov_uuid][target_type].keys())
            self._dlg_message = f'Multiple {fstr} selected. Changes will apply to all selected {fstr}.'
            # If there are multiple entities selected with the same component id, display those attributes. Otherwise
            # display an empty, default dialog.
            return UNINITIALIZED_COMP_ID if not self._selected_comp_ids else self._selected_comp_ids[0]
