"""Code dealing with mapping tables."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.tool_core.table_definition import FloatColumnType, IntColumnType, StringColumnType, TableDefinition

# 4. Local modules


def names() -> list[str]:
    """Returns the list of mapping table names.

    Returns:
        See description.
    """
    return [
        'ROUGHNESS', 'ROUGH_EXP', 'INTERCEPTION', 'RETENTION', 'GREEN_AMPT_INFILTRATION',
        'GREEN_AMPT_INITIAL_SOIL_MOISTURE', 'RICHARDS_EQN_INFILTRATION_BROOKS', 'RICHARDS_EQN_INFILTRATION_HAVERCAMP',
        'AREA_REDUCTION'
    ]


def dataset_param(name: str) -> str:
    """Returns the name of the generic model dataset parameter associated with the mapping table.

    Args:
        name: Name of the mapping table.

    Returns:
        See description.
    """
    return f'{name.lower()}_index_map_dataset'


def table_param(name: str) -> str:
    """Returns the name of the generic model table parameter associated with the mapping table.

    Args:
        name: Name of the mapping table.

    Returns:
        See description.
    """
    return f'{name.lower()}_table'


def table_def(name: str) -> TableDefinition:
    """Returns the TableDefinition for the mapping table."""
    if name == 'TIME_SERIES_INDEX':
        columns = [
            IntColumnType(header='ID', default=1, tool_tip=''),
            StringColumnType(header='Time series name', default=''),
        ]
    else:
        columns = [
            IntColumnType(header='Index value', default=1, tool_tip=''),
            StringColumnType(header='Description1', default=''),
            StringColumnType(header='Description2', default=''),
        ]
        map_specific_columns = _columns(name)
        for header, tool_tip in map_specific_columns:
            columns.append(FloatColumnType(header=header, tool_tip=tool_tip, default=0.0))
    return TableDefinition(columns)


def _columns(name: str) -> list[tuple[str, str]]:
    """Given the mapping table name, returns the list of column tuples (column header, tool tip)."""
    match name:
        case 'ROUGHNESS':
            return [('SURF_ROUGH', 'Surface roughness (n)')]
        case 'ROUGH_EXP':
            return [('ROUGH_EXP', 'Surface Roughness Exponent (b)')]
        case 'INTERCEPTION':
            return [('STOR_CAPY', 'Storage capacity (mm)'), ('INTER_COEF', 'Interception coeff')]
        case 'RETENTION':
            return [('RETENTION_DEPTH', 'Retention depth (d<sub>ret</sub>) (mm)')]
        case 'GREEN_AMPT_INFILTRATION':
            return [
                ('HYDR_COND', 'K<sub>s</sub> Saturated Hydraulic conductivity (cm/hr)'),
                ('CAPIL_HEAD', '&psi;<sub>f</sub> Capillary Suction Head (cm)'),
                ('POROSITY', '&theta;<sub>s</sub> Porosity'),
                ('PORE_INDEX', '&lambda; Pore Index Value'),
                ('RESID_SAT', '&theta;<sub>r</sub> Residual Saturation'),
                ('FIELD_CAPACITY', '&theta;<sub>f</sub> Field Capacity'),
                ('WILT_POINT', '&theta;<sub>wp</sub> Wilting point'),
            ]
        case 'RICHARDS_EQN_INFILTRATION_BROOKS':
            return [
                ('HYD_COND', 'K<sub>s</sub> Saturated Hydraulic conductivity (cm/hr)'),
                ('POROSITY', '&theta;<sub>s</sub> Porosity'),
                ('RESID_SAT', '&theta;<sub>r</sub> Residual Saturation'),
                ('SOIL_MOIST', '&theta;<sub>i</sub> Initial soil moisture'),
                ('WILT_POINT', '&theta;<sub>wp</sub> Wilting point'),
                ('FIELD_CAPACITY', '&theta;<sub>fc</sub> Field capacity'),
                ('DEPTH', 'd Total layer depth (cm)'),
                ('LAMBDA', '&lambda;'),
                ('BUB_PRESS', '&psi;<sub>b</sub> Bubbling pressure (cm)'),
                ('DELTA_Z', '&Delta;z Numerical solver layer depth (cm)'),
            ]
        case 'RICHARDS_EQN_INFILTRATION_HAVERCAMP':
            return [
                ('HYD_COND', 'K<sub>s</sub> Saturated Hydraulic conductivity (cm/hr)'),
                ('POROSITY', '&theta;<sub>s</sub> Porosity'),
                ('RESID_SAT', '&theta;<sub>r</sub> Residual Saturation'),
                ('SOIL_MOIST', '&theta;<sub>i</sub> Initial soil moisture'),
                ('WILT_POINT', '&theta;<sub>wp</sub> Wilting point'),
                ('FIELD_CAPACITY', '&theta;<sub>fc</sub> Field capacity'),
                ('DEPTH', 'd Total layer depth (cm)'),
                ('ALPHA', '&alpha; Havercamp factor alpha'),
                ('BETA', '&beta; Havercamp factor beta'),
                ('A', 'A Havercamp factor A'),
                ('B', 'B Havercamp factor B'),
                ('DELTA_Z', '&Delta;z Numerical solver layer depth (cm)'),
            ]
        case 'GREEN_AMPT_INITIAL_SOIL_MOISTURE':
            return [('SOIL_MOISTURE', '&theta;<sub>i</sub> Initial soil moisture')]
        case 'AREA_REDUCTION':
            return [('IMPERVIOUS_AREA', 'Impervious area fraction')]
        case 'TIME_SERIES_INDEX':
            return [('ID', ''), ('Name', '')]
        case 'OVERLAND_BOUNDARY':
            return [('BDY_TYPE', 'Boundary Type Code'), ('BDY_PARAM', 'Second parameter')]
        case _:
            raise ValueError(f'Mapping table {name} unsupported.')
