"""Module for the debug_pause function."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['debug_pause']

# 1. Standard Python modules
import os

# 2. Third party modules
from PySide2.QtCore import QObject, Qt, QThread, Signal, Slot

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv

# 4. Local modules
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.dialog_util import ensure_qapplication_exists


def debug_pause(message: str = ''):
    """
    Opens an OK dialog.

    Can be used by writing this code where you want to debug:

    ```
    from xms.guipy import debug pause
    debug_pause()
    ```

    ...then set a breakpoint right after the call and run the code. When the dialog pops up, you can attach a debugger
    to the process ID shown in the dialog and then click OK. The debugger should break in and pause the script. If you
    have multiple pauses, you can pass a message to distinguish them.

    Note that attaching a debugger can be slow sometimes. Make sure it says it's really finished attaching before
    clicking OK or you might miss your chance to break in.

    Args:
        message: Message to display in the message box. If provided, it will be shown below the process ID.
    """
    if ensure_qapplication_exists().thread() == QThread.currentThread():
        # We're running on the main thread. It's safe to pop up a dialog here.
        message = f'{os.getpid()}\n{message}'
        app_name = XmEnv.xms_environ_app_name()
        message_box.message_with_ok(parent=None, message=message, app_name=app_name)
    else:
        # We're not on the main thread. Probably a feedback worker thread. We need to migrate.
        DebugPauseProxy(message)


class DebugPauseProxy(QObject):
    """
    A class that makes an OK dialog appear when constructed.

    If you pop up a dialog on the GUI thread, everything works fine, but trying to do it from another thread
    results in an empty dialog and hangs the whole application.

        "Qt does not support gui operations of any kind outside the main thread. Use `QThread` and send a signal back
        to the main thread with the message you want to show."

        -- ekhumoro, https://stackoverflow.com/questions/54428169/


    This proxy object is based on an answer by Tim Woocker at https://stackoverflow.com/a/68137720 and does just that.

    Constructing this on the main thread will hang the application.
    """

    called = Signal()

    def __init__(self, message):
        """
        Initialize the object.

        Args:
            message: The message to display.
        """
        super().__init__()
        self.message = message
        app = ensure_qapplication_exists()
        main_thread = app.thread()

        if QThread.currentThread() == main_thread:
            # We're going to block whichever thread constructed us and wait until the main thread does stuff.
            # If the main thread constructed us, then we'll block the main thread until the main thread does
            # stuff, which will never happen, because it's blocked.
            #
            # Better to die promptly with a clear error than hang deep in the guts of Qt.
            raise AssertionError('Constructing DebugPauseProxy on the main thread will hang the application.')

        # We need to get Qt to run a slot on the main thread. Qt always runs slots on the thread that owns an object,
        # so we'll make the main thread our owner and signal Qt to run our slot.
        self.moveToThread(main_thread)

        # Unsure if this matters, but it was in the source material. Setting our parent prevents us from being
        # garbage-collected at a bad time.
        self.setParent(app)

        # Use of `Qt.BlockingQueuedConnection` means whenever the signal is emitted, the emitter blocks until all the
        # slots finish. This prevents the emitting thread (which constructed this object) from continuing until the
        # slot finishes (on the main thread).
        self.called.connect(self.pause, type=Qt.BlockingQueuedConnection)
        self.called.emit()

    @Slot()
    def pause(self):
        """Pause until the user clicks OK."""
        # This is part of the implementation for debug_pause, so it's allowed to call debug_pause.
        debug_pause(self.message)  # noqa: AQU100

        # We set our parent to the main app in the constructor to avoid being garbage collected. Our job is done
        # now, so remove the parent and allow garbage-collection again.
        self.setParent(None)
