"""XMS project explorer tree item picker model."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QIcon, QStandardItem, QStandardItemModel

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util

# 4. Local modules
from xms.guipy.resources import resources_util

col0 = 0  # Column 0 (to avoid magic number 0's everywhere).


class ProjectExplorerModel(QStandardItemModel):
    """A model showing items from the Project Explorer in a tree view."""
    def __init__(
        self,
        parent=None,
        root_node=None,
        selectable_item_type='',
        previous_selection='',
        override_icon=None,
        show_root=False,
        selectable_xms_types=None,
        allow_multi_select=False
    ):
        """
        Initializes the class.

        Args:
            parent (QObject): The parent Qt object.
            root_node (TreeNode): Root of the tree to display in the dialog
            selectable_item_type (str): Type of item in project_tree that the user can select. Only these items
                and their ancestors will be shown in the widget.
            parent: Parent window
            previous_selection (str or iterable): UUID of the previous selection. If provided, that item will be
                selected when the dialog appears. Can be an iterable container of previously selected UUIDs (implies
                allow_multi_select=True).
            override_icon (callable): Callable method to provide icons that override the default for a tree
                item type. Method should take a TreeNode and return an icon path if it is overriden, else empty string.
                Icon path must be in whatever format that works in the calling package.
            show_root (bool): If True, root of the tree will be shown
            selectable_xms_types (list): XMS tree item types of selectable items that are not directly supported
                by xms.api.
            allow_multi_select (bool): If True, multiple tree items may be selected.
        """
        super().__init__(parent)
        self._root_node = root_node
        self._selectable_item_type = selectable_item_type  # Type of items we want to select.

        # Selectable XMS tree item types that are not directly supported by xms.api.
        self._selectable_xms_types = selectable_xms_types if selectable_xms_types else []
        self._override_icon = override_icon
        self._show_root = show_root
        self._in_on_item_changed = False  # Flag to avoid recursion when changing checked state
        self._allow_multi_select = allow_multi_select
        self._icon_paths = {}

        # Signals
        self.itemChanged.connect(self._on_item_changed)

        self.initialize(
            root_node, previous_selection, override_icon, show_root, self._selectable_xms_types, allow_multi_select
        )

    def initialize(
        self, root_node, previous_selection, override_icon, show_root, selectable_xms_types, allow_multi_select
    ):
        """
        Initializes the dialog using the arguments supplied.

        Args:
            root_node (TreeNode): Root of the tree to display in the dialog
            previous_selection (str or iterable): UUID of the previous selection. If provided, that item will be
                selected when the dialog appears. Can be an iterable container of previously selected UUIDs (implies
                allow_multi_select=True).
            override_icon (callable): Callable method to provide icons that override the default for a tree item
                type. Method should take a TreeNode and return an icon path if it is overriden, else empty string. Icon
                path must be in whatever format that works in the calling package.
            show_root (bool): If True, root of the tree will be shown
            selectable_xms_types (list): XMS tree item types of selectable items that are not directly
                supported by xms.api.
            allow_multi_select (bool): If True, multiple tree items may be selected.
        """
        self._show_root = show_root
        self._override_icon = override_icon if override_icon else resources_util.get_tree_icon_from_xms_typename
        self._selectable_xms_types = selectable_xms_types
        self._allow_multi_select = allow_multi_select
        self._root_node = tree_util.trim_tree_to_items_of_type(
            root_node, [self._selectable_item_type], self._selectable_xms_types
        )
        if not self._root_node:
            raise Exception('Nothing.')
        self._add_tree_items(self._root_node, self)

        if not previous_selection:
            return  # No known previous selection.

        # If the previous selection is a single UUID, put it in a list. When multi-select is enabled, there may be
        # more than one previously selected item.
        if isinstance(previous_selection, str):
            previous_selection = [previous_selection]

        # Check the previously selected item(s).
        for item in self.tree_items():
            if item.flags() & Qt.ItemIsUserCheckable and item.data(Qt.UserRole).uuid in previous_selection:
                item.setCheckState(Qt.Checked)
                if not self._allow_multi_select:
                    return  # Only one item may be selected at a time.

    def _uncheck_other_items(self, except_item):
        """
        Unchecks all checkable items except for except_item.

        Called from self._on_item_changed

        Args:
            except_item(QStandardItem): Item to leave unchanged.
        """
        for item in self.tree_items():
            if item and item.flags() & Qt.ItemIsUserCheckable and item is not except_item:
                item.setCheckState(Qt.Unchecked)

    def _on_item_changed(self, item, column=col0):
        """
        Slot called when itemChanged signal sent. Unchecks other items.

        Args:
            item (QStandardItem): The item that was changed.
            column (int): The column.
        """
        if self._in_on_item_changed:
            return  # We're already in this function. Return to stop recursion.

        self._in_on_item_changed = True
        item_checked = item and item.flags() & Qt.ItemIsUserCheckable and item.checkState() == Qt.Checked
        if item_checked and not self._allow_multi_select:
            # If the item was checked and multi-select is disabled, uncheck all other items (act like a radio group).
            self._uncheck_other_items(item)
        self._in_on_item_changed = False

    def _set_icon(self, tree_node, standard_item):
        """
        Sets the icon for the item.

        Args:
            tree_node (TreeNode): Project explorer tree item to set icon for.
            standard_item (QStandardItem): The item.
        """
        if self._override_icon:  # Check override method first to see if an icon has been defined for this tree node.
            icon_path = self._override_icon(tree_node)
            if icon_path:
                standard_item.setIcon(QIcon(icon_path))
                return

        item_type = type(tree_node.data)
        icon_path = ':resources/icons/folder.svg'  # Default to folder icon
        if item_type == xmd.UGridItem:
            icon_path = ':resources/icons/UGrid_Module_Icon.svg'
        elif item_type == xmd.DatasetItem:
            icon_path = ':resources/icons/dataset_cells_active.svg'
            if tree_node.data_location == 'NODE':
                icon_path = ':resources/icons/dataset_points_active.svg'
        elif item_type == xmd.SimulationItem:
            icon_path = ':resources/icons/simulation.svg'
        elif item_type == xmd.CoverageItem:
            icon_path = ':resources/icons/coverage_active.svg'
        elif item_type == xmd.ComponentItem:
            icon_path = ':resources/icons/component.svg'
            # Check if a specific icon has been passed in for this component type.
            comp_type = f'{item_type}#{tree_node.model_name}#{tree_node.unique_name}'
            if comp_type in self._icon_paths:
                icon_path = self._icon_paths[comp_type]
        standard_item.setIcon(QIcon(icon_path))

    def _add_tree_items(self, tree_data, parent):
        """
        Recursively adds items to the tree.

        Args:
            tree_data (TreeNode): Root of the project explorer tree to add items to
            parent (QStandardItem): The parent QStandardItem. The first time it should be self
                (QStandardItemModel).
        """
        if not tree_data:
            return

        if self._show_root and parent is self:
            item = QStandardItem()
            self.appendRow(item)
            self._add_single_tree_item(item, tree_data)
            parent = item

        for child in tree_data.children:
            item = QStandardItem() if parent is self else QStandardItem(parent)
            if not self._show_root and parent is self:
                self.appendRow(item)
            elif parent is not self:
                parent.appendRow(item)

            self._add_single_tree_item(item, child)

            # Recurse on children
            self._add_tree_items(child, item)

    def _add_single_tree_item(self, q_item, tree_node):
        """
        Add a tree item node to the model.

        Args:
            q_item (QStandardItem): The Qt item to add
            tree_node (TreeNode): The tree item node to add
        """
        q_item.setText(tree_node.name)
        q_item.setFlags(q_item.flags() & ~(Qt.ItemIsSelectable | Qt.ItemIsUserCheckable))  # Make nothing selectable
        self._set_icon(tree_node, q_item)
        if tree_node.uuid != '00000000-0000-0000-0000-000000000000':
            q_item.setData(tree_node, Qt.UserRole)  # Store uuid

        # Set selectable item to have checkboxes and store it's uuid and type
        types_match = type(tree_node.data) is self._selectable_item_type
        types_match = types_match or tree_node.item_typename in self._selectable_xms_types
        if not tree_node.is_ptr_item and types_match:
            q_item.setFlags(q_item.flags() | Qt.ItemIsUserCheckable)
            q_item.setCheckState(Qt.Unchecked)

    def get_selected_item_node(self):
        """
        Returns the selected item's TreeNode or None if nothing selected.

        If multi-select is enabled, return value is a list of the selected tree items. Empty list if no items
        selected.

        Returns:
            See description.
        """
        # Ensure return value is a list if multi-select is enabled and a single str (or None) when disabled.
        selected_nodes = [] if self._allow_multi_select else None

        for item in self.tree_items():
            if item.flags() & Qt.ItemIsUserCheckable and item.checkState() == Qt.Checked:
                data = item.data(Qt.UserRole)
                if self._allow_multi_select:  # If more than one item can be selected, iterate the whole tree.
                    selected_nodes.append(data)
                else:  # If only one item can be selected, we are done.
                    return data

        return selected_nodes

    def has_selectable_items(self):
        """Return True if trimmed tree has items."""
        return self._root_node is not None

    def tree_items(self, parent=None):
        """
        Iterates through the first column of the tree.

        Args:
            parent (QStandardItem): The parent item to traverse.

        Yields:
            (QStandardItem): The item in column 0 of the tree.
        """
        if parent:
            row_count = parent.rowCount()
        else:
            row_count = self.rowCount()
        for row in range(row_count):
            if parent:
                item = parent.child(row, col0)
            else:
                item = self.item(row, col0)
            yield item
            yield from self.tree_items(item)
