"""Topology Processor for printing and calculating values."""
# 1. Standard python modules
import math

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.hecras._TopologyCreatorFromDatasets import TopologyCreatorFromDatasets


class TopologyProcessor:
    """Topology Processor for printing and calculating values."""
    rawpolygons = None
    nodes = None
    units = None
    vertices = None     # Coordinates of the nodes
    centers = None
    realcenters = None
    boundary = None
    bound_in_link = None
    link_nodes = None       # From_node To_Node
    link_polygons = None        # Left_Polygon Right_Polygon
    perimeter = None
    number_of_polygons = None
    sortedlinks = None
    nodecells = None        # Cells surrounding each node oriented counterclockwise
    nodelinks = None
    cells_areas = None
    cells_center_coordinate = None
    cells_face_and_orientation_info = None
    cells_face_and_orientation_values = None
    cells_face_point_indexes = None
    face_points_cell_info = None
    face_points_cell_index_values = None
    face_points_coordinate = None
    face_points_face_and_orientation_info = None
    face_points_face_and_orientation_values = None
    face_points_is_perimeter = None
    faces_cell_indexes = None
    faces_face_point_indexes = None
    faces_normal_unit_vector_and_length = None
    faces_perimeter_info = None
    faces_perimeter_values = None

    def __init__(self, rawpolygons, nodes, units):
        """Constructor.

        Args:
            rawpolygons (list): Polygons of the geometry
            nodes (list): Locations of the geometry points
            units (str): The geometry's horizontal projection units
        """
        self.rawpolygons = rawpolygons
        self.nodes = nodes
        self.units = units

    def process(self):
        """Formatted printer and parent function for the Class."""
        print("    ==============================")
        print("    Creating datasets for HDF file")
        print("    ==============================")
        self.generate_topo()
        self.get_cells_center_coordinate()
        self.get_cells_face_and_orientation()
        self.get_cells_face_point_indexes()
        self.get_face_points_cell_index()
        self.get_face_points_coordinate()
        self.get_face_points_face_and_orientation()
        self.get_face_points_is_perimeter()
        self.get_faces_cell_indexes()
        self.get_faces_face_point_indexes()
        self.get_faces_normal_unit_vector_and_length()
        self.get_faces_perimeter()
        self.get_perimeter()
        print("    ================")
        print("    Datasets created")
        print("    ================")

    def generate_topo(self):
        """Assigns values for all the data members."""
        topo = TopologyCreatorFromDatasets(self.rawpolygons, self.nodes, self.units)
        topo.process()
        self.units = topo.units
        self.vertices = topo.topo_vertices
        self.link_nodes = topo.topo_link_nodes
        self.link_polygons = topo.topo_link_polygons
        self.perimeter = topo.topo_perimeter
        self.centers = topo.topo_centers
        self.boundary = topo.topo_boundary
        self.bound_in_link = topo.topo_bound_in_link[:-1]       # The last value is equal to the first
        self.realcenters = topo.realcenters
        self.number_of_polygons = len(topo.polygons)
        self.nodecells = topo.topo_vertices_polygons
        self.celllinks = topo.topo_polygon_links
        self.nodelinks = topo.topo_vertices_links
        self.cells_areas = topo.polygons_area

    def get_cells_center_coordinate(self):
        """Assigns cells_center_coordinate with the centers value."""
        self.cells_center_coordinate = self.centers

    def get_cells_face_and_orientation(self):
        """Assigns values for cells face and orientation info and values."""
        celllinks = self.celllinks
        info = []
        values = []
        accum = 0
        for celllink in celllinks:
            link = celllink[1]
            counter = len(link)
            for elem in link:
                values.append(elem)
            info.append([accum, counter])
            accum = accum + counter
        self.cells_face_and_orientation_info = info
        self.cells_face_and_orientation_values = values
        pass

    def get_cells_face_point_indexes(self):
        """Assigns cells face point indexes with a list of nodes and tails.

        Must run after getCellFacePointIndex() to have a populated sortedlinks
        """
        listnodes = []
        maxnodes = 0
        for celllink in self.celllinks:
            links = celllink[1]
            nodes = []
            for link in links:
                nodeini = self.link_nodes[link[0]][0]
                nodefin = self.link_nodes[link[0]][1]
                if nodeini not in nodes:
                    nodes.append(nodeini)
                if nodefin not in nodes:
                    nodes.append(nodefin)
            listnodes.append(nodes)
        for node in listnodes:
            if len(node) > maxnodes:
                maxnodes = len(node)
        newlistnodes = []
        for node in listnodes:
            tail = []
            if len(node) < maxnodes:
                for _ in range(0, maxnodes - len(node)):
                    tail.append(-1)
            newlistnodes.append(node + tail)
        self.cells_face_point_indexes = newlistnodes

    def find_links_for_polygon(self, polygon_i):
        """Finds links for given polygon.

        Args:
            polygon_i (:obj:`xms.data_objects.Parameters.Spatial.Polygon`): Polygon to find links for.

        Returns:
            (:obj:`list` of int): List of links.
        """
        links = []
        for link_i in range(0, len(self.link_polygons)):
            polygons = self.link_polygons[link_i]
            if polygon_i in polygons:
                links.append(link_i)
        return links

    def neighbor_polygons(self, polygon1, polygon2):
        """Checks if the given polygons are neighbors.

        Args:
            polygon1 (:obj:`xms.data_objects.Parameters.Spatial.Polygon`): First polygon.
            polygon2 (:obj:`xms.data_objects.Parameters.Spatial.Polygon`): Second polygon.

        Returns:
            (bool): True if polygons are neighbors, False if not.
        """
        links1 = self.find_links_for_polygon(polygon1)
        links2 = self.find_links_for_polygon(polygon2)
        neighbors = False
        for l1 in links1:
            if l1 in links2:
                neighbors = True
        if len(links1) == 1 and len(links2) == 1:  # Trick for the consecutive border one-link-polygons.
            neighbors = True
        return neighbors

    def get_face_points_cell_index(self):
        """Assigns face points cell info and index values."""
        nodecells = self.nodecells
        info = []
        values = []
        accum = 0
        for polygon in nodecells:
            counter = len(polygon)
            for elem in polygon:
                values.append(elem)
            info.append([accum, counter])
            accum = accum + counter
        self.face_points_cell_info = info
        self.face_points_cell_index_values = values

    def get_face_points_coordinate(self):
        """Assigns face points coordinate with vertices."""
        self.face_points_coordinate = self.vertices

    def get_face_points_face_and_orientation(self):
        """Assigns face points face and orientation info and values."""
        nodelinks = self.nodelinks
        info = []
        values = []
        accum = 0
        for nodelink in nodelinks:
            link = nodelink[1]
            counter = len(link)
            for elem in link:
                values.append(elem)
            info.append([accum, counter])
            accum = accum + counter
        self.face_points_face_and_orientation_info = info
        self.face_points_face_and_orientation_values = values

    def get_face_points_is_perimeter(self):
        """Populates face points is perimeter."""
        # Simplest way to do it. Go through the links that are perimeter and extract their node_i.
        # Since they are oriented it would work.

        listperimeter = []
        # Create empty list
        for _ in range(0, len(self.vertices)):
            listperimeter.append(0)     # Not in perimeter by default
        for link_i in range(0, len(self.link_nodes)):       # For each link
            link_n = self.link_nodes[link_i]
            node_i = link_n[0]
            if self.perimeter[link_i] == 1:
                listperimeter[node_i] = 1
        self.face_points_is_perimeter = listperimeter

    def get_faces_cell_indexes(self):
        """Assigns faces cell indexex with link_polygons."""
        self.faces_cell_indexes = self.link_polygons

    def get_faces_face_point_indexes(self):
        """Assigns faces_face_point_indexes with link_nodes."""
        self.faces_face_point_indexes = self.link_nodes

    def get_faces_normal_unit_vector_and_length(self):
        """Populates faces_normal_unit_vector_and_length."""
        values = []
        for link_n in self.link_nodes:
            p1 = self.vertices[link_n[0]]
            p2 = self.vertices[link_n[1]]
            x1 = p1[0]
            y1 = p1[1]
            x2 = p2[0]
            y2 = p2[1]
            length = ((x2 - x1) ** 2 + (y2 - y1) ** 2) ** 0.5
            angle = math.atan2(y2 - y1, x2 - x1)
            sin = math.sin(angle)
            cos = math.cos(angle)
            values.append([sin, -cos, length])
        self.faces_normal_unit_vector_and_length = values

    def get_faces_perimeter(self):
        """Assigns faces_perimeter_info and  _values."""
        values = self.boundary
        info = []
        b_i = 0
        boundary = 0
        for i in range(0, len(self.link_nodes)):
            if b_i < len(self.bound_in_link):
                link = self.bound_in_link[b_i]
            else:
                link = -1
            if i == link:
                info.append([boundary, 1])
                b_i = b_i + 1
                boundary = boundary + 1
            else:
                info.append([boundary, 0])
        self.faces_perimeter_info = info
        self.faces_perimeter_values = values

    def get_perimeter(self):
        """Assigns perimeter to boundary."""
        self.perimeter = self.boundary
