"""ComponentCreator class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query
from xms.api.tree import tree_util
from xms.data_objects.parameters import Component, Simulation

# 4. Local modules
from xms.mf6.components import dmi_util
from xms.mf6.data import data_util
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.file_io import io_util
from xms.mf6.geom.ugrid_builder import UGridBuilder
from xms.mf6.misc.settings import Settings

# ftype: (XML unique name, XML take name parameter)
_comp_info = {
    'MFSIM6': ('Sim_Manager', 'MODFLOW 6#Sim_Manager'),
    'TDIS6': ('TDIS6', 'TDIS'),
    'IMS6': ('IMS6', 'IMS'),
    'EMS6': ('EMS6', 'EMS'),

    # Exchanges
    'GWF6-GWF6': ('GWF6-GWF6', 'GWF-GWF'),
    'GWF6-GWT6': ('GWF6-GWT6', 'GWF-GWT'),
    'GWF6-GWE6': ('GWF6-GWE6', 'GWF-GWE'),
    'GWF6-PRT6': ('GWF6-PRT6', 'GWF-PRT'),
    'GWT6-GWT6': ('GWT6-GWT6', 'GWT-GWT'),
    'GWE6-GWE6': ('GWE6-GWE6', 'GWE-GWE'),

    # GWF
    'BUY6': ('BUY6', 'BUY'),
    'CHD6': ('CHD6', 'CHD'),
    'CSUB6': ('CSUB6', 'CSUB'),
    'DIS6': ('DIS6', 'DIS'),
    'DISV6': ('DISV6', 'DISV'),
    'DISU6': ('DISU6', 'DISU'),
    'DRN6': ('DRN6', 'DRN'),
    'EVT6': ('EVT6', 'EVT'),
    'EVTA6': ('EVTA6', 'EVTA'),
    'GHB6': ('GHB6', 'GHB'),
    'GNC6': ('GNC6', 'GNC'),
    'GWF6': ('GWF6', 'GWF'),
    'HFB6': ('HFB6', 'HFB'),
    'IC6': ('IC6', 'IC'),
    'LAK6': ('LAK6', 'LAK'),
    'MAW6': ('MAW6', 'MAW'),
    'MVR6': ('MVR6', 'MVR'),
    'NPF6': ('NPF6', 'NPF'),
    'OBS6': ('OBS6', 'OBS'),
    'OC6': ('OC6', 'OC'),
    'POBS6': ('POBS6', 'POBS'),
    'RCH6': ('RCH6', 'RCH'),
    'RCHA6': ('RCH6', 'RCH'),
    'RIV6': ('RIV6', 'RIV'),
    'SFR6': ('SFR6', 'SFR'),
    'STO6': ('STO6', 'STO'),
    'SWI6': ('SWI6', 'SWI'),
    'UZF6': ('UZF6', 'UZF'),
    'VSC6': ('VSC6', 'VSC'),
    'WEL6': ('WEL6', 'WEL'),
    'ZONE6': ('ZONE6', 'Zones'),

    # GWT
    'ADV6': ('ADV6', 'ADV'),
    'CNC6': ('CNC6', 'CNC'),
    'DSP6': ('DSP6', 'DSP'),
    'FMI6': ('FMI6', 'FMI'),
    'GWT6': ('GWT6', 'GWT'),
    'IST6': ('IST6', 'IST'),
    'LKT6': ('LKT6', 'LKT'),
    'MDT6': ('MDT6', 'MDT'),
    'MST6': ('MST6', 'MST'),
    'MVT6': ('MVT6', 'MVT'),
    'MWT6': ('MWT6', 'MWT'),
    'SFT6': ('SFT6', 'SFT'),
    'SRC6': ('SRC6', 'SRC'),
    'SSM6': ('SSM6', 'SSM'),
    'UZT6': ('UZT6', 'UZT'),

    # GWE
    'GWE6': ('GWE6', 'GWE'),
    'CND6': ('CND6', 'CND'),
    'CTP6': ('CTP6', 'CTP'),
    'ESL6': ('ESL6', 'ESL'),
    'EST6': ('EST6', 'EST'),
    'LKE6': ('LKE6', 'LKE'),
    'MVE6': ('MVE6', 'MVE'),
    'MWE6': ('MWE6', 'MWE'),
    'SFE6': ('SFE6', 'SFE'),
    'UZE6': ('UZE6', 'UZE'),

    # PRT
    'PRT6': ('PRT6', 'PRT'),
    'MIP6': ('MIP6', 'MIP'),
    'PRP6': ('PRP6', 'PRP'),
}


class ComponentCreator:
    """Creates components for a simulation."""
    def __init__(self):
        """Initializes the class."""
        self.model_mname_uuids = {}  # Dict of model mnames and the model uuids

    def create_components_from_simulation(self, mfsim, query, ugrid_uuid, model_str) -> str:
        """Creates all the components required by the simulation.

        Args:
            mfsim (MfsimData): Data class containing all simulation data.
            query (xmsapi.dmi.Query): Object for communicating with GMS
            ugrid_uuid (str): UUID of the UGrid.
            model_str (str): If coming from Add Package > GWF (or GWT), is 'GWF' or 'GWT' respectively

        Returns:
            The sim uuid.
        """
        rename = None
        native_import = False
        if not ugrid_uuid:  # Model native import
            native_import = True
            sim_uuid = str(uuid.uuid4())
            sim = Simulation(name=mfsim.pname, model='MODFLOW 6', sim_uuid=sim_uuid)
            # when importing a simulation we need to set the mainfile for the hidden simulation component
            comp = _create_component(mfsim)
            query.add_simulation(sim, [comp])
        elif model_str:  # Adding a GWF or GWT
            model_uuid = query.current_item_uuid()
            model_node = tree_util.find_tree_node_by_uuid(query.project_tree, model_uuid)
            sim_uuid = model_node.parent.uuid
            rename = (model_uuid, mfsim.models[0].pname)
        else:  # New simulation
            sim_uuid = query.parent_item_uuid()
            rename = (sim_uuid, mfsim.pname)

        self._make_solution_pnames_unique(mfsim, native_import)

        # Get the TDIS, IMS etc packages
        package_list = self._get_packages_under_sim(mfsim)
        for package in package_list:
            _create_add_and_link_component(package=package, parent_uuid=sim_uuid, query=query)

        dogrid_set = set()
        self.model_mname_uuids = {}  # Dict of model mnames and the model uuids
        grid_builder = UGridBuilder(True)
        if native_import:
            self._find_existing_dis_grids(grid_builder, query.project_tree)
        for model in mfsim.models:
            if not model_str:
                model_comp = _create_add_and_link_component(package=model, parent_uuid=sim_uuid, query=query)
                model_uuid = model_comp.uuid
            else:
                model_node.name = model.pname
                model.mname = model.pname
            self.model_mname_uuids[model.mname] = model_uuid

            # Handle UGrid
            if ugrid_uuid:  # New simulation
                query.link_item(model_uuid, ugrid_uuid)
            elif native_import:
                if not model.get_dogrid():
                    dogrid_list = grid_builder.build(mfsim, model)
                    if dogrid_list:
                        dogrid = dogrid_list[0]
                        mfsim.xms_data.set_dogrid(model_uuid, dogrid)
                        ugrid_node = tree_util.find_tree_node_by_uuid(query.project_tree, dogrid.uuid)
                        # Don't add multiple identical UGrids - just link to the existing one
                        if not ugrid_node and dogrid not in dogrid_set:
                            query.add_ugrid(dogrid)
                            dogrid_set.add(dogrid)
                        query.link_item(model_uuid, dogrid.uuid)

            # Handle model packages
            for p in model.packages:
                _create_add_and_link_component(package=p, parent_uuid=model_uuid, query=query)
                p.update_displayed_cell_indices()
                if p.ftype in BaseFileData.displayable_ftypes():
                    if p.ftype != 'HFB6':
                        dmi_util.ensure_disp_opts_file_exists(p.filename, f'{p.ftype}_display_options.json')
                    else:
                        dmi_util.ensure_disp_opts_file_exists(p.filename, 'HFB6_faces_display_options.json')
                        dmi_util.ensure_disp_opts_file_exists(p.filename, 'HFB6_lines_display_options.json')

        if rename:  # We have to do this last until Andrew fixes it
            query.rename_item(rename[0], rename[1])

        # Connect exchanges and IMSes to models via uuids. Do this last
        self._connect_exchanges_to_models(mfsim)
        self._connect_solution_to_models(mfsim)
        return sim_uuid

    def _find_existing_dis_grids(self, grid_builder, tree_node):
        """Look for existing DIS packages and reuse their UGrid if identical to one being imported.

        Args:
            grid_builder (UGridBuilder): The UGrid builder
            tree_node (TreeNode): The project tree to search.
        """
        # Find all the existing DIS packages.
        items, packages = self._find_dis_packages(tree_node, 'DIS6')
        for item, package in zip(items, packages):
            grid_builder.set_dis_grid_as_existing(item, package)
        items, packages = self._find_dis_packages(tree_node, 'DISU6')
        for item, package in zip(items, packages):
            grid_builder.set_dis_grid_as_existing(item, package)
        items, packages = self._find_dis_packages(tree_node, 'DISV6')
        for item, package in zip(items, packages):
            grid_builder.set_dis_grid_as_existing(item, package)

    def _find_dis_packages(self, tree_node, ftype):
        """Parallel list of all the specified DIS-type component tree items and their data.

        Args:
            tree_node (TreeNode): The tree node to search
            ftype (str): ftype/unique_name of the DIS type

        Returns:
            (tuple[list[TreeNode], list[DisData]]): See Description
        """
        items = tree_util.descendants_of_type(
            tree_node, xms_types=['TI_COMPONENT'], unique_name=ftype, model_name='MODFLOW 6'
        )
        packages = [BaseFileData.from_file(item.main_file, ftype=ftype) for item in items]
        return items, packages

    def _get_packages_under_sim(self, mfsim):
        """Returns a list of packages under the simulation (not the GWF).

        Args:
            mfsim (MfsimData): Data class containing all simulation data.

        Returns:
            See description.
        """
        package_list = []
        if mfsim.tdis:
            package_list.append(mfsim.tdis)
        if mfsim.solution_groups:
            for group in mfsim.solution_groups:
                package_list.extend(group.solution_list)
        if mfsim.exchanges:
            for exg in mfsim.exchanges:
                package_list.append(exg)
        return package_list

    def _make_solution_pnames_unique(self, mfsim, native_import):
        """If there are multiple IMS packages, make sure pnames are unique."""
        all_solutions = []
        for solution_group in mfsim.solution_groups:
            for solution in solution_group.solution_list:
                all_solutions.append(solution)

        if len(all_solutions) > 1:  # noqa: W503
            if native_import:
                # use the filename as part of the pname
                for solution in all_solutions:
                    prefix = solution.ftype[:-1]
                    solution.pname = f'{prefix}-{os.path.splitext(os.path.basename(solution.filename))[0]}'
            else:
                # use the fname as part of the pname
                for solution in all_solutions:
                    prefix = solution.ftype[:-1]
                    solution.pname = f'{prefix}-{os.path.splitext(solution.fname)[0]}'

    def _uuid_from_model_name(self, model_name):
        """Get the uuid from model name.

        Args:
            model_name (str): the model name

        Returns:
            (str): the uuid, empty string if model name not found
        """
        return self.model_mname_uuids.get(model_name, '')

    def _connect_exchanges_to_models(self, mfsim):
        """Saves model uuids in the exchange settings.json file so if model is renamed, we still know what it is.

        Args:
            mfsim (MfsimData): Data class containing all simulation data.
        """
        for exchange in mfsim.exchanges:
            exgmnamea_uuid = self._uuid_from_model_name(exchange.exgmnamea)
            exgmnameb_uuid = self._uuid_from_model_name(exchange.exgmnameb)
            if not exgmnamea_uuid or not exgmnameb_uuid:
                raise RuntimeError(f'Could not connect {exchange.exgtype} exchange to model.')
            settings_dict = Settings.read_settings(main_file=exchange.filename)
            settings_dict['exgmnamea'] = exgmnamea_uuid
            settings_dict['exgmnameb'] = exgmnameb_uuid
            Settings.write_settings(main_file=exchange.filename, settings=settings_dict)

    def _connect_solution_to_models(self, mfsim):
        """Saves model uuids in the IMS settings.json file so if model is renamed, we still know what it is.

        Args:
            mfsim (MfsimData): Data class containing all simulation data.
        """
        for solution_group in mfsim.solution_groups:
            for solution in solution_group.solution_list:
                slnmnames_uuids = []
                for slnmname in solution.slnmnames:
                    model_uuid = self._uuid_from_model_name(slnmname)
                    if not model_uuid:
                        raise RuntimeError('Could not connect IMS package to model.')
                    slnmnames_uuids.append(model_uuid)
                Settings.set(main_file=solution.filename, variable='slnmnames', value=slnmnames_uuids)


def _create_component(package):
    """Adds the component to the list.

    Args:
        package: The package.

    Returns:
        component (Component): The new component.
    """
    ftype = data_util.fix_ftype(package.ftype, with_the_a=package.readasarrays)
    if ftype not in _comp_info:  # seems like this is to make sure we add package info to comp_info
        raise RuntimeError(f'{ftype} not in Component Info dict.')

    xml_unique_name = _comp_info[ftype][0]
    return create_data_objects_component(package.filename, package.pname, 'MODFLOW 6', xml_unique_name)


def create_data_objects_component(main_file: str | Path, name: str, model_name: str, unique_name: str) -> Component:
    """Creates and returns a new data_objects.Component.

    Args:
        main_file: The main file.
        name: Name that appears in XMS.
        model_name: Model name from xml.
        unique_name: Unique name from xml.
    """
    comp_uuid = io_util.uuid_from_path(main_file)
    return Component(
        name=name,
        comp_uuid=comp_uuid,
        main_file=str(main_file),
        model_name=model_name,
        unique_name=unique_name,
        locked=False
    )


def create_add_component_action(component, disp_opts_method: str) -> ActionRequest:
    """Create the ActionRequest to add the component.

    Args:
        component: Object derived from ComponentBase.
        disp_opts_method: Method to call to get initial display options.

    Return:
        ActionRequest.
    """
    return ActionRequest(
        modality='NO_DIALOG',
        main_file=component.main_file,
        class_name=component.class_name,
        module_name=component.module_name,
        method_name=disp_opts_method
    )


def add_and_link(query: Query, do_comp: Component, actions: list[ActionRequest], parent_uuid: str) -> None:
    """Adds the mapped component to xms."""
    query.add_component(do_component=do_comp, actions=actions)
    query.link_item(taker_uuid=parent_uuid, taken_uuid=do_comp.uuid)


def _create_add_and_link_component(package, parent_uuid, query) -> Component:
    """Creates the component, adds it to the query, and links it to the parent.

    Args:
        package: The package.
        parent_uuid (str): Uuid of the new component's parent
        query (xmsapi.dmi.Query): Object for communicating with GMS

    Returns:
        (Component): The new component.
    """
    do_comp = _create_component(package)
    add_and_link(query, do_comp, [], parent_uuid)
    return do_comp
