"""Code to run map to modflow with feedback."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import TreeNode
from xms.guipy.dialogs import process_feedback_dlg
from xms.guipy.dialogs.feedback_thread import FeedbackThread

# 4. Local modules
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.gui.map_from_coverage_dialog import MapCoverageDlgOutputs, MapOpt
from xms.mf6.mapping import modflow_mapper
from xms.mf6.mapping.mapper_inputs import CovAndAtts, CovData, MapperInputs


def map_from_coverage(package_dialog, dialog_rv: MapCoverageDlgOutputs) -> bool:
    """Handles map from coverage command.

    Args:
        package_dialog: The package dialog calling this.
        dialog_rv: Outputs from the Map From Coverage dialog.

    Returns:
        True if there were no errors.
    """
    query = package_dialog.dlg_input.query

    # Get inputs
    cov_data: CovData = {}
    for item_uuid in dialog_rv.item_uuids:
        coverage = query.item_with_uuid(item_uuid)
        att_files = query.coverage_attributes(cov_uuid=item_uuid, points=True, arcs=True, polygons=True)
        cov_data[item_uuid] = CovAndAtts(coverage, att_files)

    inputs = MapperInputs(
        coverage_data=cov_data,
        package=package_dialog.dlg_input.data,
        append_or_replace=dialog_rv.append_or_replace,
        override_layers=dialog_rv.override_layers,
        layer_filepath=dialog_rv.layer_filepath
    )
    thread = MapToModflowFeedbackThread(inputs, query)
    return process_feedback_dlg.run_feedback_dialog(thread, package_dialog)


def map_from_shapefile(
    query: Query, component, shapefilepath: Path | str, append_or_replace: MapOpt, win_cont: QWidget
) -> tuple[TreeNode, BaseFileData]:
    """Handles map from shapefile command.

    Args:
        query: Object for communicating with GMS
        component: The component.
        shapefilepath: Shapefile filepath.
        append_or_replace: APPEND to make Append radio button on, REPLACE to make Replace radio button on.
        win_cont: The window container.

    Returns:
        (tuple): tuple containing:
            - Gwf tree node.
            - The package's data object.
    """
    mfsim, model, package = component.read_sim(query)

    # Get inputs
    inputs = MapperInputs(
        package=package,
        append_or_replace=append_or_replace,
        shapefilepath=str(shapefilepath),
    )
    thread = MapToModflowFeedbackThread(inputs, query)
    process_feedback_dlg.run_feedback_dialog(thread, win_cont)
    return model.tree_node, package


class MapToModflowFeedbackThread(FeedbackThread):
    """Thread for mapping to MODFLOW."""
    def __init__(self, inputs: MapperInputs, query: Query):
        """Initializes the class."""
        super().__init__(query, is_export=False)
        self._query = query
        self._inputs = inputs
        object_str = 'coverages' if inputs.coverage_data else 'shapefiles'
        self.display_text |= {
            'title': f'Map from {object_str.title()}',
            'working_prompt': f'Mapping the {object_str} to the MODFLOW package...',
            'error_prompt': f'Error(s) encountered while mapping the {object_str} to the package.',
            'warning_prompt': f'Warning(s) encountered while mapping the {object_str} to the package.',
            'success_prompt': f'Successfully mapped the {object_str} to the package.',
        }

    def _run(self) -> None:
        """Does the work."""
        success, ugrid_data = modflow_mapper.map_to_modflow(self._inputs)
        if success and ugrid_data:
            from xms.mf6.components.dis_component_base import update_dis_packages
            from xms.mf6.components.xms_data import update_ugrid_values
            ugrid_uuid = ugrid_data['ugrid_uuid']
            dis_data = ugrid_data['dis']
            tops, bottoms, idomain = dis_data.get_update_ugrid_data()
            # Update the UGrid top and bottom elevations and the model on/off cells
            update_ugrid_values(ugrid_uuid, tops, bottoms, idomain, self._query)

            # Update all the other DIS packages that use this same UGrid
            update_dis_packages(dis_data, ugrid_uuid, self._query)
