"""BaseFileData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
from typing import Any, TYPE_CHECKING

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
from xms.api.tree import TreeNode

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.grid_info import GridInfo
# I300 TYPE_CHECKING block should have one newline above
if TYPE_CHECKING:  # flake8 is of 2 minds about the blank line above
    from xms.mf6.data.mfsim_data import MfsimData  # noqa: AQU104 extra line between import group comments
from xms.mf6.data.options_block import OptionsBlock  # noqa: I202 Additional newline in a group of imports.
from xms.mf6.file_io import io_factory
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.gui import gui_util


class BaseInfo:
    """Basic file info."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            filename (str | Path): Filepath.
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, exchanges (things below mfsim)
            tree_node: TreeNode in the Project Explorer
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis.
            nper: Number of stress periods. Only used when testing.
        """
        self.ftype: str = kwargs.get('ftype', '')
        self.filename: str = str(kwargs.get('filename', f'model{data_util.extension_from_ftype(self.ftype)}'))
        self.comments: list[str] = []
        self.gms_comments: list[str] = []  # Comments that start with # GMSCOMMENT
        self.options_block: OptionsBlock | None = None
        self.gms_options = {}  # May become dict of additional GMS specific options
        self.fname: str = os.path.basename(kwargs.get('filename', ''))  # fname from model name file. No path.
        self.pname: str = ''  # User friendly name from model name file. Also used as tree name
        self.periods_db: str = ''  # filepath to the periods.db file
        self.mfsim: MfsimData | None = kwargs.get('mfsim')
        self.model = kwargs.get('model')  # GwfData, GwtData, or GweData
        self.tree_node: TreeNode | None = kwargs.get('tree_node', None)  # TreeNode in the Project Explorer
        self.can_import = True
        self.can_export = True
        self.readasarrays: bool = False  # Will be True for EvtArrayData, RchArrayData...

        # Only used when testing. Call grid_info() normally (see below).
        self._grid_info: GridInfo | None = kwargs.get('grid_info')
        self._nper: int = kwargs.get('nper', -1)  # Only for testing. Otherwise comes from mfsim and gwf/gwt


class BaseFileData:
    """Base class for file data."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            filename (str | Path): Filepath.
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            grid_info (GridInfo): Only used when testing individual packages. Otherwise, it comes from model and dis.
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, exchanges (things below mfsim)
            nper (int): Only for testing. Number of stress periods.
            tree_node: TreeNode in the Project Explorer
        """
        self._base = BaseInfo(**kwargs)
        self._base.options_block = self._setup_options()
        self._base.gms_options = self._setup_gms_options()

    @property
    def ftype(self) -> str:
        """Return the ftype - e.g. 'WEL6'."""
        return self._base.ftype

    @ftype.setter
    def ftype(self, value: str) -> None:
        """Set the ftype - e.g. 'WEL6'.

        Args:
            value: The value.
        """
        self._base.ftype = value

    @property
    def filename(self) -> str:
        """Return the filename."""
        return self._base.filename

    @filename.setter
    def filename(self, value: str) -> None:
        """Set the filename.

        Args:
            value: The value.
        """
        self._base.filename = value

    @property
    def comments(self) -> list:
        """Return the comments."""
        return self._base.comments

    @comments.setter
    def comments(self, value: list) -> None:
        """Set the comments.

        Args:
            value: The value.
        """
        self._base.comments = value

    @property
    def gms_comments(self) -> list:
        """Return the GMS comments."""
        return self._base.gms_comments

    @gms_comments.setter
    def gms_comments(self, value: list) -> None:
        """Set the GMS comments.

        Args:
            value: The value.
        """
        self._base.gms_comments = value

    @property
    def options_block(self) -> OptionsBlock | None:
        """Return the options block."""
        return self._base.options_block

    @options_block.setter
    def options_block(self, value: OptionsBlock | None) -> None:
        """Set the options block.

        Args:
            value: The value.
        """
        self._base.options_block = value

    @property
    def gms_options(self) -> dict:
        """Return the GMS options."""
        return self._base.gms_options

    @gms_options.setter
    def gms_options(self, value: dict) -> None:
        """Set the GMS options.

        Args:
            value: The value.
        """
        self._base.gms_options = value

    @property
    def fname(self) -> str:
        """Return the fname - from GWF/GWT model name file - no path."""
        return self._base.fname

    @fname.setter
    def fname(self, value: str) -> None:
        """Set the fname - from GWF/GWT model name file - no path.

        Args:
            value: The value.
        """
        self._base.fname = value

    @property
    def pname(self) -> str:
        """Return the user friendly name from model name file - also used as tree name."""
        return self._base.pname

    @pname.setter
    def pname(self, value: str) -> None:
        """Set the user friendly name from model name file - also used as tree name.

        Args:
            value: The value.
        """
        self._base.pname = value

    @property
    def periods_db(self) -> str:
        """Return the periods database filepath."""
        return self._base.periods_db

    @periods_db.setter
    def periods_db(self, value: str) -> None:
        """Set the periods database filepath.

        Args:
            value: The value.
        """
        self._base.periods_db = value

    @property
    def mfsim(self) -> 'MfsimData':
        """Return the simulation."""
        return self._base.mfsim

    @mfsim.setter
    def mfsim(self, value: 'MfsimData') -> None:
        """Set the simulation.

        Args:
            value: The value.
        """
        self._base.mfsim = value

    @property
    def model(self):
        """Return the model - something derived from ModelDataBase."""
        return self._base.model

    @model.setter
    def model(self, value) -> None:
        """Set the model - something derived from ModelDataBase.

        Args:
            value: The value.
        """
        self._base.model = value

    @property
    def tree_node(self) -> TreeNode:
        """Return the tree node."""
        return self._base.tree_node

    @tree_node.setter
    def tree_node(self, value: TreeNode) -> None:
        """Set the tree node.

        Args:
            value: The value.
        """
        self._base.tree_node = value

    @property
    def can_import(self) -> bool:
        """Return whether 'Import' is enabled."""
        return self._base.can_import

    @can_import.setter
    def can_import(self, value: bool) -> None:
        """Set whether 'Import' is enabled.

        Args:
            value: The value.
        """
        self._base.can_import = value

    @property
    def can_export(self) -> bool:
        """Return whether 'Export' is enabled."""
        return self._base.can_export

    @can_export.setter
    def can_export(self, value: bool) -> None:
        """Set whether 'Export' is enabled.

        Args:
            value: The value.
        """
        self._base.can_export = value

    @property
    def readasarrays(self) -> bool:
        """Return readasarrays."""
        return self._base.readasarrays

    @readasarrays.setter
    def readasarrays(self, value: bool) -> None:
        """Set readasarrays.

        Args:
            value: The value.
        """
        self._base.readasarrays = value

    def grid_info(self) -> GridInfo | None:
        """Returns the GridInfo object.

        Typically, will get it from the model and dis package but when testing can be stored here in base.

        Returns:
            See description.
        """
        if self._base._grid_info:
            return self._base._grid_info
        elif self._base.model:
            return self._base.model.grid_info()
        else:
            return None

    def nper(self) -> int:
        """Returns the number of stress periods.

        Typically, will get it from the mfsim and tdis package but when testing can be stored here in base.

        Returns:
            (int): See description.
        """
        if self._base._nper > -1:
            return self._base._nper
        elif self._base.mfsim and self._base.mfsim.tdis:
            return self._base.mfsim.tdis._nper
        else:
            return 1

    @classmethod
    def read_or_init_gms_options(cls, parent_dir: str | Path) -> dict[Any, Any]:
        """Some packages have additional GMS options, which this reads and returns or initializes to defaults.

        Args:
            parent_dir (str|Path): Path to the directory where the gms options file is located.

        Returns:
            See description.
        """
        return {}

    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        return OptionsBlock(definitions=[])

    # @overrides
    def _setup_gms_options(self) -> dict[Any, Any]:
        """Some packages have additional GMS options, which this reads and returns or initializes to defaults.

        Returns:
            (OptionsBlock): See description.
        """
        if self._base.filename:
            return self.read_or_init_gms_options(Path(self._base.filename).parent)
        return {}

    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        return [], {}, {}

    def get_column_delegate_info(self, block):
        """Returns a list of tuples of [0] column index and [1] list of strings."""
        return None

    def update_displayed_cell_indices(self) -> None:
        """Updates the cell indices file used to display symbols."""
        pass

    @staticmethod
    def displayable_ftypes():
        """Returns the set of ftypes whose packages have data that can be displayed.

        E.g. WEL6 can be displayed, but IC6 cannot.

        Returns:
            See description.
        """
        return {
            'CHD6',
            'CNC6',
            'CTP6',
            'DRN6',
            'EVT6',
            'ESL6',
            'GHB6',
            # 'GNC6', TODO:
            'HFB6',
            'LAK6',
            'MAW6',
            # 'OBS6', TODO:
            'RCH6',
            'RIV6',
            'SFR6',
            'SRC6',
            'UZF6',
            'WEL6',
        }

    def read_csv_file_into_dataframe(self, block, filename, column_names, column_types) -> pd.DataFrame:
        """Base version that reads a csv file and returns a dataframe.

        Args:
            block (str): Name of the block.
            filename (str): Filepath.
            column_names (list): Column names.
            column_types (dict of str -> type): Column names -> column types.

        Returns:
            (pandas.DataFrame)
        """
        del block  # Unused parameter
        return gui_util.read_csv_file_into_dataframe(filename, column_names, column_types)

    def dataframe_to_temp_file(self, block, dataframe):
        """Writes the dataframe to a csv file.

        Args:
            block (str): Name of the block.
            dataframe (pandas.DataFrame): The dataframe

        Returns:
            (str): Filepath of file created.
        """
        del block  # Unused parameter
        return gui_util.dataframe_to_temp_file(dataframe)

    def block_with_aux(self):
        """Returns the name of the block that can have aux variables.

        Returns:
            (str): The name of the block that can have aux variables.
        """
        return ''

    def write(self, writer_options: WriterOptions = None) -> None:
        """Writes the package.

        Args:
            writer_options: WriterOptions object.
        """
        writer = io_factory.writer_from_ftype(self._base.ftype, writer_options)
        writer.write(self)

    @staticmethod
    def from_file(filepath: Path | str, ftype: str, **kwargs):
        """Returns the package by reading the file.

        Args:
            filepath:
            ftype: The ftype.
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
            resolve_paths (bool): If True, relative file paths are resolved to absolute paths.

        Returns:
            The package data class.
        """
        reader = io_factory.reader_from_ftype(ftype)
        return reader.read(filepath, **kwargs)
