"""MawData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util, units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxButton, CheckboxField


class MawData(ListPackageData):
    """Data class to hold the info from a MAW package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'MAW6'
        self.block_with_cellids = 'CONNECTIONDATA'
        self.list_blocks = {'PACKAGEDATA': '', 'CONNECTIONDATA': ''}

    # @overrides
    def get_column_delegate_info(self, block):
        """Returns a list of tuples of [0] column index and [1] list of strings.

        Returns:
            (tuple): tuple containing:
                - index (int): Column index.
                - strings (list of str): List of strings allowed in the column.
        """
        delegate_info = None
        if block == 'PACKAGEDATA':
            delegate_info = [(4, ['SPECIFIED', 'THIEM', 'SKIN', 'CUMULATIVE', 'MEAN'])]

        return delegate_info

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the DIS package in use and the AUX variables.
        The package specific and AUX columns are type object because they
        might contain time series strings.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        if block.upper() == 'PACKAGEDATA':
            columns = {
                'WELLNO': (np.int32, 1),
                'RADIUS': (np.float64, 0.0),
                'BOTTOM': (np.float64, 0.0),
                'STRT': (np.float64, 0.0),
                'CONDEQN': (object, 'SPECIFIED'),
                'NGWFNODES': (np.int32, 1),
            }
            self.add_aux_columns_to_dict(columns, use_aux=use_aux)
            data_util.add_boundname_columns_to_dict(self.options_block, columns)

        elif block.upper() == 'CONNECTIONDATA':
            id_columns = data_util.get_id_column_dict(self.grid_info())
            columns = {
                **{
                    'WELLNO': (np.int32, 1),
                    'ICON': (np.int32, 1),
                },
                **id_columns,
                **{
                    'SCRN_TOP': (np.float64, 0.0),
                    'SCRN_BOT': (np.float64, 0.0),
                    'HK_SKIN': (np.float64, 0.0),
                    'RADIUS_SKIN': (np.float64, 0.0),
                }
            }

        else:  # This would be the stress periods
            columns = {
                'WELLNO': (np.int32, 1),
                'MAWSETTING': (object, ''),
                'VALUE1': (object, ''),
                'VALUE2': (object, ''),
                'VALUE3': (object, ''),
            }

        names, types, defaults = gui_util.column_info_tuple_from_dict(columns)
        return names, types, defaults

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        if block.upper() == 'PACKAGEDATA':
            length_units = units_util.string_from_units(self, units_util.UNITS_LENGTH)
            return {
                names.index('WELLNO'): 'Well number',
                names.index('RADIUS'): f'Radius for the multi-aquifer well {length_units}',
                names.index('BOTTOM'): f'Bottom elevation of the multi-aquifer well {length_units}.',
                names.index('STRT'): f'Starting head for the multi-aquifer well {length_units}.',
                names.index('CONDEQN'):
                    'Defines the conductance equation that is used to calculate the saturated'
                    ' conductance for the multi-aquifer well.',
                names.index('NGWFNODES'): 'Number of GWF nodes connected to this multiaquifer well.',
            }
        elif block.upper() == 'CONNECTIONDATA':
            length_units = units_util.string_from_units(self, units_util.UNITS_LENGTH)
            return {
                names.index('WELLNO'): 'Well number',
                names.index('ICON'): 'GWF connection number for this multi-aquifer well connection entry',
                names.index('SCRN_TOP'):
                    f'Top elevation of the screen for the multi-aquifer well connection'
                    f' {length_units}',
                names.index('SCRN_BOT'):
                    f'Bottom elevation of the screen for the multi-aquifer well connection'
                    f' {length_units}',
                names.index('HK_SKIN'):
                    'The skin (filter pack) hydraulic conductivity (if CONDEQN for the'
                    ' multiaquifer well is SKIN, CUMULATIVE, or MEAN) or conductance (if'
                    ' CONDEQN for the multiaquifer well is SPECIFIED) for each GWF node'
                    ' connected to the multi-aquifer well (NGWFNODES)',
                names.index('RADIUS_SKIN'):
                    'Skin radius (filter pack radius) for the multi-aquifer well'
                    f' {length_units}',
            }
        else:  # stress periods
            names, _types, _defaults = self.get_column_info('')
            return {
                names.index('WELLNO'):
                    'Well number',
                names.index('MAWSETTING'):
                    (
                        'Keyword to start MAW setting line: STATUS, FLOWING_WELL, RATE, WELL_HEAD,'
                        ' HEAD_LIMIT, SHUT_OFF, RATE_SCALING, and AUXILIARY'
                    ),  # noqa: E123 closing bracket doesn't match opening bracket indentation
            }

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        raise NotImplementedError()
        # return [], {}, {}

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Multi-Aquifer Well (MAW) Package'

    @override
    def get_time_series_columns(self) -> list[int]:
        """Returns a list of the column indices that can contain time series.

        Returns:
            List of indices of columns that can contain time series.
        """
        return []

    def block_with_aux(self):
        """Returns the name of the block that can have aux variables.

        Returns:
            (str): The name of the block that can have aux variables.
        """
        return 'PACKAGEDATA'

    def block_with_boundnames(self):
        """Returns the name of the block that can have aux variables.

        Returns:
            (str): The name of the block that can have aux variables.
        """
        return 'PACKAGEDATA'

    # @overrides
    def update_displayed_cell_indices(self) -> None:
        """Updates the cell indices file used to display symbols."""
        self._update_displayed_cell_indices_in_block('CONNECTIONDATA')

    def stress_id_columns(self):
        """Returns the column name where the id exists that can be used to help identify this stress across periods.

        Typically is 'CELLIDX' which is added by GMS but is 'RNO' for SFR.

        Returns:
            See description.
        """
        return ['WELLNO']

    def plottable_columns(self):
        """Returns a set of columns (0-based) that can be plotted with the XySeriesEditor.

        Returns:
            See description.
        """
        column_count = len(self.get_column_info('')[0])
        return set(range(2, column_count))  # Start after the MAWSETTING column

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        return OptionsBlock(
            [
                CheckboxButton(
                    'AUXILIARY',
                    button_text='Auxiliary Variables...',
                    check_box_method='on_chk_auxiliary',
                    button_method='on_btn_auxiliary'
                ),
                Checkbox('BOUNDNAMES', brief='Allow boundary names', check_box_method='on_chk_boundnames'),
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
                Checkbox('PRINT_HEAD', brief='Print head to listing file'),
                Checkbox('PRINT_FLOWS', brief='Print flows to listing file'),
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                CheckboxField('HEAD FILEOUT', brief='Save head to file', type_='str'),
                CheckboxField('BUDGET FILEOUT', brief='Save budget to file', type_='str'),
                CheckboxField('BUDGETCSV FILEOUT', brief='Save budget to CSV file', type_='str'),
                Checkbox('NO_WELL_STORAGE', brief='Do not include well storage in multiaquifer well equation'),
                Checkbox('FLOW_CORRECTION', brief='Activate flow corrections'),
                Checkbox('FLOWING_WELLS', brief='Activate the flowing wells option'),
                CheckboxField(
                    'SHUTDOWN_THETA',
                    brief='Weight for wells that limit the water level in a discharging well',
                    type_='float',
                    value=0.7
                ),
                CheckboxField(
                    'SHUTDOWN_KAPPA',
                    brief='Weight for wells that limit the water level in a discharging well',
                    type_='float',
                    value=0.0001
                ),
                CheckboxField('MAW_FLOW_REDUCE_CSV FILEOUT', brief='Record when flow rate is reduced', type_='str'),
                CheckboxButton(
                    'TS6 FILEIN', brief='Time-series files', button_text='Files...', button_method='on_btn_ts6_filein'
                ),
                CheckboxButton(
                    'OBS6 FILEIN',
                    brief='Observation files',
                    button_text='Files...',
                    button_method='on_btn_obs6_filein'
                ),
                Checkbox('MOVER', brief='Can be used with the Water Mover (MVR) Package'),
            ]
        )

    def map_info(self, feature_type):
        """Returns info needed for Map from Coverage.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
        """
        if feature_type != 'points':
            return {}
        return {'Q': None}

    def map_import_info(self, feature_type):
        """Returns info needed for mapping shapefile or transient data file to package data.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'

        Returns:
            (dict): See description
        """
        if feature_type != 'points':
            return {}
        info = {'Q': None}
        self._add_aux_and_boundname_info(info)
        return info
