"""MstData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxComboBox, CheckboxField


class MstData(GriddataBase):
    """Data class to hold the info from a MST package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'MST6'
        self.add_block('GRIDDATA', ['POROSITY', 'DECAY', 'DECAY_SORBED', 'BULK_DENSITY', 'DISTCOEF', 'SP2'])

    def get_units(self, array_name: str) -> str:
        """Returns the units string for the array.

        Args:
            array_name (str): The name of a array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        match array_name:
            case 'POROSITY':
                units_spec = units_util.UNITS_UNITLESS
            case 'DECAY':
                d = self.options_block.get('FIRST_ORDER_DECAY')
                if d:
                    units_spec = '[1/T]'
                else:
                    units_spec = '[M/L^3/T]'
            case 'DECAY_SORBED':
                d = self.options_block.get('FIRST_ORDER_DECAY')
                if d:
                    units_spec = '[1/T]'
                else:
                    units_spec = '[M/M/T]'
            case 'BULK_DENSITY':
                units_spec = units_util.UNITS_BULK_D
            case 'DISTCOEFF':
                units_spec = units_util.UNITS_DIST_C
            case 'SP2':
                units_spec = units_util.UNITS_UNITLESS
            case _:
                units_spec = ''  # This is an error
        return units_spec

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return False

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() == 'POROSITY'

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Mobile Storage and Transfer (MST) Package'

    # def map_info(self, feature_type):
    #     """Returns info needed for Map from Coverage.
    #
    #     Args:
    #         feature_type (str): 'points', 'arcs', or 'polygons'
    #
    #     Returns:
    #         (dict): Dict describing how to get the MODFLOW variable from the shapefile or att table fields.
    #     """
    #     if feature_type != 'polygons':
    #         return {}
    #     return {'HK': Mapping(mf_name='K'), 'VK': Mapping(mf_name='K33'), 'HANI': Mapping(mf_name='K22'),
    #             'VANI': Mapping(mf_name='K33')}

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        return OptionsBlock(
            [
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                Checkbox('FIRST_ORDER_DECAY', brief='First order decay will occur'),
                Checkbox('ZERO_ORDER_DECAY', brief='Zero order decay will occur'),
                CheckboxComboBox(
                    'SORPTION',
                    brief='Sorption will be activated',
                    items=['LINEAR', 'FREUNDLICH', 'LANGMUIR'],
                    value='LINEAR'
                ),
                CheckboxField('SORBATE FILEOUT', brief='Save sorbate concentrations', type_='str'),
            ]
        )

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'POROSITY':
            txt = (
                'Mobile domain porosity, defined as the mobile domain pore volume per mobile domain volume. Additional'
                ' information on porosity within the context of mobile and immobile domain transport simulations is'
                ' included in the MODFLOW 6 Supplemental Technical Information document.'
            )
        elif tab_text == 'DECAY':
            txt = (
                'Rate coefficient for first or zero-order decay for the aqueous phase of the mobile domain. A negative'
                ' value indicates solute production. The dimensions of decay for first-order decay is one over time.'
                ' The dimensions of decay for zero-order decay is mass per length cubed per time. decay will have no'
                ' effect on simulation results unless either first- or zero-order decay is specified in the options'
                ' block.'
            )
        elif tab_text == 'DECAY_SORBED':
            txt = (
                'Rate coefficient for first or zero-order decay for the sorbed phase of the mobile domain. A negative'
                ' value indicates solute production. The dimensions of decay_sorbed for first-order decay is one over'
                ' time. The dimensions of decay_sorbed for zero-order decay is mass of solute per mass of aquifer per'
                ' time. If decay_sorbed is not specified and both decay and sorption are active, then the program will'
                ' terminate with an error. decay_sorbed will have no effect on simulation results unless the SORPTION'
                ' keyword and either first- or zero-order decay are specified in the options block.'
            )
        elif tab_text == 'BULK_DENSITY':
            txt = (
                'Bulk density of the aquifer in mass per length cubed. bulk_density is not required unless the SORPTION'
                ' keyword is specified. Bulk density is defined as the mobile domain solid mass per mobile domain'
                ' volume. Additional information on bulk density is included in the MODFLOW 6 Supplemental Technical'
                ' Information document.'
            )
        elif tab_text == 'DISTCOEF':
            txt = (
                'Distribution coefficient for the equilibrium-controlled linear sorption isotherm in dimensions of'
                ' length cubed per mass. If the Freunchlich isotherm is specified, then discoef is the Freundlich'
                ' constant. If the Langmuir isotherm is specified, then distcoef is the Langmuir constant. distcoef is'
                ' not required unless the SORPTION keyword is specified.'
            )
        elif tab_text == 'SP2':
            txt = (
                'Exponent for the Freundlich isotherm and the sorption capacity for the Langmuir isotherm. sp2 is not'
                ' required unless the SORPTION keyword is specified in the options block. If the SORPTION keyword is'
                ' not specified in the options block, sp2 will have no effect on simulation results.'
            )
        return txt
