"""MfsimWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.file_io import io_factory, io_util
from xms.mf6.file_io.package_writer_base import PackageWriterBase
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.misc import util


class MfsimWriter(PackageWriterBase):
    """Writes a MODFLOW 6 simulation."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()

        self._fp = None  # file object

    def _write_timing(self):
        """Writes the time block."""
        self._fp.write('\n')
        self._fp.write('BEGIN TIMING\n')
        auto_name = ''
        if not self._writer_options.dmi_sim_dir and self._writer_options.auto_file_naming:
            auto_name = data_util.auto_file_name(self._data.tree_node.name, 'TDIS6')
        self._data.tdis.filename, relative_path = self._get_new_filename(self._data.tdis, auto_name)
        self._fp.write(f'{io_util.mftab}TDIS6 {io_util.quote(relative_path)}\n')

        if not self._writer_options.just_name_file:
            writer = io_factory.writer_from_ftype('TDIS6', self._writer_options)
            if not self._data.gui_edit_active:
                writer.write(self._data.tdis)
            else:
                writer.write_settings(self._data.tdis)
        self._fp.write('END TIMING\n')

    def _write_models(self):
        """Writes the time block."""
        self._fp.write('\n')
        self._fp.write('BEGIN MODELS\n')
        auto_file_names = []
        if not self._writer_options.dmi_sim_dir and self._writer_options.auto_file_naming:
            auto_file_names = data_util.compute_auto_names(self._data.models, '')
        for i, model in enumerate(self._data.models):
            auto_file_name = auto_file_names[i] if auto_file_names else ''
            model.filename, relative_path = self._get_new_filename(model, auto_file_name)
            mname = '' if not model.mname else f' {model.mname.replace(" ", "_")}'
            self._fp.write(f'{io_util.mftab}{model.mtype} {io_util.quote(relative_path)}{mname}\n')

            if not self._writer_options.just_name_file:
                writer = io_factory.writer_from_ftype('GWF6', self._writer_options)
                if not self._data.gui_edit_active:
                    writer.write(model)
                else:
                    writer.write_settings(model)
        self._fp.write('END MODELS\n')

    def _write_exchanges(self):
        """Writes the exchanges block."""
        self._fp.write('\n')
        self._fp.write('BEGIN EXCHANGES\n')
        if self._data.exchanges:
            auto_file_names = []
            if not self._writer_options.dmi_sim_dir and self._writer_options.auto_file_naming:
                auto_file_names = data_util.compute_auto_names(self._data.exchanges, self._data.tree_node.name)
            for i, exchange in enumerate(self._data.exchanges):
                # old_filename = exchange.filename
                auto_file_name = auto_file_names[i] if auto_file_names else ''
                exchange.filename, relative_path = self._get_new_filename(exchange, auto_file_name)

                if not self._writer_options.just_name_file:
                    writer = io_factory.writer_from_ftype(exchange.exgtype, self._writer_options)
                    if not self._data.gui_edit_active:
                        writer.write(exchange)
                    else:
                        writer.write_settings(exchange)

                # Prepare exgmnames (convert uuids to names if necessary)
                if not exchange.exgmnamea or not exchange.exgmnameb:
                    if not exchange.exgmnamea:
                        self._log.error(f'{exchange.exgtype} EXGMNAMEA not set in Simulation Options')
                    if not exchange.exgmnameb:
                        self._log.error(f'{exchange.exgtype} EXGMNAMEB not set in Simulation Options')
                else:
                    exgmnames = [exchange.exgmnamea, exchange.exgmnameb]
                    if util.is_valid_uuid(exgmnames[0]):  # Hope people don't name their models with a uuid
                        # Convert uuid to a name
                        for i, exgmname in enumerate(exgmnames):
                            tree_node = self._exch_tree_node_from_uuid(exgmname)
                            # tree_node = tree_util.find_tree_node_by_uuid(self._data.tree_node, exgmname)
                            if not tree_node:
                                raise RuntimeError('Error writing exgmnamea, exgmnameb in mfsim.nam')
                            exgmnames[i] = tree_node.name
                    self._fp.write(f'{io_util.mftab}{exchange.exgtype} {relative_path} {exgmnames[0]} {exgmnames[1]}\n')
        self._fp.write('END EXCHANGES\n')

    def _write_solutiongroup(self):
        """Writes the SOLUTIONGROUP block."""
        auto_file_names = []
        if not self._writer_options.dmi_sim_dir and self._writer_options.auto_file_naming:
            all_solutions = []
            for group in self._data.solution_groups:
                for solution in group.solution_list:
                    all_solutions.append(solution)
            auto_file_names = data_util.compute_auto_names(all_solutions, self._data.tree_node.name)

        solution_count = 0
        for i, group in enumerate(self._data.solution_groups):
            self._fp.write('\n')
            self._fp.write(f'BEGIN SOLUTIONGROUP {i + 1}\n')
            if group.use_mxiter:
                self._fp.write(f'{io_util.mftab}MXITER {group.mxiter}\n')
            for solution in group.solution_list:
                auto_file_name = auto_file_names[solution_count] if auto_file_names else ''
                solution_count += 1
                solution.filename, relative_path = self._get_new_filename(solution, auto_file_name)
                self._fp.write(f'{io_util.mftab}{solution.ftype} {io_util.quote(relative_path)}')
                # slnmnames
                for name in solution.slnmnames:
                    if util.is_valid_uuid(name):  # Hope people don't name their models with a uuid
                        # Convert uuid to a name
                        tree_node = self._solution_tree_node_from_uuid(name)
                        if not tree_node:
                            raise RuntimeError('Error writing slnmnames in mfsim.nam')
                        name = tree_node.name.replace(' ', '_')
                    self._fp.write(f' {name}')
                self._fp.write('\n')

                if not self._writer_options.just_name_file:
                    writer = io_factory.writer_from_ftype(solution.ftype, self._writer_options)
                    if not self._data.gui_edit_active:
                        writer.write(solution)
                    else:
                        writer.write_settings(solution)
            self._fp.write('END SOLUTIONGROUP\n')

    def _exch_tree_node_from_uuid(self, uuid):
        """Gets the tree node from uuid. Useful to mock for testing.

        Args:
            uuid (str): the tree item uuid

        Returns:
            (TreeNode)
        """
        return tree_util.find_tree_node_by_uuid(self._data.tree_node, uuid)

    def _solution_tree_node_from_uuid(self, uuid):
        """Gets the tree node from uuid. Useful to mock for testing.

        Args:
            uuid (str): the tree item uuid

        Returns:
            (TreeNode)
        """
        return tree_util.find_tree_node_by_uuid(self._data.tree_node, uuid)

    def write_gms_options(self, mfsim_dir):
        """Writes the persistent data file."""
        if not self._writer_options.dmi_sim_dir:  # Don't write when exporting
            return

        io_util.write_gms_options(mfsim_dir, self._data.gms_options)

    def write_settings(self, data):
        """If writing to components area, save fname to a settings file.

        Args:
            data: The package data.
        """
        pass  # mfsim.nam has no settings

    def _write_package(self, data):
        """Writes the mf6 simulation.

        Args:
            data (MfsimData): the modflow simulation
        """
        WriterOptions._dmi_file_list = []

        self._data = data

        self._data.filename, _ = self._get_new_filename(self._data)

        # Setup input and output dirs. At simulation level (tdis, exg...) there's no subfolders
        if self._writer_options.use_open_close:
            self._writer_options.open_close_dir = os.path.dirname(self._data.filename)
        if self._writer_options.use_output_dir:
            self._writer_options.output_dir = os.path.dirname(self._data.filename)

        with open(self._data.filename, 'w') as self._fp:
            self._writer_options.mfsim_dir = os.path.dirname(self._data.filename)
            self.write_gms_options(self._writer_options.mfsim_dir)
            self._write_options(self._fp)
            if self._data.tdis:
                self._write_timing()
            self._write_models()
            self._write_exchanges()
            if self._data.solution_groups:
                self._write_solutiongroup()
