"""AdvancedPackageDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.gui import options_util
from xms.mf6.gui.data_tabs_widget import DataTabsWidget
from xms.mf6.gui.package_dialog_base import PackageDialogBase


class AdvancedPackageDialog(PackageDialogBase):
    """A dialog used for the LAK, MAW, SFR, and UZF packages."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)
        self.data_tabs_widget = None
        self.setup_ui()

    @override
    def clear_sections(self) -> None:
        """Delete all section widgets."""
        self.close_db()  # Do this whether dialog is locked or not
        self.data_tabs_widget = None
        super().clear_sections()

    @override
    def setup_ui(self) -> None:
        """Set up everything dealing with sections."""
        self.data_tabs_widget = DataTabsWidget(self.dlg_input, self)
        super().setup_ui()

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'DATA']
        self.default_sections = ['DATA']

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'DATA':
            self.setup_data_section()
        else:
            super().setup_section(section_name)

    def do_enabling(self):
        """Enables and disables the widgets appropriately."""
        super().do_enabling()
        self.data_tabs_widget.do_enabling()

    def setup_common_options(self, head_word, widgets, layout):
        """Sets up the BOUNDNAMES, PRINT_INPUT, PRINT_STAGE, PRINT_FLOWS, SAVE_FLOWS options.

        Args:
            head_word (str): PRINT_HEAD checkbox. 'HEAD' for maw6, 'STAGE' for sfr6 and lak6, '' for no checkbox (UZF).
            widgets (dict): Dict of options widgets.
            layout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        options_util.checkbox_or_label('BOUNDNAMES', '', widgets, layout)
        options_util.checkbox_or_label('PRINT_INPUT', '', widgets, layout)
        if head_word:
            options_util.checkbox_or_label(f'PRINT_{head_word}', '', widgets, layout)
        options_util.checkbox_or_label('PRINT_FLOWS', '', widgets, layout)
        options_util.checkbox_or_label('SAVE_FLOWS', '', widgets, layout)

    def setup_mover_option(self, widgets, layout):
        """Sets up the Mover option.

        Args:
            widgets (dict): Dict of options widgets.
            layout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        options_util.checkbox_or_label('MOVER', '', widgets, layout)

    def setup_data_section(self):
        """Sets up the DATA section."""
        name = 'DATA'
        self.add_group_box_to_scroll_area(name)

        self.uix[name]['data_tabs_widget'] = self.data_tabs_widget
        self.uix[name]['layout'].addWidget(self.data_tabs_widget)
        self.data_tabs_widget.setup()

    def on_chk_boundnames(self, checked):
        """Warns user that this will add or remove the boundname column and lets them cancel.

        Args:
            checked (bool): True if checkbox is checked.
        """
        block = self.dlg_input.data.block_with_boundnames()
        table_widget = self.get_table_widget(block)
        if table_widget:
            self.dlg_input.data.options_block.set(key='BOUNDNAMES', on=checked, value=None)
            use_aux = 'chk_auxiliary' in self.options_gui.uix and self.options_gui.uix['chk_auxiliary'].isChecked()
            table_widget.change_columns(block, use_aux)

    # @overrides
    def get_table_widget(self, block):
        """Returns the table view from the given block.

        Args:
            block: The block.
        """
        return self.data_tabs_widget.get_table_widget(block)

    def close_db(self):
        """Closes the database connections."""
        self.data_tabs_widget.close_db()

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.data_tabs_widget.accept()
            self.save_list_blocks_to_temp()

    def accept(self):
        """Called when user hits the OK button. Saves the widget state and closes the dialog."""
        super().accept()
        self.close_db()  # Do this whether dialog is locked or not

    def reject(self):
        """Called when the user clicks Cancel."""
        self.data_tabs_widget.reject()
        self.close_db()  # Do this whether dialog is locked or not
        super().reject()
