"""DisDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QLabel
from typing_extensions import override

# 3. Aquaveo modules
from xms.guipy.dialogs import dialog_util

# 4. Local modules
from xms.mf6.gui import units_util
from xms.mf6.gui.dis_dialog_base import DisDialogBase
from xms.mf6.gui.options_gui import OptionsGui


class DisDialog(DisDialogBase):
    """A dialog used for the DIS package."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, ['DELR', 'DELC'], parent)

    # @overrides
    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'DIMENSIONS', 'GRIDDATA']
        self.default_sections = ['GRIDDATA']

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self._sanitize_length_units()
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

        txt = 'XORIGIN, YORIGIN, and ANGROT are defined by the grid and are not editable'
        label = QLabel(txt)
        label.setStyleSheet('font-weight: bold; color: black')
        vlayout.addWidget(label)

    def _sanitize_length_units(self):
        """Makes the length units match what we expect."""
        units = self.dlg_input.data.options_block.get('LENGTH_UNITS')
        if units:
            # Make upper case and match old abbreviations to current options
            units = units.upper()
            self.dlg_input.data.options_block.set('LENGTH_UNITS', on=True, value=units)

            # May need to add more stuff below as we come across it
            if units in {'M'}:
                self.dlg_input.data.options_block.set('LENGTH_UNITS', on=True, value='METERS')
            elif units in {'CM'}:
                self.dlg_input.data.options_block.set('LENGTH_UNITS', on=True, value='CENTIMETERS')
            elif units in {'UNDEFINED'}:
                self.dlg_input.data.options_block.set('LENGTH_UNITS', on=True, value='UNKNOWN')

    def setup_dimensions(self):
        """Sets up the dimensions section."""
        # Dimensions layout
        name = 'DIMENSIONS'
        self.add_group_box_to_scroll_area(name)

        # NLAY
        self.uix[name]['txt_nlay'] = QLabel('Number of layers (NLAY):')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_nlay'])

        # NROW
        self.uix[name]['txt_nrow'] = QLabel('Number of rows (NROW):')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_nrow'])

        # NCOL
        self.uix[name]['txt_ncol'] = QLabel('Number of columns (NCOL):')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_ncol'])

        self.uix[name]['txt_nlay'].setText(f'Number of layers (NLAY): {self.dlg_input.data.grid_info().nlay}')
        self.uix[name]['txt_nrow'].setText(f'Number of rows (NROW): {self.dlg_input.data.grid_info().nrow}')
        self.uix[name]['txt_ncol'].setText(f'Number of columns (NCOL): {self.dlg_input.data.grid_info().ncol}')

    # @overrides
    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'DIMENSIONS':
            self.setup_dimensions()
        else:
            super().setup_section(section_name)

    def on_chk_length_units(self, checked: int) -> None:
        """Called when the length unit checkbox changes.

        Args:
            checked (int): Checked state.
        """
        length_units = self.options_gui.uix['cbx_length_units'].currentText()
        self.dlg_input.data.options_block.set('LENGTH_UNITS', on=bool(checked), value=length_units)
        self._update_units()

    def on_cbx_length_units(self, text: str) -> None:
        """Called when the length unit changes.

        Args:
            text (str): New combo box text.
        """
        self.dlg_input.data.options_block.set('LENGTH_UNITS', on=True, value=text)
        self._update_units()

    def _update_units(self):
        tab_name = self.griddata_widget.current_tab_name()
        units_spec = self.dlg_input.data.get_units(tab_name)
        units_str = units_util.string_from_units(self.dlg_input.data, units_spec=units_spec)
        self.griddata_widget.on_cbx_length_units(units_str)

    def _ugrid_data_changed(self) -> bool:
        """Return True if any of these arrays have been changed: ['TOP', 'BOT', 'BOTM', 'IDOMAIN']."""
        changed = any(item in self.griddata_widget.changed_arrays for item in ['TOP', 'BOT', 'BOTM', 'IDOMAIN'])
        modified = any(item in self.griddata_widget.modified_arrays for item in ['TOP', 'BOT', 'BOTM', 'IDOMAIN'])
        return changed or modified

    def _update_ugrid_values(self):
        """Update the UGrid top and bottom elevations from the MODFLOW array data."""
        if not self.dlg_input.ugrid_uuid or not self._ugrid_data_changed():
            return

        with dialog_util.wait_cursor_context():
            tops, bottoms, idomain = self.dlg_input.data.get_update_ugrid_data()
            # import just these functions here or there is a circular dependency
            from xms.mf6.components.dis_component_base import update_dis_packages
            from xms.mf6.components.xms_data import update_ugrid_values
            ugrid_uuid = self.dlg_input.ugrid_uuid
            query = self.dlg_input.query
            # Update the UGrid top and bottom elevations and the model on/off cells
            update_ugrid_values(ugrid_uuid, tops, bottoms, idomain, query)

            # Update all the other DIS packages that use this same UGrid
            update_dis_packages(self.dlg_input.data, ugrid_uuid, query)

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.griddata_widget.save_array_to_temp()
            if self._ugrid_data_changed():
                self._update_ugrid_values()
