"""GncDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import (
    QLabel,
    QPlainTextEdit,
)

# 3. Aquaveo modules
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.mf6.data.grid_info import DisEnum
from xms.mf6.gui.event_filter import EventFilter
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog_base import PackageDialogBase


class GncDialog(PackageDialogBase):
    """A dialog used for the GNC package."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DlgInput): DlgInput class.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)
        self.event_filter = EventFilter(dialog=self)
        self.setup_ui()

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'GNCDATA']
        self.default_sections = ['GNCDATA']

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def setup_gncdata_section(self):
        """Sets up the gnc data section."""
        name = 'GNCDATA'
        self.add_group_box_to_scroll_area(name)

        txt = 'Editing of GNCDATA is not yet implemented.'
        self.uix[name]['txt_read_only'] = QLabel(txt)
        self.uix[name]['txt_read_only'].setStyleSheet('font-weight: bold; color: red')
        self.uix[name]['layout'].addWidget(self.uix[name]['txt_read_only'])

        # Add widgets
        self.uix[name]['edt_gncdata'] = QPlainTextEdit()
        self.uix[name]['edt_gncdata'].setLineWrapMode(QPlainTextEdit.NoWrap)
        self.uix[name]['edt_gncdata'].setReadOnly(True)
        self.uix[name]['layout'].addWidget(self.uix[name]['edt_gncdata'])

        h2 = ''
        h3 = ''
        grid_info = self.dlg_input.data.grid_info()
        for i in range(0, self.dlg_input.data.numalphaj):
            if grid_info.dis_enum == DisEnum.DIS:
                h2 += f'cellidsj-K{i}\tcellidsj-I{i}\tcellidsj-J{i}\t'
            elif grid_info.dis_enum == DisEnum.DISV:
                h2 += f'cellidsj-K{i}\tcellidsj-CELL2D{i}\t'
            else:
                h2 += f'cellidsj-CELLID{i}\tcellidsj-CELLID{i}\t'
            h3 += f'alphasj-{i}\t'

        if grid_info.dis_enum == DisEnum.DIS:
            h1 = 'cellidn-K\tcellidn-I\tcellidn-J\tcellidm-K\tcellidm-I\tcellidm-J\t'
        elif grid_info.dis_enum == DisEnum.DISV:
            h1 = 'cellidn-K\tcellidn-CELL2D\tcellidm-K\tcellidm-CELL2D\t'
        else:
            h1 = 'cellidn-CELLID\tcellidm-CELLID\t'
        header = f'{h1}{h2}{h3}'

        if 'GNCDATA' in self.dlg_input.data.list_blocks:
            widget_builder.fill_edt_with_file(
                self.dlg_input.data.list_blocks['GNCDATA'], self.uix[name]['edt_gncdata'], header
            )

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'GNCDATA':
            self.setup_gncdata_section()
        else:
            super().setup_section(section_name)
