"""SwiDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import QPushButton

# 3. Aquaveo modules
from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.mf6.data.grid_info import DisEnum
from xms.mf6.data.gwf import swi_data
from xms.mf6.gui import options_util
from xms.mf6.gui.dialog_input import DialogInput
from xms.mf6.gui.griddata_dialog import GriddataDialog
from xms.mf6.gui.gwf import swi_set_up_dialog
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.tools import swi_tools
from xms.mf6.tools.swi_tools import SwiSetUpInputs, SwiSetUpOutputs


class SwiDialog(GriddataDialog):
    """A dialog for the SWI package."""
    def __init__(self, dlg_input: DialogInput, parent=None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, read_only_tabs=None, parent=parent)
        self._outputs: SwiSetUpOutputs | None = None
        self._inputs: SwiSetUpInputs | None = None
        self._init_set_up_inputs()

    # @overrides
    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['GMS Options', 'COMMENTS', 'OPTIONS', 'GRIDDATA']
        self.default_sections = ['GMS Options', 'GRIDDATA']
        self.blocks_requiring_spacer.extend(['GMS Options'])

    def setup_section(self, section_name):
        """Sets up a section of widgets.

        Args:
            section_name (str): name of the section
        """
        if section_name == 'GMS Options':
            self._setup_gms_options_section()
        else:
            super().setup_section(section_name)

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def _setup_gms_options_section(self):
        """Sets up the GMS Options section."""
        section = 'GMS Options'
        self.add_group_box_to_scroll_area(section)

        # Button
        hlay_widget = options_util.create_hlayout_widget('setup_package', self.uix[section])
        btn_text = 'Set up DRN, GHB, RCH, and TVA6'
        w = self.uix[section]['btn_set_up'] = QPushButton(btn_text)
        hlay_widget.layout().addWidget(w)
        hlay_widget.layout().addStretch()
        w.clicked.connect(self._on_btn_set_up)
        self.uix[section]['layout'].addWidget(hlay_widget)

        self.enable_layout_items(self.uix[section]['layout'], not self.dlg_input.locked)

    def _on_btn_set_up(self) -> None:
        """Called when the button is clicked."""
        # Abort if DISU
        if self.dlg_input.data.grid_info().dis_enum == DisEnum.DISU:
            msg = 'DISU package in use. This is only supported for models using DIS or DISV.'
            message_box.message_with_ok(parent=self, message=msg)
            return

        inputs: SwiSetUpInputs = swi_set_up_dialog.run_dialog(self.dlg_input.data.filename, self._inputs, self)
        if inputs is None:
            return

        # Create the GHB and DRN data
        self._outputs = self._set_up_swi(inputs)

    def _set_up_swi(self, inputs: SwiSetUpInputs) -> SwiSetUpOutputs:
        """Returns new/fixed related packages with the BCs in the correct stress periods / cells.

        Args:
            inputs: Data needed to create/fix the related packages.

        Returns:
            See description.
        """
        # Run the SWI setup tool
        data = self.dlg_input.data
        outputs = swi_tools.set_up_swi(data.mfsim, data, inputs)
        if outputs.errors:
            details = '\n'.join(outputs.errors)
            message_box.message_with_ok(parent=self, message=f'Error: {details}', details=details)
            return outputs

        # Turn on TVA6 option if needed
        if outputs.tva_filepath is not None:
            self.dlg_input.data.options_block.set(key='TVA6 FILEIN', on=True, value=[str(outputs.tva_filepath)])
            self.options_gui.uix['chk_tva6_filein'].setChecked(True)

        msg = self._get_result_message(inputs, outputs)
        message_box.message_with_ok(parent=self, message=msg)
        return outputs

    def _get_result_message(self, inputs: SwiSetUpInputs, outputs: SwiSetUpOutputs) -> str:
        """Return a message that tells the user what happened.

        Args:
            inputs: The inputs.
            outputs: The outputs.

        Returns:
            See description.
        """
        msg = ''
        if outputs.ghb:
            msg += 'GHB package created.\n\n'
        if outputs.drn:
            msg += 'DRN package created.\n\n'
        if inputs.fix_rch:
            if outputs.rch_data:
                if len(outputs.rch_data) == 1:
                    msg += 'RCH package copied and modified.\n\n'
                else:
                    msg += 'RCH packages copied and modified.\n\n'
            else:
                msg += 'No RCH packages needed to be fixed.\n\n'
        if outputs.tva_filepath is not None:
            msg += 'SALTWATER_HEAD TVA6 file created.\n\n'
        if outputs.ghb or outputs.drn or outputs.rch_data:
            msg += 'New package(s) will be added when exiting the dialog.'
        elif outputs.tva_filepath is None:
            msg += 'No changes made.'
        return msg

    def _init_set_up_inputs(self) -> None:
        """Initializes set up inputs so we remember them."""
        inputs = SwiSetUpInputs()
        gms_options = self.dlg_input.data.gms_options
        inputs.sea_level_csv = gms_options.get(swi_data.SEA_LEVEL_CSV, '')
        inputs.create_ghb = gms_options.get(swi_data.CREATE_GHB, True)
        inputs.ghb_cond = gms_options.get(swi_data.GHB_CONDUCTANCE, swi_data.DEFAULT_GHB_CONDUCTANCE)
        inputs.create_drn = gms_options.get(swi_data.CREATE_DRN, True)
        inputs.drn_cond = gms_options.get(swi_data.DRN_CONDUCTANCE, swi_data.DEFAULT_DRN_CONDUCTANCE)
        inputs.fix_rch = gms_options.get(swi_data.FIX_RCH, True)
        inputs.create_tva_file = gms_options.get(swi_data.CREATE_SALTWATER_HEAD_TVA, True)
        self._inputs = inputs

    def _store_set_up_inputs(self) -> None:
        """Stores set up inputs in the gms_options."""
        gms_options = self.dlg_input.data.gms_options
        gms_options[swi_data.SEA_LEVEL_CSV] = self._inputs.sea_level_csv
        gms_options[swi_data.CREATE_GHB] = self._inputs.create_ghb
        gms_options[swi_data.GHB_CONDUCTANCE] = self._inputs.ghb_cond
        gms_options[swi_data.CREATE_DRN] = self._inputs.create_drn
        gms_options[swi_data.DRN_CONDUCTANCE] = self._inputs.drn_cond
        gms_options[swi_data.FIX_RCH] = self._inputs.fix_rch
        gms_options[swi_data.CREATE_SALTWATER_HEAD_TVA] = self._inputs.create_tva_file

    def accept(self):
        """OK button clicked."""
        super().accept()
        if not self.dlg_input.locked:
            self._store_set_up_inputs()
            if self._outputs:
                swi_tools.add_packages(self.dlg_input, self._outputs)
