"""MapFromShapefileDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QDialog, QWidget

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.mf6.gui import gui_util
from xms.mf6.gui.map_from_coverage_dialog import MapOpt
from xms.mf6.gui.map_from_shapefile_dialog_ui import Ui_MapFromShapefileDialog


def run_dialog(shapefilepath: Path | str, append_or_replace: MapOpt, parent: QWidget) -> tuple[str, bool]:
    """Runs the Map from Shapefile dialog.

    Args:
        shapefilepath: The last shapefile the user picked this session.
        append_or_replace: 0 to make Append radio button on, 1 to make Replace radio button on.
        parent: The parent window.

    Return:
        The dataframe, or None if there was a problem.
    """
    dialog = MapFromShapefileDialog(shapefilepath, append_or_replace, parent)
    if dialog.exec() == QDialog.Accepted:
        return dialog.shapefilepath, dialog.replace
    else:
        return '', True


class MapFromShapefileDialog(XmsDlg):
    """Dialog used to get the shapefile and append/replace option from the user."""
    def __init__(self, shapefilepath: Path | str, append_or_replace: MapOpt, parent: QWidget | None = None):
        """Initializes the class, sets up the ui, and loads the simulation.

        Args:
            shapefilepath: The last shapefile the user picked this session.
            append_or_replace: 0 to make Append radio button on, 1 to make Replace radio button on.
            parent: The parent window.
        """
        super().__init__(parent=parent, dlg_name='xms.mf6.gui.map_from_shapefile_dialog')
        self.ui = Ui_MapFromShapefileDialog()
        self.ui.setupUi(self)

        # Init private variables
        self.help_getter = gui_util.help_getter(gui_util.help_id_from_key('MapFromShapefileDialog'))
        self._shapefilepath = str(shapefilepath)
        self._append_or_replace: MapOpt = append_or_replace

        # Set widgets
        gui_util.set_read_only_and_grey(self.ui.edt_shapefile, True)
        self.ui.edt_shapefile.setText(self._shapefilepath)
        self.ui.rbt_replace.setChecked(self._append_or_replace)

        # Signals
        self.ui.btn_browse.clicked.connect(self._on_btn_browse)
        self.ui.buttonBox.helpRequested.connect(self.help_requested)

    @property
    def shapefilepath(self) -> str:
        """Return the shapefile path."""
        return self._shapefilepath

    @property
    def append_or_replace(self) -> MapOpt:
        """Return True if the Replace option is selected."""
        return self._append_or_replace

    def _on_btn_browse(self) -> None:
        """Gets the sea level time series .csv file path from the user and reads it.

        Return:
            The dataframe, or None if there was a problem.
        """
        filter_str = 'Shapefiles (*.shp)'
        shapefilepath = gui_util.run_open_file_dialog(self, 'Select Shapefile', self._shapefilepath, filter_str)
        if shapefilepath:
            self.ui.edt_shapefile.setText(shapefilepath)

    def accept(self):
        """OK button clicked."""
        self._shapefilepath = self.ui.edt_shapefile.text()
        self._append_or_replace = self.ui.rbt_replace.isChecked()
        super().accept()
