"""Classes used to store option definitions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class OptionDefinitionBase:
    """Class to store what's needed to create a check box."""
    def __init__(self, option, brief='', check_box_method=''):
        """Initializes the class.

        Args:
            option (str): The option string that the checkbox will display.
            brief (str): Brief explanation of the option that follows the checkbox.
            check_box_method (str): Name of method to call when checkbox state changes (stateChanged signal).
        """
        self.option = option
        self.brief = brief
        self.check_box_method = check_box_method


class Checkbox(OptionDefinitionBase):
    """Class to store what's needed to create a check box."""
    def __init__(self, option, brief='', check_box_method=''):
        """Initializes the class.

        Args:
            option (str): The option string that the checkbox will display.
            brief (str): Brief explanation of the option that follows the checkbox.
            check_box_method (str): Name of method to call when checkbox state changes (stateChanged signal).
        """
        super().__init__(option, brief, check_box_method)


class CheckboxButton(OptionDefinitionBase):
    """Class to store what's needed to create a check box and push button."""
    def __init__(self, option, brief='', button_text='', check_box_method='', button_method=None):
        """Initializes the class.

        Args:
            option (str): The option string that the checkbox will display.
            brief (str): Brief explanation of the option that follows the checkbox.
            button_text (str): What is displayed on the push button.
            check_box_method (str): Name of method to call when checkbox state changes (stateChanged signal).
            button_method: Method to call when the check box is clicked.
        """
        super().__init__(option, brief)
        self.button_text = button_text
        self.check_box_method = check_box_method
        self.button_method = button_method


class CheckboxComboBox(OptionDefinitionBase):
    """Class to store what's needed to create a check box and combo box."""
    def __init__(self, option, brief='', items=None, value='', check_box_method='', combo_box_method=''):
        """Initializes the class.

        Args:
            option (str): The option string that the checkbox will display.
            brief (str): Brief explanation of the option that follows the checkbox.
            items (list(str)): The strings for the combo box.
            value: The initial value of the combo box.
            check_box_method (str): Name of method to call when checkbox state changes (stateChanged signal).
            combo_box_method (str): Name of method to call when combo box text changes(currentTextChanged signal).
        """
        super().__init__(option, brief)
        self.items = items
        self.value = value
        self.check_box_method = check_box_method
        self.combo_box_method = combo_box_method


class CheckboxCheckbox(OptionDefinitionBase):
    """Class to store what's needed to create a check box and another check box."""
    def __init__(self, option, brief='', option2='', brief2=''):
        """Initializes the class.

        Args:
            option (str): The option string that the checkbox will display.
            brief (str): Brief explanation of the option that follows the checkbox.
            option2 (str): The option string that the second checkbox will display.
            brief2 (str): Brief explanation of the second option that follows the second checkbox.
        """
        super().__init__(option, brief)
        self.option2 = option2
        self.brief2 = brief2


class CheckboxField(OptionDefinitionBase):
    """Class to store what's needed to create a check box and field (line edit or spin box)."""
    def __init__(self, option, brief='', type_='', value=None, read_only=False, minimum=None, maximum=None):
        """Initializes the class.

        Args:
            option (str): The option string that the checkbox will display.
            brief (str): Brief explanation of the option that follows the checkbox.
            type_ (str): 'str' for string, 'int' for int, 'float' for float
            value: Value to put in the field.
            read_only (bool): If True, checkbox and field are made read-only.
            minimum: The minimum value.
            maximum: The maximum value.
        """
        super().__init__(option, brief)
        self.type_ = type_
        self.value = value
        self.read_only = read_only
        self.minimum = minimum
        self.maximum = maximum


class LineEdit(OptionDefinitionBase):
    """Class to store what's needed to create a line edit."""
    def __init__(self, option, brief='', type_='', value=None, read_only=False):
        """Initializes the class.

        Args:
            option (str): The option string that the checkbox will display.
            brief (str): Brief explanation of the option that follows the checkbox.
            type_ (str): 'str' for string, 'int' for int, 'float' for float
            value: Value to put in the field.
            read_only (bool): If True, checkbox and field are made read-only.
        """
        super().__init__(option, brief)
        self.type_ = type_
        self.value = value
        self.read_only = read_only


class Checkbox3Fields(OptionDefinitionBase):
    """Class to store what's needed to create a check box and 3 fields."""
    def __init__(
        self,
        option,
        brief='',
        option1='',
        type1='',
        value1=None,
        option2='',
        type2='',
        value2=None,
        option3='',
        type3='',
        value3=None
    ):
        """Initializes the class.

        Args:
            option (str): The option string that the checkbox will display.
            brief (str): Brief explanation of the option that follows the checkbox.
            option1 (str): The option string displayed to the left of the first field.
            type1 (str): 'str' for string, 'int' for int, 'float' for float
            value1: Value to put in the field.
            option2 (str): The option string displayed to the left of the first field.
            type2 (str): 'str' for string, 'int' for int, 'float' for float
            value2: Value to put in the field.
            option3 (str): The option string displayed to the left of the first field.
            type3 (str): 'str' for string, 'int' for int, 'float' for float
            value3: Value to put in the field.
        """
        super().__init__(option, brief)
        self.option1 = option1
        self.type1 = type1
        self.value1 = value1

        self.option2 = option2
        self.type2 = type2
        self.value2 = value2

        self.option3 = option3
        self.type3 = type3
        self.value3 = value3


class CheckboxFieldButton(OptionDefinitionBase):
    """Class to store what's needed to create a check box and field (line edit or spin box) and button."""
    def __init__(
        self,
        option,
        brief='',
        type_='',
        value=None,
        button_text='',
        check_box_method='',
        field_method='',
        button_method=''
    ):
        """Initializes the class.

        Args:
            option (str): The option string that the checkbox will display.
            brief (str): Brief explanation of the option that follows the checkbox.
            type_ (str): 'str' for string, 'int' for int, 'float' for float
            value: Value to put in the field.
            button_text (str): Text to put on the push button.
            check_box_method (str): Name of method to call when checkbox state changes (stateChanged signal).
            field_method (str): Name of method to call when edit field is done being edited (editingFinished signal).
            button_method (str): Name of method to call when button is pushed (clicked signal).
        """
        super().__init__(option, brief)
        self.type_ = type_
        self.value = value
        self.button_text = button_text
        self.check_box_method = check_box_method
        self.field_method = field_method
        self.button_method = button_method


class CheckboxPrintFormat(OptionDefinitionBase):
    """Class to store what's needed to create a check box and field (line edit or spin box)."""
    def __init__(self, option, brief=''):
        """Initializes the class.

        Args:
            option (str): The option string that the checkbox will display.
            brief (str): Brief explanation of the option that follows the checkbox.
        """
        super().__init__(option, brief)
