"""Module for the PtmRunManager class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['PtmModelRunManager']

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.components.dmi.model_run_manager import ModelRunManager
from xms.components.dmi.xms_data import XmsData
from xms.guipy.dialogs.process_feedback_dlg import run_feedback_dialog

# 4. Local modules
from xms.ptm.entry_points import model_run, model_run_tracker
from xms.ptm.feedback.load_solution_thread import PtmLoadSolutionThread


class PtmModelRunManager(ModelRunManager):
    """Class that finds executables for running PTM."""
    def find_model_run_script(self, file_location: str):
        """
        Find the model run and progress scripts.

        Args:
            file_location: Folder where XMS thinks the model native files should be.
        """
        control_file = f'{self.data.simulation_name}.pcf'
        control_path = Path(file_location) / control_file
        if not control_path.exists():
            self.report_error(
                f'Unable to run PTM because "{control_path}" was not found. Ensure the simulation has been exported.'
            )
            return

        model_path = self.find_model_executable('PTM Executable')
        if not model_path:
            self.report_error(
                'Unable to run PTM because the model executable was not set. '
                'Check the "PTM Executable" file location in the SMS Preferences.'
            )
            return
        elif not Path(model_path).exists():
            self.report_error(
                'Unable to run PTM because the model executable does not exist. '
                'Check the "PTM Executable" file location in the SMS Preferences.'
            )
            return
        elif not Path(model_path).is_file():
            self.report_error(
                'Unable to run PTM because the model executable is not a file. '
                'Check the "PTM Executable" file location in the SMS Preferences.'
            )
            return

        self.runner = model_run.__file__
        self.runner_args.append(str(model_path))
        self.runner_args.append(str(control_path))

        self.tracker = model_run_tracker.__file__

    def load_solution(self, data: XmsData, parent: QWidget):
        """Called by XMS when it wants to load the solution."""
        thread = PtmLoadSolutionThread(data)
        run_feedback_dialog(thread, parent=parent)
